# Ejemplos de Código - Woox Normalizer

**Fecha:** 2026-01-14
**Broker:** Woox
**Status:** ✅ 100% Hash Compatible - NO Critical Changes Required

---

## Tabla de Contenidos

1. [Hash Computation - ✅ 100% Compatible](#hash-computation)
2. [Symbol Transformation - ✅ Already Correct](#symbol-transformation)
3. [Type Precision - ✅ Critical for Hash](#type-precision)
4. [Timestamp Format - ✅ STRING Requirement](#timestamp-format)
5. [Side Mapping - ✅ Already Correct](#side-mapping)
6. [Fee Calculation - Simplification](#fee-calculation)
7. [Pip Value - Simplification](#pip-value)
8. [Skip Conditions - Architecture Change](#skip-conditions)
9. [Tests Examples](#tests-examples)

---

## Hash Computation

### ✅ STATUS: 100% COMPATIBLE (NO CHANGES NEEDED)

**Hallazgo Clave:** A diferencia de Oanda/Propreports (0% hash match), Woox YA tiene 100% hash compatibility.

#### Legacy Implementation (woox_export.py:112-113, 425-426)

**Step 1: Add created_at fields (Lines 112-113):**
```python
order['created_at'] = int(order['executed_timestamp'])/1000
order['created_at_formated'] = datetime.utcfromtimestamp(
    int(order['executed_timestamp'])/1000
).strftime('%Y-%m-%d %H:%M:%S')
```

**Step 2: Compute hash (Lines 425-426):**
```python
check_json = json.dumps(order)
check_json = hashlib.md5(check_json.encode('utf-8')).hexdigest()
```

**Formula:** `MD5(json.dumps(order))` where order includes created_at fields

#### New Implementation (woox.py:128-173)

**Complete Hash Computation:**
```python
# Lines 134-139: Timestamp conversion
exec_time_ms = int(exec_timestamp)
created_at = exec_time_ms / 1000  # Convert ms to seconds as FLOAT
created_at_formated = datetime.utcfromtimestamp(created_at).strftime('%Y-%m-%d %H:%M:%S')

# Lines 145-148: Symbol transformation
legacy_symbol = str(symbol)
if legacy_symbol.startswith("PERP_") and legacy_symbol.endswith("_USDT"):
    legacy_symbol = legacy_symbol[5:-5] + "USDT"  # PERP_BTC_USDT -> BTCUSDT

# Lines 151-153: Number type handling (CRITICAL)
def to_number(val):
    f = float(val or 0)
    return int(f) if f == int(f) else f  # Returns int for whole numbers

# Lines 156-170: Hash order (EXACT field order - no sort_keys)
order_for_hash = {
    "id": int(exec_id),
    "symbol": legacy_symbol,
    "fee": float(fee or 0),
    "side": str(side),
    "executed_timestamp": str(exec_timestamp),  # STRING not int
    "order_id": int(order_id),
    "executed_price": to_number(exec_price),  # Int if whole
    "executed_quantity": float(exec_qty or 0),
    "fee_asset": str(fee_asset),
    "is_maker": int(is_maker),
    "order_tag": str(order_tag),
    "created_at": created_at,  # Float
    "created_at_formated": created_at_formated,
}

# Line 173: Final hash (NO sort_keys to preserve order)
file_row_hash = hashlib.md5(json.dumps(order_for_hash).encode('utf-8')).hexdigest()
```

**Formula:** `MD5(json.dumps(order_for_hash))` with precise field order and type precision

**CRITICAL NOTES:**
- WooX does NOT have a category field (unlike Bybit/Kucoin/Binance)
- `executed_price` must be **int for whole numbers** (100112 not 100112.0)
- `executed_timestamp` must be **STRING not int**
- Symbol transformation is PERP_*_USDT format only
- No sort_keys in json.dumps() - order matters!
- Field order is CRITICAL - must match legacy exactly

#### Verification

**314 records:** 100% hash match ✅

**Key Difference from Oanda/Propreports:**
- **Oanda:** Legacy incluía "buy/sell", nueva no → 0% match ⚠️⚠️⚠️
- **Propreports:** Legacy NO incluía portfolio, nueva SÍ → 0% match ⚠️⚠️⚠️
- **Woox:** Ambas usan exact field order + type precision → 100% match ✅✅✅

#### Documentation in Code

```python
# woox.py:9-45 - Clear documentation
"""
file_row Hash Formula (Legacy Compatibility):
---------------------------------------------
The file_row field is computed as an MD5 hash for deduplication against
the legacy TraderSync system. The formula transforms API fields to snake_case:

    file_row = MD5(json.dumps(transformed_order))

Steps:
    1. Transform API keys from camelCase to snake_case
    2. Transform symbol: PERP_BTC_USDT -> BTCUSDT
    3. Build order with legacy key order and types:
       {
         "id": int,
         "symbol": str (simplified),
         "fee": float,
         "side": str,
         "executed_timestamp": str (not int!),
         "order_id": int,
         "executed_price": int if whole number else float,
         "executed_quantity": float,
         "fee_asset": str,
         "is_maker": int,
         "order_tag": str,
         "created_at": float (timestamp / 1000),
         "created_at_formated": str ('%Y-%m-%d %H:%M:%S')
       }
    4. Hash: hashlib.md5(json.dumps(order).encode('utf-8')).hexdigest()

Note: WooX does NOT have a category field like Bybit/Kucoin/Binance.
Note: executed_price must be int for whole numbers (100112 not 100112.0)

Match Rate: 100% against legacy data (verified with 314 records).
"""
```

---

## Symbol Transformation

### ✅ STATUS: ALREADY CORRECT

Both legacy and new implementations transform PERP_*_USDT symbols correctly.

#### Legacy Implementation (woox_export.py:318-319)

```python
# Lines 318-319
SymbolList = row_order['symbol'].split("_")
row_order['symbol'] = SymbolList[-2] + SymbolList[-1]
```

**Example:**
- Input: `"PERP_BTC_USDT"` → Split: `["PERP", "BTC", "USDT"]`
- Take last 2: `SymbolList[-2]` = "BTC", `SymbolList[-1]` = "USDT"
- Result: `"BTCUSDT"` ✅

#### New Implementation (woox.py:145-148)

```python
# Lines 145-148
legacy_symbol = str(symbol)
if legacy_symbol.startswith("PERP_") and legacy_symbol.endswith("_USDT"):
    # Extract the middle part: PERP_BTC_USDT -> BTC, then add USDT
    legacy_symbol = legacy_symbol[5:-5] + "USDT"
```

**Example:**
- Input: `"PERP_BTC_USDT"`
- Check: `startswith("PERP_")` ✅ and `endswith("_USDT")` ✅
- Transform: `[5:-5]` = "BTC" (skip first 5 chars "PERP_", skip last 5 chars "_USDT")
- Add "USDT": `"BTC" + "USDT"` = `"BTCUSDT"` ✅

**Result:** Both implementations produce identical symbol transformation ✅

**More Examples:**
```python
"PERP_BTC_USDT" → "BTCUSDT"   ✅
"PERP_ETH_USDT" → "ETHUSDT"   ✅
"PERP_SOL_USDT" → "SOLUSDT"   ✅
"PERP_MATIC_USDT" → "MATICUSDT" ✅
```

---

## Type Precision

### ✅ STATUS: CRITICAL FOR HASH (ALREADY CORRECT)

This is one of the MOST CRITICAL aspects for hash compatibility. The executed_price field must be INT for whole numbers, FLOAT otherwise.

#### Why Type Precision Matters

**Problem:** Python's `json.dumps()` represents numbers differently:
```python
json.dumps({"price": 100112})    # → '{"price": 100112}'
json.dumps({"price": 100112.0})  # → '{"price": 100112.0}'
```

**Impact:** Different JSON representations → Different MD5 hashes!

#### Legacy Behavior

Legacy system stored executed_price as received from API:
- If API returned `100112` (int) → stored as int → hash includes `"executed_price": 100112`
- If API returned `153.9` (float) → stored as float → hash includes `"executed_price": 153.9`

#### New Implementation (woox.py:151-153, 163)

```python
# Lines 151-153: Helper function
def to_number(val):
    f = float(val or 0)
    return int(f) if f == int(f) else f  # Returns int for whole numbers

# Line 163: Usage in hash order
"executed_price": to_number(exec_price),  # Int if whole, else float
```

**Examples:**
```python
to_number(100112.0)  # → int(100112)     ✅
to_number(100112)    # → int(100112)     ✅
to_number(153.9)     # → float(153.9)    ✅
to_number(1.0)       # → int(1)          ✅
to_number(1.5)       # → float(1.5)      ✅
to_number(0.001)     # → float(0.001)    ✅
```

#### Hash Impact Example

**Scenario 1: Whole Number (100112)**

**Correct (INT):**
```python
order = {"executed_price": 100112}  # int
json.dumps(order)  # → '{"executed_price": 100112}'
md5_hash = "abc123..."  # Correct hash
```

**Incorrect (FLOAT):**
```python
order = {"executed_price": 100112.0}  # float
json.dumps(order)  # → '{"executed_price": 100112.0}'
md5_hash = "def456..."  # Different hash! ⚠️
```

**Result:** Hash mismatch due to `.0` suffix!

**Scenario 2: Decimal Number (153.9)**

**Correct (FLOAT):**
```python
order = {"executed_price": 153.9}  # float
json.dumps(order)  # → '{"executed_price": 153.9}'
md5_hash = "ghi789..."  # Correct hash
```

**Incorrect (INT would lose precision):**
```python
order = {"executed_price": 153}  # int
json.dumps(order)  # → '{"executed_price": 153}'
md5_hash = "jkl012..."  # Different hash + data loss! ⚠️⚠️
```

#### Test Verification

**Test:** `tests/brokers/test_woox.py::TestFileRowHash::test_hash_legacy_formula`

```python
def test_hash_legacy_formula():
    """Test that hash matches exact legacy formula with type precision."""
    json_content = json.dumps([{
        "id": "999888777",
        "symbol": "PERP_BTC_USDT",
        "orderId": "123456",
        "executedPrice": 100112,  # Whole number → should be int in hash
        "executedQuantity": 0.819,
        "fee": 0.01,
        "feeAsset": "USDT",
        "orderTag": "default",
        "side": "SELL",
        "executedTimestamp": 1705392000000,
        "isMaker": 0
    }])

    df = WooxInterpreter.parse_json_content(json_content)

    # Expected hash computed with legacy formula
    expected_order = {
        "id": 999888777,
        "symbol": "BTCUSDT",  # Transformed
        "fee": 0.01,
        "side": "SELL",
        "executed_timestamp": "1705392000000",  # STRING
        "order_id": 123456,
        "executed_price": 100112,  # INT (whole number)
        "executed_quantity": 0.819,
        "fee_asset": "USDT",
        "is_maker": 0,
        "order_tag": "default",
        "created_at": 1705392000.0,
        "created_at_formated": "2024-01-16 00:00:00"
    }

    expected_hash = hashlib.md5(
        json.dumps(expected_order).encode('utf-8')
    ).hexdigest()

    assert df["_file_row_hash"][0] == expected_hash  # ✅ PASS
```

---

## Timestamp Format

### ✅ STATUS: STRING REQUIREMENT (ALREADY CORRECT)

The executed_timestamp field MUST be stored as STRING in the hash, not int.

#### Why STRING is Required

**Legacy Behavior:** The executed_timestamp was stored as string in the order object before hashing.

#### Legacy Implementation (woox_export.py:320)

```python
# Line 320
row_order['executed_timestamp'] = str(row_order['executed_timestamp']).replace('.', '')
```

**Example:**
- Input: `1705392000000` (int)
- Transform: `str(1705392000000)` → `"1705392000000"` (string)
- Remove dots: `.replace('.', '')` → `"1705392000000"` (no change if no dots)

#### New Implementation (woox.py:161)

```python
# Line 161
"executed_timestamp": str(exec_timestamp),  # String in legacy
```

**Example:**
```python
exec_timestamp = 1705392000000  # int from API
str(exec_timestamp)  # → "1705392000000" (string)
```

#### Hash Impact Example

**Correct (STRING):**
```python
order = {"executed_timestamp": "1705392000000"}  # string
json.dumps(order)  # → '{"executed_timestamp": "1705392000000"}'
md5_hash = "abc123..."  # Correct hash
```

**Incorrect (INT):**
```python
order = {"executed_timestamp": 1705392000000}  # int
json.dumps(order)  # → '{"executed_timestamp": 1705392000000}'  # No quotes!
md5_hash = "def456..."  # Different hash! ⚠️
```

**Result:** Hash mismatch due to missing quotes!

#### Additional Fields

**created_at (FLOAT):**
```python
# Lines 135-136
exec_time_ms = int(exec_timestamp)
created_at = exec_time_ms / 1000  # Convert ms to seconds as FLOAT
```

**Example:**
```python
1705392000000 / 1000  # → 1705392000.0 (float)
```

**created_at_formated (STRING):**
```python
# Line 136
created_at_formated = datetime.utcfromtimestamp(created_at).strftime('%Y-%m-%d %H:%M:%S')
```

**Example:**
```python
datetime.utcfromtimestamp(1705392000.0).strftime('%Y-%m-%d %H:%M:%S')
# → "2024-01-16 00:00:00" (string)
```

---

## Side Mapping

### ✅ STATUS: ALREADY CORRECT

Both legacy and new implementations map to "BUY" / "SELL" correctly.

#### Legacy Implementation (woox_export.py:435-439)

```python
# Lines 435-439
action = order['buy/sell'].upper() if 'buy/sell' in order \
    else order['side'].upper() if 'side' in order \
    else order['transaction category'].upper() if 'transaction category' in order \
    else ''
action = 'BUY' if action == 'BUY' or action=='COVER' \
    else 'SELL' if action == 'SELL' or action == 'SHORT' \
    else ''
```

**For Woox:** Uses `order['side']` field → `"BUY"` or `"SELL"`

#### New Implementation (woox.py:245-248 in normalize method)

```python
# Lines 245-248
# side - map to BUY/SELL
pl.when(pl.col("side") == "BUY")
.then(pl.lit("BUY"))
.otherwise(pl.lit("SELL"))
.alias("side"),
```

**Result:** Both map to "BUY" / "SELL" correctly ✅

---

## Fee Calculation

### ⭐ STATUS: SIMPLIFIED (ACCEPTABLE)

Legacy had complex fee priority logic, new implementation simplifies to 0.0 (fees handled in separate stage).

#### Legacy Implementation (woox_export.py:552-561)

```python
# Lines 552-561: Priority-based fee selection
if 'execFee' in order:
    order['fees'] = (float(order['execFee']) * float(pip_value_btc)) \
        if 'execFee' in order else '0.00'
elif 'cumExecFee' in order:
    order['fees'] = order['cumExecFee'] \
        if 'cumExecFee' in order else '0.00'
else:
    order['fees'] = (float(order['feeAmount']) * float(pip_value_btc)) \
        if 'feeAmount' in order else '0.00'
```

**Fee Priority:**
1. `execFee` * `pip_value_btc` (primary)
2. `cumExecFee` (fallback)
3. `feeAmount` * `pip_value_btc` (final fallback)

**USD Adjustment (Lines 559-561):**
```python
if ImportParams.isfloat(order['fees']) \
      and 'USD' in order['symbol'] \
      and not 'USDT' in order['symbol'] \
      and not 'BTCUSD' in order['symbol']:
    order['fees'] = float(order['fees']) * float(price)
```

#### New Implementation (woox.py:263)

```python
# Line 263
pl.lit(0.0).alias("fees"),
```

**New Logic:** Fixed 0.0 for all executions

**Impact:** ⭐ BAJA - Fees are now calculated in p04_calculate stage (separation of concerns)
**Action:** NINGUNA - Architectural improvement

---

## Pip Value

### ⭐ STATUS: SIMPLIFIED (ACCEPTABLE)

Legacy had complex pip_value calculation based on execValue and symbol type, new implementation simplifies to 1.0.

#### Legacy Implementation (woox_export.py:503-533)

**Complex pip_value Logic:**

```python
# Lines 503-533
pip_value = 1

# Case 1: No execValue field
if not 'execValue' in order:
    pip_value = self.convert_usdt(order, pip_value_order, date)

# Case 2: USD symbol
elif sym in ['USD']:
    value_order = float(order['executed_quantity']) * float(order['executed_price'])
    exec_value = float(order['execValue']) if ImportParams.isfloat(order['execValue']) else ''
    pip_value_exec = (exec_value / value_order)
    pip_value = pip_value_exec * price

# Case 3: USDT symbol
elif 'USDT' in order['symbol']:
    pip_value = 1

# Case 4: Other symbols with execValue
else:
    value_order = float(order['executed_quantity']) * float(order['executed_price'])
    exec_value = float(order['execValue']) if ImportParams.isfloat(order['execValue']) else ''
    if exec_value and order['symbol']:
        if value_order != 0:
            pip_value = (exec_value / value_order)
        else:
            pip_value = exec_value
        pip_value = pip_value * float(order['price'])

# BTC Special Handling (Lines 527-533)
if order['symbol'] == 'BTCUSD':
    row_btc = '{}{}'.format(date, sym)
    if not row_btc in pip_value_order_btc:
        pip_value_btc = ImportParams.pip_value_crypto_(order['date'], base='usdt', currency='btc')
        pip_value_order_btc[row_btc] = pip_value_btc
    else:
        pip_value_btc = pip_value_order_btc[row_btc]
```

#### New Implementation (woox.py:284)

```python
# Line 284
pl.lit(1.0).alias("pip_value"),
```

**New Logic:** Fixed 1.0 for all executions

**Impact:** ⭐ BAJA - Pip value is now calculated in p04_calculate stage (separation of concerns)
**Action:** NINGUNA - Architectural improvement

---

## Skip Conditions

### ✅ STATUS: ARCHITECTURE CHANGE (CORRECT)

Legacy had 4-tier deduplication cascade, new has 1 validation point. This is NOT a bug - it's proper separation of concerns.

#### Legacy Implementation (woox_export.py:427-499)

**4-Tier Deduplication Cascade:**

**Tier 1: Duplicate in Current Batch (Lines 427-428)**
```python
check_json = hashlib.md5(check_json.encode('utf-8')).hexdigest()
if check_json in new_list:
    continue
```

**Tier 2: Database Hash Check (Lines 477-479)**
```python
if njson in orders_filerow:
    verify_njson_len = verify_njson_len + 1
    continue
```

**Tier 3: Order ID Check (Lines 480-484)**
```python
if 'id' in original_file_row and original_file_row['id']:
    order_id = '{}'.format(original_file_row['id'])
    if order_id in orders_filerow:
        verify_njson_len = verify_njson_len + 1
        continue
```

**Tier 4: Composite Hash Check (Lines 488-499)**
```python
njson3 = '{}{}{}{}{}{}{}'.format(
    float(fp) if fp else 0.00,
    var_date,
    option,
    1 if action == 'BUY' else 2,
    float(str(order['executed_quantity']).replace(',', '')) if 'executed_quantity' in order else '',
    float(strike) if ImportParams.isfloat(strike) else 0.0,
    expire
)
if njson3 in orders_filerow:
    verify_njson_len = verify_njson_len + 1
    continue
```

**Additional Skip (Lines 450-451):**
```python
if not order['symbol'] or not order['date'] or not order['price'] \
      or not action or order['type'] != "TRADE":
    continue
```

**Legacy:** 4-tier deduplication + validation (all in parser)

#### New Implementation (woox.py:73-80)

```python
# Lines 73-80
@classmethod
def can_handle(cls, df: pl.DataFrame, metadata: dict) -> bool:
    """Check if this interpreter can handle the data."""
    required = {"id", "symbol", "side", "executedQuantity", "executedPrice", "executedTimestamp"}
    return required.issubset(set(df.columns))
```

**New:** 1 validation point (required fields check only)

**Where did deduplication go?** → **p02_deduplicate stage** ✅

#### Architecture Improvement

**Old Architecture (Mixed Concerns):**
```
woox_export.py (636 lines):
  - API calls
  - Data parsing
  - Validation
  - Deduplication ← MIXED
  - Hash computation
  - Fee calculation
  - Pip value calculation
  - Database writes
```

**New Architecture (Separation of Concerns):**
```
Pipeline Stages:
  p01_normalize/
    - woox.py (316 lines): Parse + Transform ONLY

  p02_deduplicate/
    - deduplicate.py: Deduplication logic ONLY

  p03_group/
    - grouping.py: Group executions → trades

  p04_calculate/
    - calculations.py: P&L, fees, pip value calculations

  p05_write/
    - writer.py: Database writes
```

**Result:** ✅ Architecture change is CORRECT - separation of concerns is best practice.

---

## Tests Examples

### Current Test Suite (test_woox.py - 587 lines, 32+ tests)

#### Test Class 1: TestWooxInterpreter (17 tests)

```python
class TestWooxInterpreter:
    """Test Woox interpreter parsing and normalization."""

    def test_parse_json_content(self):
        """Test parsing valid Woox JSON."""
        json_content = json.dumps([{
            "id": "999888777",
            "symbol": "PERP_BTC_USDT",
            "orderId": "123456",
            "executedPrice": 100112,
            "executedQuantity": 0.819,
            "fee": 0.01,
            "feeAsset": "USDT",
            "orderTag": "default",
            "side": "SELL",
            "executedTimestamp": 1705392000000,
            "isMaker": 0
        }])

        df = WooxInterpreter.parse_json_content(json_content)
        assert len(df) == 1
        assert df["symbol"][0] == "PERP_BTC_USDT"

    def test_normalize_basic(self):
        """Test basic normalization to 20-column schema."""

    def test_normalize_side_mapping(self):
        """Test side mapping (BUY/SELL)."""

    def test_symbol_uppercase(self):
        """Test symbol is uppercased."""

    def test_quantity_field(self):
        """Test quantity field (executedQuantity)."""

    def test_price_field(self):
        """Test price field (executedPrice)."""

    def test_commission_field(self):
        """Test commission field (always 0.0)."""

    def test_timestamp_field(self):
        """Test timestamp field (ms → datetime)."""

    def test_currency_field(self):
        """Test currency field (feeAsset)."""

    def test_asset_field(self):
        """Test asset field (always 'crypto')."""

    def test_execution_id_field(self):
        """Test execution_id field (id converted to string)."""

    def test_original_file_row_determinism(self):
        """Test original_file_row is deterministic."""

    def test_empty_array(self):
        """Test empty array returns empty DataFrame."""

    def test_parse_with_missing_fields(self):
        """Test handling executions with missing optional fields."""

    def test_normalize_empty_data(self):
        """Test normalization with empty DataFrame."""

    def test_normalize_with_nulls(self):
        """Test normalization handles null values correctly."""

    def test_multiple_executions(self):
        """Test processing multiple executions."""
```

#### Test Class 2: TestFileRowHash (5 tests - CRITICAL)

```python
class TestFileRowHash:
    """Test file_row hash computation for legacy compatibility."""

    def test_hash_md5_format(self):
        """Test hash is valid MD5 format (32 hex characters)."""
        json_content = json.dumps([{
            "id": "999888777",
            "symbol": "PERP_BTC_USDT",
            "orderId": "123456",
            "executedPrice": 100112,
            "executedQuantity": 0.819,
            "fee": 0.01,
            "feeAsset": "USDT",
            "orderTag": "default",
            "side": "SELL",
            "executedTimestamp": 1705392000000,
            "isMaker": 0
        }])

        df = WooxInterpreter.parse_json_content(json_content)
        hash_val = df["_file_row_hash"][0]

        # Check it's 32 hex characters
        assert len(hash_val) == 32
        assert all(c in '0123456789abcdef' for c in hash_val)

    def test_hash_deterministic(self):
        """Test hash is deterministic (same input → same hash)."""
        json_content = json.dumps([{
            "id": "999888777",
            "symbol": "PERP_BTC_USDT",
            "orderId": "123456",
            "executedPrice": 100112,
            "executedQuantity": 0.819,
            "fee": 0.01,
            "feeAsset": "USDT",
            "orderTag": "default",
            "side": "SELL",
            "executedTimestamp": 1705392000000,
            "isMaker": 0
        }])

        df1 = WooxInterpreter.parse_json_content(json_content)
        df2 = WooxInterpreter.parse_json_content(json_content)

        assert df1["_file_row_hash"][0] == df2["_file_row_hash"][0]  # ✅

    def test_hash_legacy_formula(self):
        """Test hash matches legacy formula with type precision."""
        # This is the MOST IMPORTANT test - verifies 100% compatibility
        json_content = json.dumps([{
            "id": "999888777",
            "symbol": "PERP_BTC_USDT",
            "orderId": "123456",
            "executedPrice": 100112,  # Whole number → int in hash
            "executedQuantity": 0.819,
            "fee": 0.01,
            "feeAsset": "USDT",
            "orderTag": "default",
            "side": "SELL",
            "executedTimestamp": 1705392000000,
            "isMaker": 0
        }])

        df = WooxInterpreter.parse_json_content(json_content)

        # Expected hash computed with legacy formula
        expected_order = {
            "id": 999888777,
            "symbol": "BTCUSDT",  # Transformed
            "fee": 0.01,
            "side": "SELL",
            "executed_timestamp": "1705392000000",  # STRING
            "order_id": 123456,
            "executed_price": 100112,  # INT (whole number)
            "executed_quantity": 0.819,
            "fee_asset": "USDT",
            "is_maker": 0,
            "order_tag": "default",
            "created_at": 1705392000.0,
            "created_at_formated": "2024-01-16 00:00:00"
        }

        expected_hash = hashlib.md5(
            json.dumps(expected_order).encode('utf-8')
        ).hexdigest()

        assert df["_file_row_hash"][0] == expected_hash  # ✅ PASS

    def test_hash_snake_case_keys(self):
        """Test hash handles snake_case keys (legacy format)."""

    def test_hash_different_executions(self):
        """Test different executions produce different hashes."""
```

#### Test Class 3: TestEdgeCases (6 tests)

```python
class TestEdgeCases:
    """Test edge cases and special scenarios."""

    def test_empty_array(self):
        """Test empty array returns empty DataFrame."""

    def test_zero_fee(self):
        """Test handling of zero fee."""

    def test_maker_execution(self):
        """Test maker execution (isMaker=1)."""

    def test_multiple_symbols(self):
        """Test multiple PERP symbols (BTC, ETH, SOL)."""
        json_content = json.dumps([
            {
                "id": "111",
                "symbol": "PERP_BTC_USDT",  # → BTCUSDT
                "executedPrice": 100112,
                # ...
            },
            {
                "id": "222",
                "symbol": "PERP_ETH_USDT",  # → ETHUSDT
                "executedPrice": 4567.89,
                # ...
            },
            {
                "id": "333",
                "symbol": "PERP_SOL_USDT",  # → SOLUSDT
                "executedPrice": 123.45,
                # ...
            }
        ])

        df = WooxInterpreter.parse_json_content(json_content)
        assert len(df) == 3

        # Verify symbol transformation
        # Note: original symbol is preserved in DataFrame,
        # transformation is only for hash computation

    def test_very_small_quantity(self):
        """Test very small quantity (0.00001)."""

    def test_timestamp_conversion(self):
        """Test milliseconds to datetime conversion."""
```

### Running Tests

```bash
# Run all tests
pytest tests/brokers/test_woox.py -v

# Run specific test class
pytest tests/brokers/test_woox.py::TestFileRowHash -v

# Run with coverage
pytest tests/brokers/test_woox.py --cov=brokers.woox --cov-report=html

# Run specific critical test
pytest tests/brokers/test_woox.py::TestFileRowHash::test_hash_legacy_formula -v
```

---

## Summary

### Code Changes Summary

| Item | Legacy | New | Status | Action |
|------|--------|-----|--------|--------|
| **Hash Computation** | MD5(order) | MD5(order_for_hash) | ✅ 100% MATCH | NINGUNA |
| **Symbol Transformation** | Split + concat | Slice + concat | ✅ MATCH | NINGUNA |
| **Type Precision** | Dynamic (INT/FLOAT) | to_number() helper | ✅ MATCH | NINGUNA |
| **Timestamp Format** | STRING | STRING | ✅ MATCH | NINGUNA |
| **Side Mapping** | BUY/SELL | BUY/SELL | ✅ MATCH | NINGUNA |
| **Fee Calculation** | Priority-based | Fixed 0.0 | ⚠️ SIMPLIFIED | ACCEPTABLE |
| **Pip Value** | Complex calculation | Fixed 1.0 | ⚠️ SIMPLIFIED | ACCEPTABLE |
| **Skip Conditions** | 4-tier dedup | Required fields only | ✅ ARCHITECTURE | NINGUNA |

### Lines of Code

| File | Legacy | New | Reduction |
|------|--------|-----|-----------|
| **Implementation** | 636 | 316 | 50% ✅ |
| **Tests** | Minimal | 587 (32+ tests) | N/A ✅ |
| **Total** | 636 | 903 | N/A |

**Key:** Effective implementation reduction is 50% (636 → 316), with comprehensive test coverage added.

### Critical Findings

1. ✅✅✅ **Hash 100% Compatible** - NO critical issues (unlike Oanda/Propreports)
2. ✅ **Type precision correct** - INT for whole, FLOAT for decimals
3. ✅ **Symbol transformation correct** - PERP_*_USDT → *USDT
4. ✅ **Timestamp format correct** - STRING not int
5. ✅ **Modern architecture** - Separation of concerns
6. ⭐ **Fee/pip value** - Simplified (acceptable architectural change)
7. ✅ **Strong test coverage** - 32+ comprehensive tests

### Recommended Actions

| Priority | Action | Estimado |
|----------|--------|----------|
| ⭐⭐⭐ HIGH | Complete documentation (this file + README.md) | 1 día ✅ |
| ⭐ LOW | Extended integration testing | 0.5 días |

**Total:** 1-1.5 días

---

**Última Actualización:** 2026-01-14
**Broker:** Woox
**Status:** ✅ EXCELENTE - 100% Hash Compatible
**Responsable:** Development Team
