# Ejemplos de Código - TradeLocker Normalizer

**Fecha:** 2026-01-14
**Broker:** TradeLocker
**Status:** ✅ 100% Hash Compatible - NO Critical Changes Required

---

## Tabla de Contenidos

1. [Hash Computation - ✅ 100% Compatible](#hash-computation)
2. [Status Filtering - ✅ Already Correct](#status-filtering)
3. [Side Mapping - ✅ Already Correct](#side-mapping)
4. [Symbol Normalization - Simplification](#symbol-normalization)
5. [Commission Logic - OSP Simplification](#commission-logic)
6. [Skip Conditions - Architecture Change](#skip-conditions)
7. [Timestamp Field - Semantic Change](#timestamp-field)
8. [Tests Examples](#tests-examples)

---

## Hash Computation

### ✅ STATUS: 100% COMPATIBLE (NO CHANGES NEEDED)

**Hallazgo Clave:** A diferencia de Oanda/Propreports (0% hash match), TradeLocker YA tiene 100% hash compatibility.

#### Legacy Implementation (tradelocker_export.py:564-567)

```python
# Lines 564-567
if 'id' in original_file_row and original_file_row['id']:
    n2 = original_file_row['id']
    njson2 = json.dumps(n2)
    njson2 = hashlib.md5(njson2.encode('utf-8')).hexdigest()
    # This hash is then used for deduplication
```

**Formula:** `MD5(json.dumps(id))`

#### New Implementation (tradelocker.py:100-102)

```python
# Lines 100-102
order_id = str(order.get("id", ""))

# Compute file_row hash using legacy formula:
# MD5(json.dumps(id))
file_row_hash = hashlib.md5(json.dumps(order_id).encode('utf-8')).hexdigest()
```

**Formula:** `MD5(json.dumps(id))` - **IDENTICAL** ✅

#### Verification

**User 49186 (12 records):** 100% hash match ✅

**Key Difference from Oanda/Propreports:**
- **Oanda:** Legacy incluía "buy/sell", nueva no → 0% match ⚠️⚠️⚠️
- **Propreports:** Legacy NO incluía portfolio, nueva SÍ → 0% match ⚠️⚠️⚠️
- **TradeLocker:** Ambas usan ID solo → 100% match ✅✅✅

#### Documentation in Code

```python
# tradelocker.py:9-26 - Clear documentation
"""
file_row Hash Formula (Legacy Compatibility):
---------------------------------------------
The file_row field is computed as an MD5 hash for deduplication against
the legacy TraderSync system. The formula is:

    file_row = MD5(json.dumps(order['id']))

Steps:
    1. Get the id field from the raw TradeLocker API response
    2. Hash: hashlib.md5(json.dumps(id).encode('utf-8')).hexdigest()

Note: The id field is a string (e.g., "7277816997868462674").
      No EXP_A suffix is used for TradeLocker.

Post-hash fields NOT included (added by legacy system):
    date_tz, broker, action, type_option, archive

Match Rate: 100% against legacy data (verified with user 49186, 12 records).
"""
```

---

## Status Filtering

### ✅ STATUS: ALREADY CORRECT

Both legacy and new implementations filter for "Filled" orders only.

#### Legacy Implementation (tradelocker_export.py:502-503)

```python
# Lines 502-503
if order['status'] != 'Filled':
    continue
```

#### New Implementation (tradelocker.py:94-96)

```python
# Lines 94-96
# Skip non-filled orders
status = str(order.get("status", "")).lower()
if status != "filled":
    continue
```

**Difference:** Case-insensitive check in new (more robust) ✅

**Result:** Both implementations correctly filter to "Filled" orders only.

---

## Side Mapping

### ✅ STATUS: ALREADY CORRECT

#### Legacy Implementation (tradelocker_export.py:480-483)

```python
# Lines 480-483
action = order['side'].upper()
# ... later ...
order['side'] = action
# Maps to 'BUY' or 'SELL'
```

#### New Implementation (tradelocker.py:178-182)

```python
# Lines 178-182 (in normalize method)
# side - map to BUY/SELL
pl.when(pl.col("side") == "buy")
.then(pl.lit("BUY"))
.otherwise(pl.lit("SELL"))
.alias("side"),
```

**Result:** Both map to "BUY" / "SELL" correctly ✅

---

## Symbol Normalization

### ⭐ STATUS: SIMPLIFIED (ACCEPTABLE)

Legacy had complex symbol formatting with position ID, new implementation is simpler.

#### Legacy Implementation (tradelocker_export.py:542-548)

```python
# Lines 542-548
try:
    if order['instruments']['type'] == 'CRYPTO':
        type = 'crypto'
        option = 'CRYPTO'
    else:
        # For forex: add position ID
        order['symbol'] = "${}|{}".format(order['symbol'], order['positionId'])
except:
    order['symbol'] = "${}|{}".format(order['symbol'], order['positionId'])
```

**Legacy Format:** `$EURUSD|123456` (forex with position ID)

#### New Implementation (tradelocker.py:175-176)

```python
# Lines 175-176
# symbol - uppercase
pl.col("symbol").str.to_uppercase().str.strip_chars().alias("symbol"),
```

**New Format:** `EURUSD` (simple, clean)

**Impact:** ⭐ BAJA - Symbol format más simple
**Action:** NINGUNA - Simplification is acceptable

---

## Commission Logic

### ⭐⭐ STATUS: OSP LOGIC SIMPLIFIED (REQUIRES VERIFICATION)

This is the ONLY medium-priority item that requires attention.

#### Legacy Implementation (tradelocker_export.py:488-497)

```python
# Lines 488-497
multiplier = 0
if self.passphrase == 'OSP-LIVE' or self.passphrase == 'OSP-DEMO':
    comm = {'':0, 'MINI':1, 'PRO':8, 'STN':7, 'VAR':0}

    # Check if currency is major and date >= threshold (March 26, 2024)
    if any(pair[0].startswith(currency) for currency in ["USD","EUR","GBP","CAD","AUD"]) \
       and (1711497600*1000) <= float(order['lastModified']):
        if len(pair) > 1:
            multiplier = comm[pair[1]]  # Get multiplier from pair suffix
        else:
            multiplier = 0

# Commission = quantity * multiplier for SELL, 0 for BUY
order['commission'] = (quantity * multiplier) if action == 'SELL' else 0
```

**OSP Commission Rules:**
- Only applies to OSP-LIVE or OSP-DEMO accounts
- Only for major currencies (USD, EUR, GBP, CAD, AUD)
- Only after March 26, 2024 (timestamp 1711497600000)
- Multipliers by pair suffix:
  - '' (no suffix): 0
  - MINI: 1
  - PRO: 8
  - STN: 7
  - VAR: 0
- Commission = quantity × multiplier (SELL only)

#### New Implementation (tradelocker.py:193-194)

```python
# Lines 193-194
# commission - not provided in order data
pl.lit(0.0).alias("commission"),
```

**New Logic:** Fixed 0.0 for all accounts

**Impact:** ⭐⭐ MEDIA - Only affects OSP accounts (if they exist)

#### Required Action: SQL Query

**Check if OSP accounts are active:**

```sql
SELECT
    COUNT(DISTINCT user_id) as osp_users,
    COUNT(*) as osp_trades,
    MAX(created_at) as last_trade
FROM import_files
WHERE broker_id = (SELECT broker_id FROM brokers WHERE broker_key = 'tradelocker')
  AND (
    metadata LIKE '%OSP-LIVE%' OR
    metadata LIKE '%OSP-DEMO%' OR
    account_id LIKE '%OSP%'
  )
  AND created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH);
```

**Decision:**
- **If OSP > 0 users:** Implement OSP commission logic (see below)
- **If OSP = 0 users:** Document as deprecated, no action needed

#### Optional Implementation (if OSP accounts exist)

**File:** `brokers/tradelocker/tradelocker.py`
**Lines:** ~194 (replace commission calculation)

```python
# Add to class constants
OSP_COMMISSION_MULTIPLIERS: ClassVar[dict] = {
    '': 0,
    'MINI': 1,
    'PRO': 8,
    'STN': 7,
    'VAR': 0
}
OSP_MAJOR_CURRENCIES: ClassVar[list] = ["USD", "EUR", "GBP", "CAD", "AUD"]
OSP_THRESHOLD_TIMESTAMP: ClassVar[int] = 1711497600000  # March 26, 2024

# In normalize() method, replace lines 193-194 with:
# commission - OSP logic if applicable
pl.when(
    # Check if OSP account (metadata would need to be passed)
    (pl.col("account_id").str.contains("OSP")) &
    # Check if major currency
    (pl.col("symbol").str.slice(0, 3).is_in(cls.OSP_MAJOR_CURRENCIES)) &
    # Check if after threshold date
    (pl.col("createdDate") >= cls.OSP_THRESHOLD_TIMESTAMP) &
    # Only SELL orders
    (pl.col("side") == "sell")
).then(
    # Extract pair suffix and calculate commission
    # This would require additional logic to parse symbol and get multiplier
    pl.col("quantity") * pl.col("multiplier_from_suffix")
).otherwise(
    pl.lit(0.0)
).alias("commission"),
```

**Tests Required (if implemented):**
```python
def test_osp_commission_calculation():
    """Test OSP commission logic for OSP-LIVE/OSP-DEMO accounts."""

def test_osp_commission_major_currencies_only():
    """Test commission only applied to major currencies."""

def test_osp_commission_after_threshold_date():
    """Test commission only for dates >= March 26, 2024."""

def test_osp_commission_sell_only():
    """Test commission only on SELL orders."""

def test_non_osp_accounts_zero_commission():
    """Test non-OSP accounts always have 0 commission."""
```

---

## Skip Conditions

### ✅ STATUS: ARCHITECTURE CHANGE (CORRECT)

Legacy had 8 skip points, new has 1. This is NOT a bug - it's proper separation of concerns.

#### Legacy Implementation (tradelocker_export.py:499-509, 562-594)

```python
# Line 500: Skip if symbol is None
if order['symbol'] == None:
    continue

# Line 503: Skip if status != "Filled"
if order['status'] != 'Filled':
    continue

# Line 507: Skip on parsing exception
except Exception as err:
    continue

# Line 509: Skip if action (side) is falsy
if not action:
    continue

# Lines 562, 570, 580, 594: Skip if duplicate (4-level cascade)
# Level 1: Check main hash
if ImportParams.verify_njson(njson, user_id, portfolio):
    verify_njson_len = verify_njson_len + 1
    continue

# Level 2: Check ID hash
if ImportParams.verify_njson(njson2, user_id, portfolio):
    verify_njson_len = verify_njson_len + 1
    continue

# Level 3: Check date+id
if ImportParams.verify_date_file_row(user_id, broker, original_file_row, 'id', portfolio):
    verify_njson_len = verify_njson_len + 1
    continue

# Level 4: Check webull-style hash
if ImportParams.verify_njson_webull(njson3, user_id):
    verify_njson_len = verify_njson_len + 1
    continue
```

**Legacy:** 8 skip points (includes deduplication in parser)

#### New Implementation (tradelocker.py:94-96)

```python
# Lines 94-96
# Skip non-filled orders
status = str(order.get("status", "")).lower()
if status != "filled":
    continue
```

**New:** 1 skip point (only status check)

**Where did deduplication go?** → **p02_deduplicate stage** ✅

#### Architecture Improvement

**Old Architecture (Mixed Concerns):**
```
tradelocker_export.py (616 lines):
  - API calls
  - Data parsing
  - Validation
  - Deduplication ← MIXED
  - Hash computation
  - Database writes
```

**New Architecture (Separation of Concerns):**
```
Pipeline Stages:
  p01_normalize/
    - tradelocker.py (249 lines): Parse + Transform ONLY

  p02_deduplicate/
    - deduplicate.py: Deduplication logic ONLY

  p03_group/
    - grouping.py: Group executions → trades

  p04_calculate/
    - calculations.py: P&L calculations

  p05_write/
    - writer.py: Database writes
```

**Result:** ✅ Architecture change is CORRECT - separation of concerns is best practice.

---

## Timestamp Field

### ⭐ STATUS: SEMANTIC CHANGE (ACCEPTABLE)

Legacy uses `lastModified`, new uses `createdDate`. Both are valid timestamps.

#### Legacy Implementation (tradelocker_export.py:484-485)

```python
# Lines 484-485
order_time = int(order['lastModified'])  # When order was last modified
order['date'] = datetime.utcfromtimestamp(order_time / 1000).isoformat()
```

**Field:** `lastModified` (milliseconds since epoch)
**Semantic:** Last time the order was modified

#### New Implementation (tradelocker.py:190-191)

```python
# Lines 190-191
# timestamp - createdDate is milliseconds since epoch
pl.col("createdDate").cast(pl.Int64).cast(pl.Datetime("ms")).alias("timestamp"),
```

**Field:** `createdDate` (milliseconds since epoch)
**Semantic:** When the order was created

#### Impact

**Difference:** Timestamps may differ slightly if orders were modified after creation.

**Example:**
- Order created: 2024-01-15 10:00:00
- Order modified: 2024-01-15 10:00:05
- Legacy timestamp: 10:00:05 (lastModified)
- New timestamp: 10:00:00 (createdDate)
- **Difference:** 5 seconds

**Impact:** ⭐ BAJA - Both timestamps are valid, different semantic meaning
**Action:** NINGUNA - Acceptable change

---

## Tests Examples

### Current Test Suite (test_tradelocker.py - 348 lines, 17 tests)

#### Test Class 1: TradeLockerInterpreter (8 tests)

```python
class TestTradeLockerInterpreter:
    """Test TradeLocker interpreter parsing and normalization."""

    def test_parse_json_content_valid(self):
        """Test parsing valid TradeLocker JSON."""
        json_content = json.dumps([{
            "id": "123456",
            "tradableInstrumentId": "1001",
            "symbol": "EURUSD",
            "side": "buy",
            "filledQty": 1000,
            "avgPrice": 1.0850,
            "createdDate": 1705392000000,
            "status": "Filled"
        }])

        df = TradeLockerInterpreter.parse_json_content(json_content)
        assert len(df) == 1
        assert df["symbol"][0] == "EURUSD"

    def test_parse_json_content_invalid(self):
        """Test parsing invalid JSON returns empty DataFrame."""

    def test_parse_valid_order(self):
        """Test parsing a valid filled order."""

    def test_parse_order_with_missing_fields(self):
        """Test handling orders with missing optional fields."""

    def test_normalize_basic(self):
        """Test basic normalization to 20-column schema."""

    def test_normalize_side_mapping(self):
        """Test side mapping (buy→BUY, sell→SELL)."""

    def test_normalize_empty_data(self):
        """Test normalization with empty DataFrame."""

    def test_normalize_with_nulls(self):
        """Test normalization handles null values correctly."""
```

#### Test Class 2: FileRowHash (5 tests)

```python
class TestFileRowHash:
    """Test file_row hash computation for legacy compatibility."""

    def test_hash_deterministic(self):
        """Test hash is deterministic (same input → same hash)."""
        json_content = json.dumps([{
            "id": "7277816997868462674",
            "tradableInstrumentId": "1001",
            "symbol": "EURUSD",
            "side": "buy",
            "filledQty": 1000,
            "avgPrice": 1.0850,
            "createdDate": 1705392000000,
            "status": "Filled"
        }])

        df1 = TradeLockerInterpreter.parse_json_content(json_content)
        df2 = TradeLockerInterpreter.parse_json_content(json_content)

        assert df1["_file_row_hash"][0] == df2["_file_row_hash"][0]

    def test_hash_uses_id_only(self):
        """Test hash only uses id field (legacy compatibility)."""
        # Verify that changing other fields doesn't change hash

    def test_hash_different_orders(self):
        """Test different order IDs produce different hashes."""

    def test_hash_legacy_compatibility(self):
        """Test hash matches legacy formula: MD5(json.dumps(id))."""
        order_id = "7277816997868462674"
        expected_hash = hashlib.md5(json.dumps(order_id).encode('utf-8')).hexdigest()

        json_content = json.dumps([{
            "id": order_id,
            # ... other fields ...
        }])

        df = TradeLockerInterpreter.parse_json_content(json_content)
        assert df["_file_row_hash"][0] == expected_hash

    def test_hash_user_49186_compatibility(self):
        """Test hash matches verified user 49186 data (12 records, 100% match)."""
        # This test verifies against real production data
```

#### Test Class 3: Detector (2 tests)

```python
class TestDetector:
    """Test format detection."""

    def test_can_handle_valid_tradelocker(self):
        """Test detector recognizes valid TradeLocker data."""

    def test_cannot_handle_invalid(self):
        """Test detector rejects invalid data."""
```

#### Test Class 4: EdgeCases (2 tests)

```python
class TestEdgeCases:
    """Test edge cases and error handling."""

    def test_status_filtering(self):
        """Test only 'Filled' orders are processed."""
        json_content = json.dumps([
            {"id": "1", "status": "Filled", ...},     # Should be included
            {"id": "2", "status": "Pending", ...},    # Should be skipped
            {"id": "3", "status": "Cancelled", ...},  # Should be skipped
        ])

        df = TradeLockerInterpreter.parse_json_content(json_content)
        assert len(df) == 1  # Only Filled order
        assert df["id"][0] == "1"

    def test_timestamp_conversion(self):
        """Test milliseconds epoch → datetime conversion."""
```

### Optional Tests (if OSP commission implemented)

```python
class TestOSPCommission:
    """Test OSP commission calculation logic."""

    def test_osp_commission_calculation(self):
        """Test OSP commission = quantity × multiplier for SELL."""
        # Test with OSP-LIVE account, EURUSD.PRO, SELL
        # Expected: commission = 1000 × 8 = 8000

    def test_osp_commission_major_currencies_only(self):
        """Test commission only for USD, EUR, GBP, CAD, AUD."""
        # Test with JPY (non-major) → commission = 0

    def test_osp_commission_after_threshold_date(self):
        """Test commission only for dates >= March 26, 2024."""
        # Before threshold: commission = 0
        # After threshold: commission = quantity × multiplier

    def test_osp_commission_sell_only(self):
        """Test commission only on SELL orders."""
        # BUY: commission = 0
        # SELL: commission = quantity × multiplier

    def test_non_osp_accounts_zero_commission(self):
        """Test non-OSP accounts always have commission = 0."""

    def test_osp_multiplier_by_suffix(self):
        """Test multipliers: MINI=1, PRO=8, STN=7, VAR=0, ''=0."""
```

### Running Tests

```bash
# Run all tests
pytest tests/brokers/test_tradelocker.py -v

# Run specific test class
pytest tests/brokers/test_tradelocker.py::TestFileRowHash -v

# Run with coverage
pytest tests/brokers/test_tradelocker.py --cov=brokers.tradelocker --cov-report=html

# Run specific test
pytest tests/brokers/test_tradelocker.py::TestFileRowHash::test_hash_legacy_compatibility -v
```

---

## Summary

### Code Changes Summary

| Item | Legacy | New | Status | Action |
|------|--------|-----|--------|--------|
| **Hash Computation** | MD5(id) | MD5(id) | ✅ 100% MATCH | NINGUNA |
| **Status Filtering** | "Filled" | "filled" (case-insensitive) | ✅ BETTER | NINGUNA |
| **Side Mapping** | BUY/SELL | BUY/SELL | ✅ MATCH | NINGUNA |
| **Symbol** | Complex + position ID | Simple uppercase | ⭐ SIMPLIFIED | NINGUNA |
| **Commission** | OSP logic | Fixed 0.0 | ⭐⭐ SIMPLIFIED | VERIFY OSP |
| **Skip Conditions** | 8 (+ dedup) | 1 (dedup in p02) | ✅ ARCHITECTURE | NINGUNA |
| **Timestamp** | lastModified | createdDate | ⭐ DIFFERENT | NINGUNA |

### Lines of Code

| File | Legacy | New | Reduction |
|------|--------|-----|-----------|
| **Implementation** | 616 | 249 | 60% ✅ |
| **Tests** | Minimal | 348 (17 tests) | N/A ✅ |
| **Total** | 616 | 597 | 3% |

**Key:** Effective implementation reduction is 60% (616 → 249), with comprehensive test coverage added.

### Critical Findings

1. ✅✅✅ **Hash 100% Compatible** - NO critical issues (unlike Oanda/Propreports)
2. ✅ **All validations already implemented** - Status filter, side mapping, etc.
3. ✅ **Modern architecture** - Separation of concerns
4. ⭐⭐ **OSP commission** - Only potential issue (requires verification)
5. ✅ **Strong test coverage** - 17 comprehensive tests

### Recommended Actions

| Priority | Action | Estimado |
|----------|--------|----------|
| ⭐⭐⭐ HIGH | Complete documentation (this file + README.md) | 1 día ✅ |
| ⭐⭐ MEDIUM | Execute SQL query for OSP verification | 0.5 días |
| ⭐⭐ MEDIUM | Implement OSP logic (IF needed) | 1-2 días |
| ⭐ LOW | Extended integration testing | 0.5 días |

**Total:** 2-4 días (depending on OSP requirement)

---

**Última Actualización:** 2026-01-14
**Broker:** TradeLocker
**Status:** ✅ EXCELENTE - 100% Hash Compatible
**Responsable:** Development Team
