# Archivos de Código - Tastyworks Broker

## Estructura de Archivos

```
brokers/tastyworks/
├── tastyworks.py.original    # Código actual (sin modificar) - 428 líneas
├── detector.py               # Detector de formato (sin cambios)
└── lookups.py               # NUEVO - Lookup tables (Fase 2)
```

## Archivos Incluidos

### 1. tastyworks.py.original
**Descripción:** Código actual del normalizador Tastyworks sin modificaciones
**Líneas:** 428
**Estado:** REFERENCIA - No modificado

**Validaciones actuales:**
- ✅ Detección de formato JSON
- ✅ Parsing de campos básicos
- ✅ Transformación a schema normalizado
- ✅ File row hash (MD5)
- ✅ Timezone conversion (UTC → Eastern)
- ✅ Filtrado por transaction-type (Trade/Receive Deliver)
- ✅ Parsing de option symbols
- ✅ Cálculo básico de fees

**Validaciones faltantes (a implementar):**
- ❌ Validación de símbolos numéricos
- ❌ Reemplazo RUTW→RUT
- ❌ Inferencia de acción desde descripción
- ❌ Filtrado assignment/exercise con cash settlement
- ❌ Normalización de precio con precisión
- ❌ Detección de cash settlement
- ❌ Cálculo de tamaño para future options

### 2. detector.py
**Descripción:** Detector de formato Tastyworks
**Estado:** SIN CAMBIOS - No requiere modificación

### 3. lookups.py (NUEVO - Fase 2)
**Descripción:** Lookup tables y constantes para Tastyworks
**Estado:** PENDIENTE CREACIÓN

**Contenido planificado:**
- Symbol replacements (RUTW→RUT)
- Future option contract sizes (/ES, /NQ, /YM, etc.)
- User-specific edge cases

## Cambios Propuestos por Fase

### FASE 1: Validaciones Críticas

#### Cambio 1: Validación de Símbolos Numéricos
**Ubicación:** Línea ~326 en tastyworks.py
**Antes:**
```python
.with_columns([
    pl.col("symbol").str.to_uppercase().str.strip_chars().alias("symbol")
])
```

**Después:**
```python
# Rechazar símbolos numéricos
.filter(~pl.col("symbol").cast(pl.Float64, strict=False).is_not_null())
# Reemplazar RUTW → RUT
.with_columns([
    pl.col("symbol")
    .str.replace("RUTW", "RUT")
    .str.to_uppercase()
    .str.strip_chars()
    .alias("symbol")
])
```

#### Cambio 2: Inferencia de Acción desde Descripción
**Ubicación:** Línea ~333 en tastyworks.py
**Antes:**
```python
.with_columns([
    pl.col("action").replace_strict(self.ACTION_MAP, default="BUY").alias("side")
])
```

**Después:**
```python
.with_columns([
    pl.when(pl.col("description").str.contains("(?i)expiration"))
    .then(pl.lit("EXP"))
    .when(pl.col("description").str.contains("(?i)assignment"))
    .then(pl.lit("BUY"))
    .when(pl.col("description").str.contains("(?i)exercise"))
    .then(pl.lit("SELL"))
    .otherwise(pl.col("action").replace_strict(self.ACTION_MAP, default="BUY"))
    .alias("side")
])
```

#### Cambio 3: Filtrado Assignment/Exercise con Cash Settlement
**Ubicación:** Línea ~280 en tastyworks.py (después de filtro transaction-type)
**Añadir:**
```python
# Identificar símbolos con cash settlement
cash_settlement_df = (
    df.filter(pl.col("description").str.contains("(?i)cash settlement"))
    .select("symbol")
    .unique()
)

# Si hay cash settlements, filtrar assignments/exercises correspondientes
if len(cash_settlement_df) > 0:
    cash_settlement_symbols = cash_settlement_df.collect()["symbol"].to_list()
    df = df.filter(
        ~(
            pl.col("description").str.contains("(?i)assignment|exercise") &
            pl.col("symbol").is_in(cash_settlement_symbols)
        )
    )
```

### FASE 2: Validaciones de Calidad

#### Cambio 4: Normalización de Precio
**Ubicación:** Línea ~355 en tastyworks.py
**Añadir después de price processing:**
```python
.with_columns([
    pl.when(pl.col("price").is_not_null() & (pl.col("price") > 0))
    .then(pl.col("price").round(6))
    .otherwise(pl.lit(0.0))
    .alias("price")
])
```

#### Cambio 5: Detección de Cash Settlement
**Ubicación:** Línea ~270 en tastyworks.py (en normalized output)
**Añadir columna:**
```python
.with_columns([
    pl.when(pl.col("description").str.contains("(?i)cash settlement"))
    .then(pl.lit(True))
    .otherwise(pl.lit(False))
    .alias("_is_cash_settlement")
])
```

#### Cambio 6: Future Option Size
**Ubicación:** Línea ~389 en tastyworks.py
**Requiere:** Crear lookups.py primero
**Modificar cálculo de multiplier:**
```python
from .lookups import FUTURE_OPTION_SIZES

# En el método normalize()
.with_columns([
    pl.when(pl.col("instrument-type") == "Future Option")
    .then(
        pl.when(pl.col("value").is_not_null())
        .then((pl.col("value").abs() / (pl.col("price") * pl.col("quantity"))).round(2))
        .otherwise(
            pl.col("underlying-symbol").replace_strict(FUTURE_OPTION_SIZES, default=100.0)
        )
    )
    .otherwise(
        pl.when(pl.col("instrument-type") == "Equity Option")
        .then(pl.lit(100.0))
        .otherwise(pl.lit(1.0))
    )
    .alias("multiplier")
])
```

## Implementación

### Para implementar los cambios:

1. **Fase 1 (Crítica):**
   ```bash
   # Copiar archivo original
   cp tastyworks.py.original tastyworks.py

   # Aplicar cambios 1, 2, 3
   # Editar tastyworks.py según cambios propuestos arriba
   ```

2. **Fase 2 (Calidad):**
   ```bash
   # Crear lookups.py
   # Aplicar cambios 4, 5, 6
   ```

3. **Testing:**
   ```bash
   # Ejecutar tests
   pytest ../../tests/brokers/test_tastyworks.py -v

   # Validación QA
   python qa/validate_broker.py --broker-id 26
   ```

## Referencias

- **Plan completo:** `../PLAN_ANALISIS_VALIDACIONES_TASTYWORKS.md`
- **Código legacy:** `../brokers_tastyworks.py`, `../tastyworks_export.py`
- **Log de cambios:** `../CAMBIOS_IMPLEMENTADOS.md`
