# Plan: Análisis de Validaciones Críticas Faltantes - Tastyworks Normalizer

## Resumen Ejecutivo

Tras analizar exhaustivamente ambas implementaciones (legacy: 1,833 líneas en `brokers_tastyworks.py` + `tastyworks_export.py` vs nueva: 446 líneas en `tastyworks.py` + `detector.py`), se identificaron **10 validaciones críticas** que faltan en la nueva implementación y podrían comprometer la integridad de datos.

## Contexto de la Migración

### Arquitectura Legacy
- Dos archivos principales: `brokers_tastyworks.py` (5 parsers CSV) + `tastyworks_export.py` (API sync)
- 5 formatos CSV diferentes con cadena de fallback (getcsv → getcsv2 → getcsv3 → getcsv4 → getcsv5)
- Validación de cuenta desde filename pattern + verificación cruzada con BD
- Deduplicación MD5 con manejo especial para usuarios específicos (47132, 242106, 49127, 137705)
- Procesamiento de Cash Settlement con actualizaciones de trades relacionados
- Filtrado Assignment/Exercise para prevenir duplicados

### Arquitectura Nueva
- Archivo único: `tastyworks.py` (normalización JSON API)
- Enfoque en parsing JSON de API (órdenes ya obtenidas)
- Arquitectura modular basada en BaseInterpreter
- Separación de responsabilidades: normalización en p01, deduplicación en p02, cálculos en p04
- File row hash (MD5) con compatibilidad legacy al 93% (7% con sufijo EXP_A)
- Manejo unificado de todas las transacciones

### Arquitectura del Pipeline (Confirmada)
```
p01_normalize → p02_deduplicate → p03_group → p04_calculate → p05_write
     ↓                 ↓                ↓            ↓             ↓
  Parse JSON      Hash check      Group legs    P&L calc      DB write
  Normalize       vs DB           Spreads       Updates
  Validate        Skip dups       Orders        Settlement
```

---

## Validaciones Críticas Faltantes

### CATEGORÍA 1: VALIDACIONES DE INTEGRIDAD DE DATOS [CRITICIDAD: ALTA]

#### ❌ 1. Validación de Símbolos Numéricos
**Ubicación Legacy:** `brokers_tastyworks.py:163, 167, 508`
```python
if ImportParams.isfloat(n['symbol']) or ImportParams.isfloat(n['symbol'][:4]):
    continue  # Skip numeric symbols
n['symbol'] = n['symbol'].replace('RUTW','RUT')  # Symbol replacement
```

**Descripción:** El legacy rechaza símbolos que sean completamente numéricos o tengan los primeros 4 caracteres numéricos. También reemplaza `RUTW` → `RUT` para compatibilidad.

**Impacto si falta:**
- Trades con símbolos inválidos/numéricos en base de datos
- Errores en cálculos de posición
- Fallos en matching con datos de mercado
- Símbolo RUTW no se normaliza correctamente

**Nivel de Criticidad:** 🔴 **ALTA** - Afecta integridad de símbolos

**Recomendación de implementación:**
```python
# En tastyworks.py, método normalize()
.filter(~pl.col("symbol").cast(pl.Float64, strict=False).is_not_null())  # No numeric
.with_columns([pl.col("symbol").str.replace("RUTW", "RUT")])  # Legacy mapping
```

**Ubicación sugerida:** `tastyworks.py` línea 326, antes de transformaciones de símbolo

---

#### ❌ 2. Inferencia de Acción desde Descripción
**Ubicación Legacy:** `brokers_tastyworks.py:144-159`
```python
n['action'] = 'EXP' if 'EXPIRATION' in n['description'].upper() \
    else 'BUY' if 'ASSIGNMENT' in n['description'].upper() \
    else 'SELL' if 'EXERCISE' in n['description'].upper() \
    else n['action']
```

**Descripción:** Para eventos de opciones especiales (expiración, assignment, exercise), el legacy infiere la acción desde el campo description en lugar del campo action estándar.

**Impacto si falta:**
- Acciones incorrectas para expiraciones de opciones (EXP no detectado)
- Assignments marcados como acción incorrecta
- Exercises no clasificados como SELL
- Errores en cálculo de P&L para eventos de opciones

**Nivel de Criticidad:** 🔴 **ALTA** - Afecta clasificación correcta de eventos de opciones

**Recomendación de implementación:**
```python
# En tastyworks.py, método normalize()
pl.when(pl.col("description").str.contains("(?i)expiration"))
.then(pl.lit("EXP"))
.when(pl.col("description").str.contains("(?i)assignment"))
.then(pl.lit("BUY"))
.when(pl.col("description").str.contains("(?i)exercise"))
.then(pl.lit("SELL"))
.otherwise(pl.col("action").replace_strict(self.ACTION_MAP))
.alias("side")
```

**Ubicación sugerida:** `tastyworks.py` línea 333, donde se mapea side

---

#### ❌ 3. Filtrado de Assignment/Exercise Duplicados
**Ubicación Legacy:** `brokers_tastyworks.py:225-228`
```python
if 'ASSIGNMENT' in description.upper() or 'EXERCISE' in description.upper():
    row_index_a = next((index for (index, d) in enumerate(result)
        if d['symbol'] == n['symbol'] and 'Cash settlement' in d['description']), -1)
    if row_index_a >= 0:
        continue  # Skip if cash settlement exists
```

**Descripción:** Si existe un registro de "Cash settlement" para un símbolo, se debe saltar el registro de "Assignment" o "Exercise" para evitar duplicados (el cash settlement es preferido).

**Impacto si falta:**
- Trades duplicados para assignments/exercises con cash settlement
- P&L calculado dos veces para el mismo evento
- Posiciones incorrectas (contadas doblemente)
- Métricas de volumen infladas

**Nivel de Criticidad:** 🔴 **ALTA** - Previene duplicados críticos

**Recomendación de implementación:**
```python
# En tastyworks.py, método normalize()
# Identificar símbolos con cash settlement
cash_settlement_symbols = (
    df.filter(pl.col("description").str.contains("(?i)cash settlement"))
    .select("symbol").unique()
)

# Filtrar assignments/exercises si existe cash settlement
.filter(
    ~(
        pl.col("description").str.contains("(?i)assignment|exercise") &
        pl.col("symbol").is_in(cash_settlement_symbols["symbol"])
    )
)
```

**Ubicación sugerida:** `tastyworks.py` línea 280, después de filtrar por transaction-type

---

### CATEGORÍA 2: VALIDACIONES DE CALIDAD DE DATOS [CRITICIDAD: MEDIA]

#### ⚠️ 4. Validación y Normalización de Precio
**Ubicación Legacy:** `brokers_tastyworks.py:255-260`
```python
fp = str(n['price']).replace('-','')
decimal = fp[::-1].find('.')
decimal = decimal if decimal > 1 else 2  # Min 2 decimal places
price = round(float(fp), 6) if ImportParams.isfloat(fp) else 0.00
```

**Descripción:** Normaliza precios eliminando guiones, valida decimales (mínimo 2 lugares), redondea a 6 decimales, default 0.00 si inválido.

**Impacto si falta:**
- Pérdida de precisión en precios
- Precios inválidos no manejados correctamente
- Inconsistencia con legacy en formato de precios

**Nivel de Criticidad:** 🟡 **MEDIA** - Afecta precisión de display

**Recomendación de implementación:**
```python
# En tastyworks.py, método normalize()
.with_columns([
    pl.when(pl.col("price").is_not_null() & (pl.col("price") > 0))
    .then(pl.col("price").round(6))
    .otherwise(pl.lit(0.0))
    .alias("price")
])
```

**Ubicación sugerida:** `tastyworks.py` línea 355, donde se procesa price

---

#### ⚠️ 5. Detección de Cash Settlement
**Ubicación Legacy:** `brokers_tastyworks.py:229-235`
```python
if 'Cash settlement' in n['description']:
    cash_settlement = ImportParams.Tastyworks_cash_settlement(
        user_id, original_file_row, portfolio=portfolio)
    if cash_settlement:
        update_trade = UpdateTrade.getTrade(cash_settlement)
```

**Descripción:** Detecta transacciones de "Cash settlement" y llama a módulo especializado para actualizar trades relacionados.

**Impacto si falta:**
- Trades de cash settlement no actualizan posiciones relacionadas
- P&L incorrecto para opciones cash-settled
- Posiciones no cerradas correctamente

**Nivel de Criticidad:** 🟡 **MEDIA-ALTA** - Afecta P&L de opciones específicas

**Recomendación de implementación:**
```python
# FASE 1: En tastyworks.py, añadir flag de detección
.with_columns([
    pl.when(pl.col("description").str.contains("(?i)cash settlement"))
    .then(pl.lit(True))
    .otherwise(pl.lit(False))
    .alias("_is_cash_settlement")
])

# FASE 2: En p04_calculate, implementar lógica de actualización
# (Requiere investigación del módulo ImportParams.Tastyworks_cash_settlement)
```

**Ubicación sugerida:**
- Fase 1: `tastyworks.py` línea 270 (detección)
- Fase 2: `p04_calculate/calculation.py` (procesamiento - INVESTIGACIÓN REQUERIDA)

---

#### ⚠️ 6. Cálculo de Tamaño para Future Options
**Ubicación Legacy:** `brokers_tastyworks.py:295-305`
```python
if n['instrument type'].upper() == 'FUTURE OPTION':
    if 'amount' in n and ImportParams.isfloat(n['amount']):
        amount = float(n['amount'])
        value = float(price) * quantity
        n['size'] = abs(round(amount / value, 2))
    else:
        future_params_symbol = ImportParams.getFutureSymbolOption(...)
        n['size'] = future_params_symbol['size']
```

**Descripción:** Las Future Options tienen multiplicadores variables (no siempre 100). Calcula: size = amount / (price × quantity), o usa lookup table.

**Impacto si falta:**
- Multiplicadores incorrectos para future options
- Posiciones calculadas erróneamente
- P&L incorrecto para /ES, /NQ y otros futuros

**Nivel de Criticidad:** 🟡 **MEDIA** - Afecta clase de activo específica

**Recomendación de implementación:**
```python
# Crear lookup table en nuevo archivo lookups.py
FUTURE_OPTION_SIZES = {
    "/ES": 50,   # E-mini S&P 500
    "/NQ": 20,   # E-mini NASDAQ
    "/YM": 5,    # E-mini Dow
    "/RTY": 50,  # E-mini Russell 2000
}

# En tastyworks.py, método normalize()
.with_columns([
    pl.when(pl.col("instrument-type") == "Future Option")
    .then(
        pl.when(pl.col("value").is_not_null())
        .then((pl.col("value").abs() / (pl.col("price") * pl.col("quantity"))).round(2))
        .otherwise(
            pl.col("underlying-symbol").replace_strict(FUTURE_OPTION_SIZES, default=100.0)
        )
    )
    .otherwise(
        pl.when(pl.col("instrument-type") == "Equity Option").then(pl.lit(100.0))
        .otherwise(pl.lit(1.0))
    )
    .alias("multiplier")
])
```

**Ubicación sugerida:** `tastyworks.py` línea 389, donde se calcula multiplier

---

### CATEGORÍA 3: VALIDACIONES DE SEGURIDAD Y NEGOCIO [CRITICIDAD: MEDIA-BAJA]

#### ⚠️ 7. Verificación de Código de Cuenta
**Ubicación Legacy:** `brokers_tastyworks.py:83-112`
```python
account_code = filename.split('tastyworks_transactions_x')[1].split('_')[0]
error, error_code, portfolio_id = ReferenceVerify.verify_code(
    account_code, user_id, broker_id, user_portfolio)
if error:
    return out_result, any_error
```

**Descripción:** Extrae código de cuenta del nombre de archivo, valida formato numérico, verifica cross-reference con base de datos, valida asociación usuario-portfolio.

**Impacto si falta:**
- Riesgo de importar datos a portfolio incorrecto
- Sin validación de que cuenta pertenece al usuario
- **NOTA:** Formato JSON API ya viene autenticado desde sesión

**Nivel de Criticidad:** ⚠️ **MEDIA-CONDICIONAL** - Crítico para CSV uploads, no aplicable para JSON API

**Recomendación de implementación:**
```python
# INVESTIGAR: ¿Se usa CSV upload o solo JSON API?
# Si se usa CSV upload, agregar validación en handler.py:
# - Extraer account_code de metadata
# - Validar contra DB antes de llamar normalize()
# Si solo JSON API, NO IMPLEMENTAR (ya autenticado)
```

**Ubicación sugerida:** `handler.py` (pre-normalización) o NO REQUERIDO

**DECISIÓN NECESARIA:** ¿Se soportan uploads CSV para Tastyworks en el nuevo sistema?

---

### CATEGORÍA 4: VALIDACIONES ARQUITECTURALES [YA MANEJADAS]

#### ✅ 8. Validación de File Row contra Importaciones Existentes
**Ubicación Legacy:** `brokers_tastyworks.py:367-382`
**Status:** ✅ **YA IMPLEMENTADO** en pipeline

**Descripción:** El legacy valida el hash MD5 del file_row contra registros existentes en BD durante normalización.

**Nueva Arquitectura:**
```
p01_normalize: Calcula file_row hash (tastyworks.py:214) ✅
p02_deduplicate: Valida contra DB usando SHA256 ✅
    - deduplicate_within_batch() para duplicados en batch
    - deduplicate_against_db() para duplicados vs DB
```

**Verificación Requerida:**
- ✅ Confirmar que p02_deduplicate/deduplication.py usa `original_file_row_hash` correctamente
- ✅ Confirmar que el hash MD5 en p01 se convierte a SHA256 en p02
- ⚠️ ACCIÓN: Revisar que el hash legacy (MD5) es compatible con nuevo (SHA256)

**CONCLUSIÓN:** NO requiere cambios en normalizer. Separación de responsabilidades correcta.

---

### CATEGORÍA 5: EDGE CASES Y COMPATIBILIDAD LEGACY [CRITICIDAD: BAJA]

#### 🔵 9. Manejo Especial de Usuarios Duplicados
**Ubicación Legacy:** `brokers_tastyworks.py:170-195`
```python
if file_row_dict and njson == file_row_dict[-1] and str(user_id) in ['47132','242106','49127','137705']:
    # Adjust timestamp by +1 second
    dt = datetime.strptime(date_tz, "%Y-%m-%dT%H:%M:%S%z")
    dt += timedelta(seconds=1)
    date_tz = dt.strftime("%Y-%m-%dT%H:%M:%S%z")
```

**Descripción:** Para 4 usuarios específicos (47132, 242106, 49127, 137705), ajusta timestamp +1 segundo si se detecta duplicado consecutivo.

**Impacto si falta:**
- Posibles duplicados no resueltos para estos 4 usuarios legacy
- **NOTA:** Workaround histórico, probablemente no necesario en nueva arquitectura

**Nivel de Criticidad:** 🔵 **BAJA** - Edge case histórico

**Recomendación de implementación:**
```python
# INVESTIGAR: ¿Estos usuarios siguen activos?
# Si sí, agregar en tastyworks.py línea 208:
user_ids_with_duplicate_issue = {47132, 242106, 49127, 137705}
if user_id in user_ids_with_duplicate_issue:
    # Add duplicate detection logic con timestamp adjustment
```

**DECISIÓN NECESARIA:** Consultar DB para verificar si estos usuarios siguen activos

---

#### 🔵 10. Validación de Contenido Vacío
**Ubicación Legacy:** `brokers_tastyworks.py:57, 712, 904, 921`
```python
if len(result_b) <= 1:
    return out_result, 1202  # Error code: empty content
```

**Descripción:** Retorna error code 1202 si el contenido CSV está vacío o solo tiene headers.

**Impacto si falta:**
- Error handling menos explícito (nuevo usa logging en vez de códigos)
- **NOTA:** Nueva implementación ya maneja con `if not orders:` (línea 153)

**Nivel de Criticidad:** 🔵 **BAJA** - Ya manejado de forma diferente

**Recomendación:** NO IMPLEMENTAR - Sistema nuevo usa excepciones/logging en vez de códigos de error numéricos

---

## Diferencias No Críticas (No Requieren Implementación)

| Funcionalidad | Legacy | Nuevo | Razón |
|---------------|--------|-------|-------|
| Soporte Multi-CSV | 5 parsers | Solo JSON API | Arquitectura cambió a API-only |
| Sistema de Error Codes | Códigos numéricos | Excepciones + logging | Mejora de arquitectura |
| 2FA y Autenticación | En normalizer | Lambda/handler layer | Separación de responsabilidades |
| Paginación API | En export | Lambda/handler layer | Separación de responsabilidades |
| Trade Grouping | En parser | p03_group stage | Separación de responsabilidades |

---

## Plan de Implementación por Fases

### **FASE 1: Validaciones Críticas de Integridad** (Prioridad ALTA - Semana 1)

**Objetivo:** Prevenir datos inválidos/duplicados en el pipeline

**Tareas:**
1. ✅ Validación de símbolos numéricos + reemplazo RUTW→RUT
2. ✅ Inferencia de acción desde descripción (EXP/ASSIGNMENT/EXERCISE)
3. ✅ Filtrado de Assignment/Exercise cuando existe Cash Settlement

**Validaciones:**
- ❌ 1. Validación de Símbolos Numéricos
- ❌ 2. Inferencia de Acción desde Descripción
- ❌ 3. Filtrado de Assignment/Exercise Duplicados

**Archivos a Modificar:**
- `brokers/tastyworks/tastyworks.py` (método `normalize()`)

**Métricas de Éxito:**
- Sin símbolos numéricos en output normalizado
- Eventos de opciones (EXP/ASSIGNMENT/EXERCISE) correctamente clasificados
- Sin duplicados cuando existen cash settlements para mismo símbolo
- Match rate ≥ baseline (sin regresión)

**Complejidad:** BAJA - Operaciones Polars autocontenidas
**Riesgo:** BAJO - Sin dependencias externas
**Estimado:** 2-3 días

---

### **FASE 2: Validaciones de Calidad de Datos** (Prioridad MEDIA - Semana 2)

**Objetivo:** Mejorar precisión y correctitud de datos

**Tareas:**
1. ✅ Normalización de precio con precisión 6 decimales
2. ✅ Detección de Cash Settlement (flag para p04)
3. ⚠️ Cálculo de multiplicador para Future Options (condicional - requiere investigación)

**Validaciones:**
- ⚠️ 4. Validación y Normalización de Precio
- ⚠️ 5. Detección de Cash Settlement
- ⚠️ 6. Cálculo de Tamaño para Future Options

**Archivos a Modificar:**
- `brokers/tastyworks/tastyworks.py` (método `normalize()`)
- `brokers/tastyworks/lookups.py` (NUEVO - lookup tables para future options)

**Métricas de Éxito:**
- Precisión de precio consistente (6 decimales)
- Cash settlements detectados con flag `_is_cash_settlement`
- Multiplicadores correctos para future options (si aplicable)
- Match rate mejorado (+2-5%)

**Complejidad:** BAJA-MEDIA
**Riesgo:** BAJO - Investigación requerida para future options
**Estimado:** 3-4 días

---

### **FASE 3: Lógica de Negocio y Actualizaciones** (Prioridad MEDIA-BAJA - Semana 3)

**Objetivo:** Implementar actualizaciones de trades relacionados

**Tareas:**
1. ⚠️ Investigar módulo `ImportParams.Tastyworks_cash_settlement()` en legacy
2. ⚠️ Implementar actualización de trades para cash settlements en p04_calculate
3. ⚠️ Investigar si verificación de cuenta es necesaria (CSV uploads vs JSON API)
4. ⚠️ Verificar si usuarios especiales (47132, 242106, 49127, 137705) siguen activos

**Validaciones:**
- ⚠️ 5. Procesamiento de Cash Settlement (Fase 2)
- ⚠️ 7. Verificación de Código de Cuenta (Condicional)
- 🔵 9. Manejo Especial de Usuarios Duplicados (Condicional)

**Archivos a Modificar:**
- `p04_calculate/calculation.py` (lógica de cash settlement)
- `handler.py` (validación de cuenta - si requerido)

**Métricas de Éxito:**
- Cash settlements actualizan trades correctamente
- Validación de cuenta funciona (si implementado)
- Match rate final ≥95%

**Complejidad:** ALTA - Requiere coordinación cross-módulo
**Riesgo:** MEDIO - Depende de módulos no explorados
**Estimado:** 5-7 días (incluye investigación)

---

### **FASE 4: Verificación y Testing** (Continuo)

**Objetivo:** Validar que cambios no introducen regresiones

**Tareas:**
1. ✅ Tests unitarios para cada validación agregada
2. ✅ Tests de integración end-to-end
3. ✅ Validación contra datos legacy de usuarios reales
4. ✅ Verificación de match rate con script QA

**Testing por Validación:**

```python
# test_tastyworks.py - Agregar ~15 tests nuevos

# FASE 1 Tests
def test_symbol_validation_rejects_numeric():
    """Verifica que símbolos numéricos son rechazados"""
    # Validación #1

def test_symbol_rutw_replacement():
    """Verifica reemplazo RUTW → RUT"""
    # Validación #1

def test_action_inference_expiration():
    """Verifica inferencia EXP desde description"""
    # Validación #2

def test_action_inference_assignment():
    """Verifica inferencia BUY desde ASSIGNMENT"""
    # Validación #2

def test_action_inference_exercise():
    """Verifica inferencia SELL desde EXERCISE"""
    # Validación #2

def test_assignment_filtered_when_cash_settlement_exists():
    """Verifica filtrado de assignment si hay cash settlement"""
    # Validación #3

# FASE 2 Tests
def test_price_normalization_precision():
    """Verifica redondeo a 6 decimales"""
    # Validación #4

def test_price_validation_handles_invalid():
    """Verifica manejo de precios inválidos (default 0.0)"""
    # Validación #4

def test_cash_settlement_detection_flag():
    """Verifica flag _is_cash_settlement"""
    # Validación #5

def test_future_option_multiplier_calculation():
    """Verifica cálculo de size para future options"""
    # Validación #6

def test_future_option_multiplier_lookup():
    """Verifica lookup table para /ES, /NQ, etc."""
    # Validación #6

# Integration Tests
def test_full_normalize_pipeline_with_validations():
    """Test end-to-end con todas las validaciones"""

def test_legacy_compatibility_match_rate():
    """Compara output con legacy para mismo dataset"""
```

**Validación QA:**
```bash
# Ejecutar validation script
python qa/validate_broker.py \
    --broker-id 26 \
    --user-id 99999 \
    --legacy-user-id <legacy_user> \
    --file-path "s3://bucket/tastyworks_test.json"

# Match rates esperados:
# - Fase 1: ≥ baseline (sin regresión)
# - Fase 2: baseline + 2-5%
# - Fase 3: ≥ 95%
```

---

## Matriz de Priorización

| # | Validación | Criticidad | Complejidad | Fase | Bloqueante | Días |
|---|------------|-----------|-------------|------|-----------|------|
| 1 | Símbolos Numéricos | 🔴 ALTA | BAJA | 1 | No | 1 |
| 2 | Acción desde Descripción | 🔴 ALTA | BAJA | 1 | No | 1 |
| 3 | Assignment/Exercise Filter | 🔴 ALTA | MEDIA | 1 | No | 1 |
| 4 | Precio Normalización | 🟡 MEDIA | BAJA | 2 | No | 1 |
| 5 | Cash Settlement Detección | 🟡 MEDIA-ALTA | BAJA | 2 | No | 1 |
| 6 | Future Option Size | 🟡 MEDIA | MEDIA | 2 | No | 2 |
| 5b | Cash Settlement Updates | 🟡 MEDIA-ALTA | ALTA | 3 | Sí (investigación) | 5 |
| 7 | Account Verification | ⚠️ CONDICIONAL | MEDIA | 3 | Sí (decisión) | 2 |
| 8 | File Row Dedup | ✅ YA MANEJADO | - | - | No | 0 |
| 9 | Usuarios Especiales | 🔵 BAJA | BAJA | 3 | No | 1 |
| 10 | Contenido Vacío | 🔵 BAJA | - | - | No | 0 |

**Total Estimado:** 15-18 días (3 semanas)

---

## Archivos Críticos a Modificar

### 1. **`brokers/tastyworks/tastyworks.py`** (PRINCIPAL)
**Líneas Actuales:** 428
**Líneas Estimadas Post-Cambios:** ~600-650
**Cambios:**
- Método `normalize()`: Agregar validaciones Fase 1 y 2
- Línea 280: Filtrado assignment/exercise
- Línea 326: Validación símbolos
- Línea 333: Inferencia acción
- Línea 355: Normalización precio
- Línea 389: Multiplicador future options

**Responsabilidad:** Todas las validaciones de Fase 1 y 2

---

### 2. **`brokers/tastyworks/lookups.py`** (NUEVO)
**Líneas Estimadas:** 50-100
**Contenido:**
```python
"""Tastyworks-specific lookup tables and constants"""

# Symbol replacements (legacy compatibility)
SYMBOL_REPLACEMENTS = {
    "RUTW": "RUT",
}

# Future option contract sizes
FUTURE_OPTION_SIZES = {
    "/ES": 50,   # E-mini S&P 500
    "/NQ": 20,   # E-mini NASDAQ-100
    "/YM": 5,    # E-mini Dow
    "/RTY": 50,  # E-mini Russell 2000
    "/CL": 1000, # Crude Oil
    "/GC": 100,  # Gold
}

# User-specific edge cases (legacy issue)
DUPLICATE_OFFSET_USERS = {47132, 242106, 49127, 137705}
```

**Responsabilidad:** Constantes y lookup tables

---

### 3. **`tests/brokers/test_tastyworks.py`** (MODIFICAR)
**Líneas Actuales:** 577
**Líneas Estimadas Post-Cambios:** ~800-900
**Cambios:**
- Agregar ~15 funciones de test para validaciones nuevas
- Test coverage para Fase 1, 2 y 3
- Tests de integración end-to-end

**Responsabilidad:** Cobertura de tests

---

### 4. **`p04_calculate/calculation.py`** (INVESTIGAR Y MODIFICAR - FASE 3)
**Líneas Actuales:** Desconocido
**Cambios:**
- Implementar lógica de actualización de trades para cash settlements
- Ported de `ImportParams.Tastyworks_cash_settlement()`
- Integración con flag `_is_cash_settlement` de p01

**Responsabilidad:** Actualizaciones de trades por cash settlement

---

### 5. **`handler.py`** (MODIFICAR CONDICIONAL - FASE 3)
**Líneas Actuales:** 102,887
**Cambios (SI SE REQUIERE):**
- Pre-validación de account_id antes de normalización
- Verificación usuario-cuenta antes de procesar
- Solo si se soportan CSV uploads (DECISIÓN PENDIENTE)

**Responsabilidad:** Validación de account ownership (condicional)

---

## Decisiones Pendientes y Preguntas para el Usuario

### 🔴 DECISIÓN CRÍTICA 1: Soporte CSV Upload
**Pregunta:** ¿El nuevo sistema soporta uploads CSV de Tastyworks, o solo sincronización vía JSON API?

**Impacto:**
- Si SÍ soporta CSV: Implementar validación #7 (Account Verification) en handler.py
- Si NO (solo JSON API): OMITIR validación #7

**Dependencias:** Validación #7 (Account Verification)

---

### 🟡 DECISIÓN MEDIA 2: Usuarios Especiales
**Pregunta:** ¿Los usuarios 47132, 242106, 49127, 137705 siguen activos en producción?

**Impacto:**
- Si SÍ activos: Implementar validación #9 (timestamp adjustment)
- Si NO: OMITIR validación #9

**Acción Sugerida:** Query DB production para verificar status
```sql
SELECT id, email, active FROM users WHERE id IN (47132, 242106, 49127, 137705);
```

---

### 🟡 DECISIÓN MEDIA 3: Future Options Data
**Pregunta:** ¿El volumen de Future Options en Tastyworks justifica implementar lookup table completa?

**Impacto:**
- Si SÍ hay volumen: Completar lookup table con todos los símbolos
- Si NO: Implementación simple con multiplicador default=100

**Acción Sugerida:** Analizar datos históricos
```sql
SELECT COUNT(*) FROM trades
WHERE broker_id = 26 AND asset_type = 'futures_options'
GROUP BY symbol;
```

---

## Riesgos y Mitigación

### Riesgo 1: Compatibilidad Hash MD5 vs SHA256
**Severidad:** ALTA
**Probabilidad:** MEDIA

**Descripción:** Legacy usa MD5 para file_row, nuevo usa SHA256 en p02_deduplicate

**Mitigación:**
- Validar que p02 acepta `original_file_row` (string completo), no hash directo
- p02 genera SHA256 del `original_file_row`, independiente del hash MD5 en p01
- ✅ VERIFICADO: p02 recibe el string completo y calcula SHA256 internamente

---

### Riesgo 2: Missing Cash Settlement Module
**Severidad:** MEDIA
**Probabilidad:** ALTA

**Descripción:** Módulo `ImportParams.Tastyworks_cash_settlement()` no está visible en codebase nuevo

**Mitigación:**
- Fase 1 y 2 son independientes (no bloqueantes)
- Fase 3 requiere investigación profunda del legacy
- Alternativa: Implementar versión simplificada si módulo no es portable

---

### Riesgo 3: Performance Degradation
**Severidad:** BAJA
**Probabilidad:** BAJA

**Descripción:** Validaciones adicionales pueden afectar performance

**Mitigación:**
- Usar operaciones vectorizadas Polars (ya en uso)
- Evitar loops Python (usar .filter(), .when(), .join())
- Profiling antes/después de cambios

---

## Estrategia de Verificación End-to-End

### Paso 1: Test Unitario (Por Validación)
```bash
pytest tests/brokers/test_tastyworks.py::test_symbol_validation_rejects_numeric -v
pytest tests/brokers/test_tastyworks.py::test_action_inference_expiration -v
# ... etc
```

### Paso 2: Test de Integración
```bash
pytest tests/brokers/test_tastyworks.py::test_full_normalize_pipeline_with_validations -v
```

### Paso 3: Validación contra Legacy (QA Script)
```bash
# Ejecutar script de validación contra usuario legacy
python qa/validate_broker.py \
    --broker-id 26 \
    --user-id 99999 \
    --legacy-user-id <legacy_user> \
    --compare-fields symbol,side,quantity,price,timestamp,option_type \
    --match-threshold 0.95
```

**Output Esperado:**
```
Broker: Tastyworks (26)
Match Rate Summary:
  - symbol: 100.0%
  - side: 98.5%  (+3.2% vs baseline)
  - quantity: 100.0%
  - price: 100.0%
  - timestamp: 93.0% (legacy EXP_A suffix issue)
  - option_type: 99.8%  (+5.1% vs baseline)

Overall Match: 97.5% ✅
Critical Fields: 100.0% ✅
```

### Paso 4: Smoke Test en Staging
```bash
# Procesar archivo real de usuario Tastyworks
curl -X POST https://staging-api.tradersync.com/pipeline/process \
  -H "Content-Type: application/json" \
  -d '{
    "broker_id": 26,
    "user_id": 99999,
    "file_path": "s3://staging-bucket/tastyworks_user_test.json"
  }'

# Verificar output en DB staging
```

---

## Métricas de Éxito del Proyecto

### Criterios de Aceptación

#### Fase 1 (Semana 1)
- ✅ Todas las validaciones Fase 1 implementadas
- ✅ Tests unitarios passing (coverage ≥ 90%)
- ✅ Test de integración passing
- ✅ Match rate ≥ baseline (sin regresión)
- ✅ No símbolos numéricos en output
- ✅ Eventos de opciones correctamente clasificados

#### Fase 2 (Semana 2)
- ✅ Validaciones Fase 2 implementadas
- ✅ Precio precision consistente (6 decimales)
- ✅ Cash settlements flagged
- ✅ Future options sized correctly (si aplicable)
- ✅ Match rate + 2-5% vs baseline

#### Fase 3 (Semana 3)
- ✅ Investigación completada (cash settlement module)
- ✅ Updates implementados (si factible)
- ✅ Account validation (si requerido)
- ✅ Match rate ≥ 95%
- ✅ Documentación actualizada

### Overall Success
- ✅ Sin pérdida de datos vs legacy
- ✅ Sin regresión de performance
- ✅ Código mantenible (separación de responsabilidades)
- ✅ Tests comprehensivos
- ✅ Documentación completa

---

## Resumen para Guardar en new_changes_tastywork/

Al finalizar, se deberá crear el directorio `new_changes_tastywork/` con:

### 1. **README.md**
- Resumen ejecutivo de cambios
- Validaciones agregadas (lista de 10)
- Impacto de cada validación
- Antes/después métricas

### 2. **ANALISIS_VALIDACIONES_TASTYWORKS.md**
- Análisis detallado legacy vs nuevo
- Cada validación con ubicación en código
- Ejemplos de código legacy y nuevo
- Decisiones de diseño y arquitectura

### 3. **CAMBIOS_IMPLEMENTADOS.md**
- Log de cambios por archivo
- Diff summary de archivos modificados
- Tests agregados
- Verificación de match rates

### 4. **brokers/tastyworks/** (código nuevo)
- `tastyworks.py` (modificado)
- `lookups.py` (nuevo)
- `tests/test_tastyworks.py` (modificado)

---

## Conclusión

Este plan identifica **10 validaciones** del código legacy, de las cuales:
- **3 son CRÍTICAS (Fase 1)** - Implementación inmediata
- **3 son MEDIA prioridad (Fase 2)** - Implementación semana 2
- **2 son CONDICIONALES (Fase 3)** - Requieren decisiones/investigación
- **2 son YA MANEJADAS/BAJAS** - No requieren acción

El enfoque de 3 fases permite:
1. ✅ **Quick wins** en Fase 1 (sin dependencias, alto impacto)
2. ⚠️ **Mejoras incrementales** en Fase 2 (investigación menor)
3. 🔍 **Investigación profunda** en Fase 3 (cross-módulo, decisiones)

**RECOMENDACIÓN FINAL:** Proceder con **Fase 1 inmediatamente** - todas las validaciones son autocontenidas, bajo riesgo, y previenen datos inválidos en el pipeline.
