# Ejemplos de Cambios de Código - Tastyworks

Este documento muestra ejemplos concretos de los cambios propuestos con código real antes/después.

---

## FASE 1: Validaciones Críticas

### Cambio 1: Validación de Símbolos Numéricos + Reemplazo RUTW→RUT

**Archivo:** `brokers/tastyworks/tastyworks.py`
**Línea:** ~326
**Criticidad:** 🔴 ALTA

#### Código Actual (ANTES)
```python
# Línea 326-330
.with_columns([
    pl.col("symbol")
    .str.to_uppercase()
    .str.strip_chars()
    .alias("symbol")
])
```

#### Código Propuesto (DESPUÉS)
```python
# Filtrar símbolos numéricos (validación crítica)
.filter(
    ~pl.col("symbol").cast(pl.Float64, strict=False).is_not_null()
)
# Reemplazar símbolos legacy + normalizar
.with_columns([
    pl.col("symbol")
    .str.replace("RUTW", "RUT")  # Legacy symbol mapping
    .str.to_uppercase()
    .str.strip_chars()
    .alias("symbol")
])
```

#### Justificación
- **Legacy:** `brokers_tastyworks.py:163, 167, 508`
- **Impacto:** Previene trades con símbolos inválidos (ej: "12345", "67.89")
- **Casos detectados:** Símbolos completamente numéricos o primeros 4 chars numéricos

#### Tests Requeridos
```python
def test_symbol_validation_rejects_numeric():
    """Verifica que símbolos numéricos son rechazados"""
    df = pl.DataFrame({
        "symbol": ["AAPL", "12345", "TSLA", "67.89"],
        "description": ["Buy", "Buy", "Buy", "Buy"],
        # ... otros campos requeridos
    })
    result = TastyworksInterpreter().normalize(df.lazy(), account_id=1).collect()
    assert len(result) == 2  # Solo AAPL y TSLA
    assert "12345" not in result["symbol"].to_list()
    assert "67.89" not in result["symbol"].to_list()

def test_symbol_rutw_replacement():
    """Verifica reemplazo RUTW → RUT"""
    df = pl.DataFrame({
        "symbol": ["RUTW", "RUT", "AAPL"],
        "description": ["Buy", "Buy", "Buy"],
        # ... otros campos
    })
    result = TastyworksInterpreter().normalize(df.lazy(), account_id=1).collect()
    assert result.filter(pl.col("symbol") == "RUTW").height == 0
    assert result.filter(pl.col("symbol") == "RUT").height == 2
```

---

### Cambio 2: Inferencia de Acción desde Descripción

**Archivo:** `brokers/tastyworks/tastyworks.py`
**Línea:** ~333
**Criticidad:** 🔴 ALTA

#### Código Actual (ANTES)
```python
# Línea 333-335
.with_columns([
    pl.col("action")
    .replace_strict(self.ACTION_MAP, default="BUY")
    .alias("side")
])
```

#### Código Propuesto (DESPUÉS)
```python
.with_columns([
    # Inferir acción desde descripción para eventos especiales de opciones
    pl.when(pl.col("description").str.contains("(?i)expiration"))
    .then(pl.lit("EXP"))
    .when(pl.col("description").str.contains("(?i)assignment"))
    .then(pl.lit("BUY"))
    .when(pl.col("description").str.contains("(?i)exercise"))
    .then(pl.lit("SELL"))
    # Fallback: usar ACTION_MAP estándar
    .otherwise(
        pl.col("action").replace_strict(self.ACTION_MAP, default="BUY")
    )
    .alias("side")
])
```

#### Justificación
- **Legacy:** `brokers_tastyworks.py:144-159`
- **Impacto:** P&L incorrecto para eventos de opciones si no se detectan correctamente
- **Eventos detectados:**
  - EXPIRATION → EXP
  - ASSIGNMENT → BUY (el holder recibe shares)
  - EXERCISE → SELL (el holder vende shares)

#### Ejemplo de Datos
```json
// Caso 1: Expiration
{
  "description": "Sold 1 SPY Put for expiration",
  "action": "SELL_TO_CLOSE",
  "side": "EXP"  // <- Inferido desde descripción
}

// Caso 2: Assignment
{
  "description": "Assignment of 100 AAPL",
  "action": "",
  "side": "BUY"  // <- Inferido desde descripción
}

// Caso 3: Exercise
{
  "description": "Exercise of 1 /ES Call",
  "action": "",
  "side": "SELL"  // <- Inferido desde descripción
}
```

#### Tests Requeridos
```python
def test_action_inference_expiration():
    """Verifica inferencia EXP desde descripción"""
    df = pl.DataFrame({
        "description": ["Sold 1 SPY Put for expiration"],
        "action": ["SELL_TO_CLOSE"],
    })
    result = normalize(df)
    assert result["side"][0] == "EXP"

def test_action_inference_assignment():
    """Verifica inferencia BUY desde ASSIGNMENT"""
    df = pl.DataFrame({
        "description": ["Assignment of 100 AAPL"],
        "action": [""],
    })
    result = normalize(df)
    assert result["side"][0] == "BUY"

def test_action_inference_exercise():
    """Verifica inferencia SELL desde EXERCISE"""
    df = pl.DataFrame({
        "description": ["Exercise of 1 /ES Call"],
        "action": [""],
    })
    result = normalize(df)
    assert result["side"][0] == "SELL"
```

---

### Cambio 3: Filtrado Assignment/Exercise con Cash Settlement

**Archivo:** `brokers/tastyworks/tastyworks.py`
**Línea:** ~280 (después del filtro transaction-type)
**Criticidad:** 🔴 ALTA

#### Código Actual (ANTES)
```python
# Línea 278-282
records = df.filter(
    pl.col("transaction-type").is_in(["Trade", "Receive Deliver"])
)
# Continúa procesamiento sin filtrado adicional...
```

#### Código Propuesto (DESPUÉS)
```python
# Filtrar por transaction-type
records = df.filter(
    pl.col("transaction-type").is_in(["Trade", "Receive Deliver"])
)

# NUEVO: Filtrar assignments/exercises si existe cash settlement
# Paso 1: Identificar símbolos que tienen cash settlement
cash_settlement_symbols_df = (
    records
    .filter(pl.col("description").str.contains("(?i)cash settlement"))
    .select("symbol")
    .unique()
)

# Paso 2: Si hay cash settlements, filtrar assignments/exercises correspondientes
if len(cash_settlement_symbols_df) > 0:
    # Materializar lista de símbolos con cash settlement
    cash_settlement_symbols = cash_settlement_symbols_df.collect()["symbol"].to_list()

    # Filtrar: remover assignment/exercise SI existe cash settlement para mismo símbolo
    records = records.filter(
        ~(
            # Detectar assignment o exercise en descripción
            pl.col("description").str.contains("(?i)assignment|exercise") &
            # Y el símbolo tiene cash settlement
            pl.col("symbol").is_in(cash_settlement_symbols)
        )
    )
```

#### Justificación
- **Legacy:** `brokers_tastyworks.py:225-228`
- **Impacto:** Previene duplicados (assignment + cash settlement para mismo evento)
- **Regla de negocio:** Si existe cash settlement, es la representación preferida del evento

#### Ejemplo de Escenario
```
Entrada (2 registros):
1. symbol: SPY, description: "Cash settlement of SPY", value: $450
2. symbol: SPY, description: "Assignment of SPY", value: $450

Salida (1 registro):
1. symbol: SPY, description: "Cash settlement of SPY", value: $450
   ↑ Se mantiene

Registro 2 FILTRADO porque:
- Es assignment/exercise
- SPY tiene cash settlement
- Cash settlement es preferido
```

#### Tests Requeridos
```python
def test_assignment_filtered_when_cash_settlement_exists():
    """Verifica filtrado de assignment si hay cash settlement"""
    df = pl.DataFrame({
        "symbol": ["SPY", "SPY"],
        "description": [
            "Cash settlement of SPY",
            "Assignment of SPY"
        ],
        "transaction-type": ["Trade", "Trade"],
        # ... otros campos
    })
    result = normalize(df)
    # Solo debe quedar 1 registro (cash settlement)
    assert len(result) == 1
    assert "Cash settlement" in result["description"][0]
    assert "Assignment" not in result["description"].to_list()

def test_exercise_filtered_when_cash_settlement_exists():
    """Verifica filtrado de exercise si hay cash settlement"""
    df = pl.DataFrame({
        "symbol": ["/ES", "/ES"],
        "description": [
            "Cash settlement of /ES",
            "Exercise of 1 /ES Call"
        ],
        "transaction-type": ["Trade", "Trade"],
    })
    result = normalize(df)
    assert len(result) == 1
    assert "Cash settlement" in result["description"][0]

def test_assignment_kept_when_no_cash_settlement():
    """Verifica que assignment se mantiene si NO hay cash settlement"""
    df = pl.DataFrame({
        "symbol": ["AAPL"],
        "description": ["Assignment of 100 AAPL"],
        "transaction-type": ["Trade"],
    })
    result = normalize(df)
    assert len(result) == 1
    assert "Assignment" in result["description"][0]
```

---

## FASE 2: Validaciones de Calidad

### Cambio 4: Normalización de Precio con Precisión

**Archivo:** `brokers/tastyworks/tastyworks.py`
**Línea:** ~355
**Criticidad:** 🟡 MEDIA

#### Código Actual (ANTES)
```python
# Línea 353-356
.with_columns([
    pl.col("price").cast(pl.Float64).alias("price"),
    pl.col("commission").abs().alias("commission"),
])
```

#### Código Propuesto (DESPUÉS)
```python
.with_columns([
    # Normalizar precio: redondear a 6 decimales, default 0.0 si inválido
    pl.when(pl.col("price").is_not_null() & (pl.col("price") > 0))
    .then(pl.col("price").cast(pl.Float64).round(6))
    .otherwise(pl.lit(0.0))
    .alias("price"),

    pl.col("commission").abs().alias("commission"),
])
```

#### Justificación
- **Legacy:** `brokers_tastyworks.py:255-260`
- **Impacto:** Consistencia en precisión de precios (6 decimales)
- **Manejo de edge cases:** Precios null o <= 0 → 0.0

---

### Cambio 5: Detección de Cash Settlement

**Archivo:** `brokers/tastyworks/tastyworks.py`
**Línea:** ~270 (en la construcción del output)
**Criticidad:** 🟡 MEDIA-ALTA

#### Código Propuesto (AÑADIR)
```python
# En el return final del método normalize(), añadir columna
.with_columns([
    # Flag para identificar cash settlements (usado en p04_calculate)
    pl.when(pl.col("description").str.contains("(?i)cash settlement"))
    .then(pl.lit(True))
    .otherwise(pl.lit(False))
    .alias("_is_cash_settlement")
])
```

#### Justificación
- **Legacy:** `brokers_tastyworks.py:229-235`
- **Uso:** p04_calculate stage puede usar este flag para actualizar trades relacionados
- **Nota:** Fase 1 solo detecta, Fase 3 implementa lógica de actualización

---

### Cambio 6: Cálculo de Tamaño para Future Options

**Archivo:** `brokers/tastyworks/tastyworks.py`
**Línea:** ~389
**Criticidad:** 🟡 MEDIA
**Requiere:** Crear `lookups.py` primero

#### Código Actual (ANTES)
```python
# Línea 389-392
.with_columns([
    pl.when(pl.col("instrument-type") == "Equity Option")
    .then(pl.lit(100.0))
    .otherwise(pl.lit(1.0))
    .alias("multiplier")
])
```

#### Código Propuesto (DESPUÉS)
```python
# Importar al inicio del archivo
from .lookups import FUTURE_OPTION_SIZES

# Línea 389+
.with_columns([
    pl.when(pl.col("instrument-type") == "Future Option")
    .then(
        # Opción 1: Calcular desde value si disponible
        pl.when(pl.col("value").is_not_null())
        .then(
            (pl.col("value").abs() / (pl.col("price") * pl.col("quantity")))
            .round(2)
        )
        # Opción 2: Lookup por símbolo
        .otherwise(
            pl.col("underlying-symbol")
            .replace_strict(FUTURE_OPTION_SIZES, default=100.0)
        )
    )
    .when(pl.col("instrument-type") == "Equity Option")
    .then(pl.lit(100.0))
    .otherwise(pl.lit(1.0))
    .alias("multiplier")
])
```

#### Justificación
- **Legacy:** `brokers_tastyworks.py:295-305`
- **Impacto:** Posiciones incorrectas para future options (/ES, /NQ, etc.)
- **Fórmula:** size = amount / (price × quantity)
- **Lookup:** Si no hay value, usar tabla por símbolo

#### Ejemplo de Cálculo
```python
# Trade de /ES Future Option
# /ES multiplier = 50

# Opción 1: Cálculo desde value
value = 1250.00
price = 5.00
quantity = 1
multiplier = 1250.00 / (5.00 * 1) = 250.00 / 5.00 = 50.0  ✅

# Opción 2: Lookup
underlying_symbol = "/ES"
multiplier = FUTURE_OPTION_SIZES["/ES"] = 50.0  ✅
```

---

## Resumen de Cambios por Archivo

### brokers/tastyworks/tastyworks.py
- **Línea ~280:** Filtrado assignment/exercise (Cambio 3)
- **Línea ~270:** Detección cash settlement (Cambio 5)
- **Línea ~326:** Validación símbolos (Cambio 1)
- **Línea ~333:** Inferencia acción (Cambio 2)
- **Línea ~355:** Normalización precio (Cambio 4)
- **Línea ~389:** Future option size (Cambio 6)

### brokers/tastyworks/lookups.py (NUEVO)
- Crear archivo completo con lookup tables

### tests/brokers/test_tastyworks.py
- Añadir ~15 funciones de test nuevas
- Coverage de todos los cambios

---

## Orden de Implementación Recomendado

1. ✅ **Crear lookups.py** (Fase 2, pero sin dependencias)
2. ✅ **Cambio 1:** Validación símbolos + tests
3. ✅ **Cambio 2:** Inferencia acción + tests
4. ✅ **Cambio 3:** Filtrado assignment/exercise + tests
5. ✅ **Validar Fase 1:** Ejecutar todos los tests, verificar match rate
6. ✅ **Cambio 4:** Normalización precio + tests
7. ✅ **Cambio 5:** Detección cash settlement + tests
8. ✅ **Cambio 6:** Future option size + tests
9. ✅ **Validar Fase 2:** Match rate final

---

**Nota:** Todos los cambios mantienen compatibilidad con la arquitectura existente y usan operaciones vectorizadas de Polars para performance óptimo.
