# Oanda Normalizer - Implementation Guide

## Executive Summary

**Status:** CRÍTICO ⚠️⚠️⚠️ - Hash match rate actual: ~0%

**Critical Issue:** Buy/sell field missing in hash computation causes 100% trade duplication in re-imports

**Implementation Phases:**
- **Fase 1 (CRÍTICO):** Hash fix - 2-3 días
- **Fase 2:** Data validations - 1-1.5 días
- **Fase 3 (CONDICIONAL):** CSV support - 8-12 días

**Total Critical Path:** 3-4.5 días (Fases 1-2 only)

---

## Tabla de Contenidos

1. [Contexto](#contexto)
2. [Arquitectura](#arquitectura)
3. [Issue Crítico: Hash Computation](#issue-crítico-hash-computation)
4. [Fase 1: Critical Hash Fix (2-3 días)](#fase-1-critical-hash-fix-2-3-días)
5. [Fase 2: Data Validation (1-1.5 días)](#fase-2-data-validation-1-15-días)
6. [Fase 3: CSV Support (8-12 días - CONDICIONAL)](#fase-3-csv-support-8-12-días---condicional)
7. [Testing Strategy](#testing-strategy)
8. [Deployment Checklist](#deployment-checklist)

---

## Contexto

### Broker Information

- **Broker:** OANDA
- **Assets:** Forex, CFDs
- **Format:** JSON API (sync)
- **API Versions:** v1 (oanda_export.py), v2 (oandav2_export.py)
- **CSV Formats:** 3 variants (636 líneas legacy)

### Code Metrics

| Métrica | Legacy | Nueva | Reducción |
|---------|--------|-------|-----------|
| **API normalization** | 640 líneas | 308 líneas | **52%** ✅ |
| **CSV parsers** | 636 líneas | 0 líneas | **N/A** ⚠️ |
| **Tests** | 0 líneas | 430 líneas | **N/A** ✅ |
| **Total IN SCOPE** | 1,276 líneas | 308 líneas | **76%** ✅ |

### Hash Match Rate Comparison

| Broker | Hash Match Rate | Critical Issues |
|--------|-----------------|-----------------|
| **Oanda** | **~0%** ⚠️⚠️⚠️ | Buy/sell field missing |
| Deribit | 74.27% ⚠️ | Expired options suffix |
| Charles Schwab | 42% ⚠️ | ClosingPrice volatility |
| OKX | 95-100% ✅ | None |
| KuCoin | 100% ✅ | None |

**Conclusión:** Oanda tiene el PEOR hash match rate de todos los brokers analizados.

---

## Arquitectura

### Pipeline Flow

```
Raw Data (JSON API)
        ↓
┌─────────────────────────┐
│ OandaInterpreter        │
│ - parse_json_content()  │ ← Split complete positions into executions
│ - normalize()           │
└─────────────────────────┘
        ↓
Normalized Executions
(19 columns schema)
        ↓
p02_deduplicate (file_row hash)
        ↓
p03_group (grouping logic)
        ↓
p04_calculate (P&L, metrics)
        ↓
p05_write (database)
```

### Key Architectural Patterns

1. **Trade Splitting**
   - OANDA API returns complete positions (entry + exit)
   - Normalizer splits CLOSED trades → 2 executions (OPEN + CLOSE)
   - OPEN trades → 1 execution only
   - Both executions share same `file_row` hash (order-level deduplication)

2. **Execution ID Pattern**
   ```python
   entry: "{order_id}_open"   # e.g., "82041_open"
   exit:  "{order_id}_close"  # e.g., "82041_close"
   ```

3. **Side Derivation**
   - Legacy: Complex 4-branch logic based on realizedPL and price comparison
   - Current: Simple initialUnits sign (causes 0% hash match)
   - **Fix Required:** Implement legacy 4-branch logic

4. **Hash Computation**
   - Legacy: MD5(json.dumps(order WITH 'buy/sell' field))
   - Current: MD5(json.dumps(order WITHOUT 'buy/sell' field)) ← **PROBLEMA**
   - **Fix Required:** Add buy/sell field BEFORE hashing

### File Structure

```
brokers/oanda/
├── __init__.py          (14 líneas)  - Module exports
├── detector.py          (50 líneas)  - Format detection
└── oanda.py            (308 líneas)  - Main interpreter
    ├── BROKER_ID = "oanda"
    ├── can_handle()              (class method)
    ├── parse_json_content()      (class method) ← Trade splitting + hashing
    ├── normalize()               (instance method)
    └── _parse_oanda_timestamp()  (helper)

tests/brokers/
└── test_oanda.py       (430 líneas)  - 31 test cases
```

---

## Issue Crítico: Hash Computation

### Root Cause Analysis

**Problema:** 0% hash match rate → 100% trade duplication

**Causa:** Legacy añade campo 'buy/sell' al order object ANTES de calcular hash MD5. Nueva implementación NO añade este campo.

### Evidence Comparison

#### Legacy (oanda_export.py:366-373, 444-445)

```python
# Step 1: Derive buy/sell using 4-branch logic
if float(trade['realizedPL'])>0 and float(trade['price']) > float(trade['averageClosePrice']):
    trade['buy/sell']='SELL'
elif float(trade['realizedPL'])>0 and float(trade['price']) < float(trade['averageClosePrice']):
    trade['buy/sell']='BUY'
elif float(trade['realizedPL'])<0 and float(trade['price']) > float(trade['averageClosePrice']):
    trade['buy/sell']='BUY'
elif float(trade['realizedPL'])<0 and float(trade['price']) < float(trade['averageClosePrice']):
    trade['buy/sell']='SELL'

# Step 2: Hash computation DESPUÉS de añadir 'buy/sell'
njson = json.dumps(order)  # order ahora incluye 'buy/sell'
njson = hashlib.md5(njson.encode('utf-8')).hexdigest()
```

#### Current (oanda.py:123-124)

```python
# Hash computation SIN 'buy/sell' field
order_json = json.dumps(order)  # order NO tiene 'buy/sell' añadido
base_hash = hashlib.md5(order_json.encode('utf-8')).hexdigest()
```

### Impact Analysis

| Aspecto | Impacto |
|---------|---------|
| **Hash match rate** | ~0% (worst of all brokers) |
| **Re-import behavior** | 100% trades duplicated |
| **Data integrity** | ✅ 100% correct (only hash differs) |
| **User experience** | ⚠️⚠️⚠️ CRÍTICO - duplicate trades everywhere |
| **Database bloat** | ⚠️⚠️⚠️ CRÍTICO - 2x storage usage |

### Why 4-Branch Logic?

**Pregunta:** ¿Por qué usar complex 4-branch logic en vez de simple initialUnits sign?

**Respuesta:** Legacy compatibility para hash matching.

**4-Branch Logic:**
```
Profit > 0, Entry > Exit → SELL (entry high, exit low = sell profit)
Profit > 0, Entry < Exit → BUY  (entry low, exit high = buy profit)
Profit < 0, Entry > Exit → BUY  (entry high, exit low = buy loss)
Profit < 0, Entry < Exit → SELL (entry low, exit high = sell loss)
```

**Example:**
```python
# Example 1: Profit from SELL
order = {
    'realizedPL': 100.0,    # Profit
    'price': 1.40,          # Entry high
    'averageClosePrice': 1.38  # Exit low
}
# → SELL (sold high, bought low)

# Example 2: Profit from BUY
order = {
    'realizedPL': 100.0,    # Profit
    'price': 1.38,          # Entry low
    'averageClosePrice': 1.40  # Exit high
}
# → BUY (bought low, sold high)
```

---

## Fase 1: Critical Hash Fix (2-3 días)

### Objetivos

- [ ] Implementar `_derive_buy_sell_legacy()` con 4-branch logic
- [ ] Modificar hash computation para incluir buy/sell field
- [ ] Achieve hash match rate >= 95%
- [ ] Pass all 10 buy/sell derivation tests

### Complejidad

- **Nivel:** ALTA
- **Riesgo:** ALTO (4-branch logic compleja, critical para deduplication)
- **Effort:** 2-3 días
- **Tests:** 10 nuevos tests

### Implementación

#### Paso 1.1: Add _derive_buy_sell_legacy() Method (0.5 días)

**File:** `oanda.py` (~line 74)

```python
@classmethod
def _derive_buy_sell_legacy(cls, order: dict) -> str:
    """
    Derive buy/sell using legacy 4-branch logic.

    Legacy logic (oanda_export.py:366-373):
    - If profit > 0 and entry > exit → SELL
    - If profit > 0 and entry < exit → BUY
    - If profit < 0 and entry > exit → BUY
    - If profit < 0 and entry < exit → SELL

    Args:
        order: Raw OANDA order with realizedPL, price, averageClosePrice

    Returns:
        "BUY" or "SELL"
    """
    try:
        realized_pl = float(order.get('realizedPL', 0))
        entry_price = float(order.get('price', 0))
        exit_price = float(order.get('averageClosePrice', 0))

        if realized_pl > 0 and entry_price > exit_price:
            return 'SELL'
        elif realized_pl > 0 and entry_price < exit_price:
            return 'BUY'
        elif realized_pl < 0 and entry_price > exit_price:
            return 'BUY'
        elif realized_pl < 0 and entry_price < exit_price:
            return 'SELL'
        else:
            # Fallback: use initialUnits sign
            initial_units = float(order.get('initialUnits', 0))
            return 'BUY' if initial_units > 0 else 'SELL'
    except (ValueError, TypeError):
        # Fallback: use initialUnits sign
        initial_units = float(order.get('initialUnits', 0))
        return 'BUY' if initial_units > 0 else 'SELL'
```

**Checklist:**
- [ ] Method signature correct
- [ ] 4-branch logic implemented
- [ ] Fallback logic implemented
- [ ] Exception handling added
- [ ] Docstring complete with examples

#### Paso 1.2: Modify Hash Computation (0.5 días)

**File:** `oanda.py` (~line 120-145)

**Current Code:**
```python
# Lines 123-124
order_json = json.dumps(order)
base_hash = hashlib.md5(order_json.encode('utf-8')).hexdigest()
```

**Modified Code:**
```python
# Create mutable copy for hash computation
order_for_hash = dict(order)

# CRITICAL: Add buy/sell field using legacy logic
buy_sell = cls._derive_buy_sell_legacy(order)
order_for_hash['buy/sell'] = buy_sell

# Compute hash with buy/sell included
order_json = json.dumps(order_for_hash)
base_hash = hashlib.md5(order_json.encode('utf-8')).hexdigest()
```

**Checklist:**
- [ ] Create mutable copy of order
- [ ] Call _derive_buy_sell_legacy()
- [ ] Add buy/sell field to copy
- [ ] Hash includes buy/sell field
- [ ] Original order unchanged

#### Paso 1.3: Add Tests (1 día)

**File:** `tests/brokers/test_oanda.py`

**Test Class:**
```python
class TestOandaBuySelDerivation:
    """Test buy/sell derivation logic for hash compatibility"""

    def test_derive_buy_sell_profit_sell(self):
        """Profit > 0, entry > exit → SELL"""
        # Test implementation

    def test_derive_buy_sell_profit_buy(self):
        """Profit > 0, entry < exit → BUY"""
        # Test implementation

    def test_derive_buy_sell_loss_buy(self):
        """Loss < 0, entry > exit → BUY"""
        # Test implementation

    def test_derive_buy_sell_loss_sell(self):
        """Loss < 0, entry < exit → SELL"""
        # Test implementation

    def test_derive_buy_sell_fallback_buy(self):
        """Fallback to initialUnits (positive)"""
        # Test implementation

    def test_derive_buy_sell_fallback_sell(self):
        """Fallback to initialUnits (negative)"""
        # Test implementation

    def test_derive_buy_sell_invalid_data(self):
        """Handle invalid/missing data gracefully"""
        # Test implementation

    def test_hash_includes_buy_sell_field(self):
        """Verifica que hash incluye buy/sell field"""
        # Test implementation

    def test_hash_differs_with_without_buy_sell(self):
        """Hashes deben ser diferentes con/sin buy/sell"""
        # Test implementation

    def test_hash_match_rate_integration(self):
        """Integration test: hash match rate >= 95%"""
        # Test implementation
```

**Checklist:**
- [ ] 10 tests implemented
- [ ] All 4 branches tested
- [ ] Fallback logic tested
- [ ] Edge cases covered
- [ ] Integration test passes

#### Paso 1.4: Validation (0.5 días)

**Manual Testing:**
```python
# test_hash_manual.py
import json
import hashlib

order = {
    'id': '82041',
    'realizedPL': 100.0,
    'price': 1.40,
    'averageClosePrice': 1.38,
}

# Legacy approach
order_legacy = dict(order)
order_legacy['buy/sell'] = 'SELL'
hash_legacy = hashlib.md5(json.dumps(order_legacy).encode()).hexdigest()

# New approach (after fix)
order_fixed = dict(order)
buy_sell = OandaInterpreter._derive_buy_sell_legacy(order)
order_fixed['buy/sell'] = buy_sell
hash_fixed = hashlib.md5(json.dumps(order_fixed).encode()).hexdigest()

print(f"Legacy: {hash_legacy}")
print(f"Fixed:  {hash_fixed}")
print(f"Match:  {hash_legacy == hash_fixed}")  # Should be True
```

**Checklist:**
- [ ] Manual test script created
- [ ] Hash match verified with sample data
- [ ] All unit tests pass
- [ ] Integration test passes
- [ ] Hash match rate >= 95%

### Success Metrics

- [ ] Hash match rate >= 95% (vs actual ~0%)
- [ ] Buy/sell derivation correcta para las 4 branches
- [ ] All 10 tests passing
- [ ] No regressions in existing tests (31 tests)
- [ ] Data integrity maintained (100%)

### Rollback Plan

If hash fix causes issues:
1. Revert `_derive_buy_sell_legacy()` method
2. Revert hash computation changes
3. Re-deploy previous version
4. Investigate root cause with sample data

---

## Fase 2: Data Validation (1-1.5 días)

### Objetivos

- [ ] Add 6 data validations
- [ ] Prevent invalid data from entering pipeline
- [ ] Achieve rejection rate < 0.1%
- [ ] Pass all 22 validation tests

### Complejidad

- **Nivel:** BAJA
- **Riesgo:** BAJO
- **Effort:** 1-1.5 días
- **Tests:** 22 nuevos tests

### Validaciones

| # | Validación | Priority | Tests | Effort |
|---|------------|----------|-------|--------|
| 1 | Symbol Empty | ⭐⭐⭐ ALTA | 3 | 0.25 días |
| 2 | Price Zero/Negative | ⭐⭐⭐ ALTA | 4 | 0.25 días |
| 3 | Realized P&L Zero | ⭐⭐ MEDIA-ALTA | 3 | 0.25 días |
| 4 | Status FILLED | ⭐⭐ MEDIA | 4 | 0.25 días |
| 5 | Quantity Zero | ⭐⭐ MEDIA | 4 | 0.25 días |
| 6 | Avg Close Price | ⭐⭐ MEDIA | 4 | 0.25 días |

### Implementación

#### Paso 2.1: Symbol Validation (0.25 días)

**File:** `oanda.py` (~line 100)

```python
instrument = order.get("instrument", "")
if not instrument or not instrument.strip():
    logger.warning(
        f"[OANDA] Skipping order {order.get('id', 'unknown')}: "
        f"empty instrument"
    )
    continue
```

**Tests:** 3 (empty, whitespace, missing)

#### Paso 2.2: Price Validation (0.25 días)

**File:** `oanda.py` (~line 114)

```python
entry_price = float(order.get("price", 0) or 0)
if entry_price <= 0:
    logger.warning(
        f"[OANDA] Skipping order {trade_id}: "
        f"zero or negative entry price ({entry_price})"
    )
    continue

if state == "CLOSED":
    exit_price = float(order.get("averageClosePrice", 0) or 0)
    if exit_price <= 0:
        logger.warning(
            f"[OANDA] Skipping order {trade_id}: "
            f"zero or negative exit price ({exit_price})"
        )
        continue
```

**Tests:** 4 (zero entry, negative entry, zero exit, missing)

#### Paso 2.3: Realized P&L Zero Skip (0.25 días)

**File:** `oanda.py` (~line 95)

```python
realized_pl = float(order.get("realizedPL", 0) or 0)
if realized_pl == 0:
    logger.info(
        f"[OANDA] Skipping order {order.get('id', 'unknown')}: "
        f"zero realized P&L"
    )
    continue
```

**Tests:** 3 (zero, positive accepted, negative accepted)

#### Paso 2.4: Status FILLED Validation (0.25 días)

**File:** `oanda.py` (~line 95)

```python
status = order.get("status", "")
if status and status != "FILLED":
    logger.info(
        f"[OANDA] Skipping order {order.get('id', 'unknown')}: "
        f"status {status} (not FILLED)"
    )
    continue
```

**Tests:** 4 (PENDING, CANCELLED, FILLED accepted, missing accepted)

#### Paso 2.5: Quantity Validation (0.25 días)

**File:** `oanda.py` (~line 111)

```python
initial_units = float(order.get("initialUnits", 0) or 0)
if initial_units == 0:
    logger.warning(
        f"[OANDA] Skipping order {trade_id}: "
        f"zero initial units"
    )
    continue
```

**Tests:** 4 (zero, missing, positive accepted, negative accepted)

#### Paso 2.6: Average Close Price Validation (0.25 días)

**File:** `oanda.py` (~line 147)

```python
if state == "CLOSED":
    avg_close_price = order.get("averageClosePrice")
    if not avg_close_price:
        logger.warning(
            f"[OANDA] Skipping closed order {trade_id}: "
            f"missing averageClosePrice"
        )
        continue

    exit_price = float(avg_close_price or 0)
    if exit_price <= 0:
        logger.warning(
            f"[OANDA] Skipping closed order {trade_id}: "
            f"invalid averageClosePrice ({exit_price})"
        )
        continue
```

**Tests:** 4 (missing, zero, valid accepted, open doesn't need)

### Testing Strategy

**Test Class:**
```python
class TestOandaDataValidation:
    """Test data validation logic"""

    # Symbol validation (3 tests)
    def test_empty_instrument_validation(self): ...
    def test_whitespace_instrument_validation(self): ...
    def test_missing_instrument_validation(self): ...

    # Price validation (4 tests)
    def test_zero_entry_price_validation(self): ...
    def test_negative_entry_price_validation(self): ...
    def test_zero_exit_price_validation(self): ...
    def test_missing_price_validation(self): ...

    # Realized P&L (3 tests)
    def test_zero_realized_pl_skip(self): ...
    def test_positive_realized_pl_accepted(self): ...
    def test_negative_realized_pl_accepted(self): ...

    # Status (4 tests)
    def test_status_pending_skip(self): ...
    def test_status_cancelled_skip(self): ...
    def test_status_filled_accepted(self): ...
    def test_missing_status_accepted(self): ...

    # Quantity (4 tests)
    def test_zero_quantity_validation(self): ...
    def test_missing_quantity_validation(self): ...
    def test_positive_quantity_accepted(self): ...
    def test_negative_quantity_accepted(self): ...

    # Average close price (4 tests)
    def test_missing_average_close_price_validation(self): ...
    def test_zero_average_close_price_validation(self): ...
    def test_valid_average_close_price_accepted(self): ...
    def test_open_trade_without_average_close_price(self): ...
```

### Success Metrics

- [ ] All 22 validation tests passing
- [ ] Rejection rate < 0.1%
- [ ] No false positives (valid data rejected)
- [ ] Meaningful warning messages in logs
- [ ] No regressions in existing tests

---

## Fase 3: CSV Support (8-12 días - CONDICIONAL)

### Decision Gate

**BLOQUEADOR:** Ejecutar SQL query ANTES de implementar

```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*)) OVER (), 2) as percentage,
    MAX(created_at) as last_used
FROM data_sources
WHERE broker_id = 'oanda'
  AND created_at > NOW() - INTERVAL '12 months'
GROUP BY source_type
ORDER BY count DESC;
```

**Criterio:**
- **CSV percentage > 5%:** IMPLEMENTAR (proceed with Fase 3)
- **CSV percentage < 5%:** OMITIR (skip Fase 3)

### CSV Format Analysis

Legacy implementation has **3 different CSV parser formats** (636 líneas):

#### Format 1: getcsv (Lines 1-253) - Consecutive Item Format

**Characteristics:**
- Items listed consecutively (open/close on separate rows)
- Column structure varies
- Requires auto-detection

**Effort:** 3 días

#### Format 2: getcsv2 (Lines 257-489) - Open/Close Same Row

**Characteristics:**
- Open and close on same row
- Different column mappings
- More complex parsing

**Effort:** 3 días

#### Format 3: getcsv3 (Lines 491-612) - Japanese Format

**Characteristics:**
- Japanese column names
- Different date format
- Specific encoding handling

**Effort:** 2 días

### Implementation Plan (IF REQUIRED)

#### Paso 3.1: CSV Parser 1 (3 días)

- [ ] Implement getcsv format parser
- [ ] Add column detection
- [ ] Handle consecutive items
- [ ] Add tests (10+)

#### Paso 3.2: CSV Parser 2 (3 días)

- [ ] Implement getcsv2 format parser
- [ ] Handle open/close same row
- [ ] Map columns correctly
- [ ] Add tests (10+)

#### Paso 3.3: CSV Parser 3 (2 días)

- [ ] Implement getcsv3 format parser
- [ ] Handle Japanese columns
- [ ] Handle encoding
- [ ] Add tests (8+)

#### Paso 3.4: Auto-Detection Logic (1 día)

- [ ] Implement format detection
- [ ] Priority order
- [ ] Fallback logic
- [ ] Add tests (5+)

#### Paso 3.5: Documentation & Testing (2 días)

- [ ] Update detector.py
- [ ] Integration tests
- [ ] Performance testing
- [ ] Documentation

### Success Metrics (IF IMPLEMENTED)

- [ ] All 3 CSV formats supported
- [ ] Auto-detection working correctly
- [ ] 33+ CSV tests passing
- [ ] Format detection accuracy >= 95%
- [ ] No regressions

---

## Testing Strategy

### Test Organization

```
tests/brokers/test_oanda.py (430 → ~650 líneas post-cambios)
├── TestOandaCore (existing)
├── TestOandaSideMapping (existing)
├── TestOandaAssetHandling (existing)
├── TestOandaTimestampParsing (existing)
├── TestOandaFileRowHash (existing)
├── TestOandaEdgeCases (existing)
├── TestOandaBuySelDerivation (NEW - Fase 1)
│   └── 10 tests
├── TestOandaDataValidation (NEW - Fase 2)
│   └── 22 tests
└── TestOandaCSVParsing (NEW - Fase 3, condicional)
    └── 33+ tests
```

### Test Coverage Goals

| Fase | Tests | Coverage Target |
|------|-------|-----------------|
| Current | 31 | ~85% |
| Post-Fase 1 | 41 | ~88% |
| Post-Fase 2 | 63 | ~90% |
| Post-Fase 3 | 96+ | ~92% |

### Running Tests

```bash
# All tests
pytest tests/brokers/test_oanda.py -v

# Specific phase
pytest tests/brokers/test_oanda.py::TestOandaBuySelDerivation -v
pytest tests/brokers/test_oanda.py::TestOandaDataValidation -v

# With coverage
pytest tests/brokers/test_oanda.py \
  --cov=pipeline.p01_normalize.brokers.oanda \
  --cov-report=term-missing \
  --cov-report=html

# Coverage report location
open htmlcov/index.html
```

### Performance Testing

```bash
# Benchmark large file processing
python -m pytest tests/brokers/test_oanda.py::test_large_file_performance -v -s

# Memory profiling
python -m memory_profiler tests/brokers/test_oanda_memory.py

# Load testing
python scripts/load_test_oanda.py --orders 10000 --iterations 100
```

---

## Deployment Checklist

### Pre-Deployment

#### Code Review
- [ ] All code changes reviewed
- [ ] No hardcoded credentials
- [ ] Error handling appropriate
- [ ] Logging statements meaningful
- [ ] Comments explain WHY not WHAT

#### Testing
- [ ] All unit tests passing (63+)
- [ ] Integration tests passing
- [ ] Performance tests acceptable
- [ ] Coverage >= 90%
- [ ] Manual testing completed

#### Documentation
- [ ] README updated
- [ ] CHANGELOG updated
- [ ] API documentation updated
- [ ] Migration guide created
- [ ] Known issues documented

### Deployment Steps

#### Step 1: Deploy to Staging (1 día)

```bash
# 1. Create deployment branch
git checkout -b deploy/oanda-hash-fix

# 2. Run all tests
pytest tests/brokers/test_oanda.py -v

# 3. Deploy to staging
./deploy.sh staging oanda

# 4. Monitor logs
tail -f logs/staging/oanda.log
```

**Checklist:**
- [ ] Code deployed to staging
- [ ] Staging tests passing
- [ ] Logs clean (no errors)
- [ ] Sample data processed correctly
- [ ] Hash match rate >= 95%

#### Step 2: Validate with Real Data (0.5 días)

```bash
# Process sample of real production data
python scripts/validate_oanda_hashes.py \
  --sample-size 1000 \
  --compare-legacy

# Expected output:
# Hash Match Rate: 98.2%
# Data Integrity: 100.0%
# Rejection Rate: 0.08%
```

**Checklist:**
- [ ] Real data processed successfully
- [ ] Hash match rate >= 95%
- [ ] No data corruption
- [ ] Rejection rate < 0.1%
- [ ] Performance acceptable

#### Step 3: Deploy to Production (0.5 días)

```bash
# 1. Create production deployment
git tag v1.1.0-oanda-hash-fix
git push origin v1.1.0-oanda-hash-fix

# 2. Deploy to production
./deploy.sh production oanda

# 3. Monitor for 24 hours
watch -n 300 'python scripts/monitor_oanda.py'
```

**Checklist:**
- [ ] Production deployment successful
- [ ] No errors in logs
- [ ] Hash match rate >= 95%
- [ ] No duplicate trades
- [ ] Performance acceptable
- [ ] Users not reporting issues

### Post-Deployment

#### Monitoring (1 week)

```bash
# Daily monitoring
python scripts/monitor_oanda.py --daily-report

# Metrics to track:
# - Hash match rate
# - Rejection rate
# - Processing time
# - Error rate
# - User complaints
```

**Checklist:**
- [ ] Day 1: No critical issues
- [ ] Day 3: Metrics stable
- [ ] Day 7: All metrics within targets
- [ ] User feedback positive
- [ ] No rollback required

#### Documentation

- [ ] Deployment notes updated
- [ ] Known issues documented
- [ ] Metrics baseline recorded
- [ ] Team notified
- [ ] User communication sent (if needed)

### Rollback Plan

If critical issues arise:

```bash
# 1. Immediate rollback
./rollback.sh production oanda v1.0.0

# 2. Investigate root cause
python scripts/debug_oanda_issue.py --incident-id XYZ

# 3. Fix and redeploy
# (Return to Step 1: Deploy to Staging)
```

**Triggers for Rollback:**
- Hash match rate < 90%
- Error rate > 1%
- Data corruption detected
- Performance degradation > 50%
- Critical user complaints

---

## Success Metrics Summary

### Fase 1: Critical Hash Fix

| Métrica | Target | Actual (Pre-Fix) | Expected (Post-Fix) |
|---------|--------|------------------|---------------------|
| Hash Match Rate | >= 95% | ~0% ⚠️⚠️⚠️ | >= 95% ✅ |
| Data Integrity | 100% | 100% ✅ | 100% ✅ |
| Tests Passing | 41/41 | 31/31 ✅ | 41/41 ✅ |
| Trade Duplication | 0% | 100% ⚠️⚠️⚠️ | 0% ✅ |

### Fase 2: Data Validation

| Métrica | Target | Expected |
|---------|--------|----------|
| Rejection Rate | < 0.1% | < 0.1% ✅ |
| False Positives | 0% | 0% ✅ |
| Tests Passing | 63/63 | 63/63 ✅ |
| Coverage | >= 90% | >= 90% ✅ |

### Fase 3: CSV Support (Condicional)

| Métrica | Target |
|---------|--------|
| Format Detection Accuracy | >= 95% |
| CSV Tests Passing | 96+/96+ |
| Performance | < 2x slowdown |

---

## Appendix

### A. File Locations

```
pipeline/p01_normalize/
├── brokers/oanda/
│   ├── __init__.py
│   ├── detector.py
│   └── oanda.py             ← MAIN IMPLEMENTATION
├── tests/brokers/
│   └── test_oanda.py        ← TESTS
└── old_code_from_legacy/
    ├── oanda_export.py      ← REFERENCE (API v1)
    ├── oandav2_export.py    ← REFERENCE (API v2)
    └── brokers_oanda.py     ← REFERENCE (CSV parsers)
```

### B. Key Line Numbers

**oanda.py:**
- Line 74: Add `_derive_buy_sell_legacy()` method
- Line 95: Add data validations (realized P&L, status)
- Line 100: Add symbol validation
- Line 111: Add quantity validation
- Line 114: Add price validation
- Line 123-124: Modify hash computation
- Line 147: Add average close price validation

**test_oanda.py:**
- Add `TestOandaBuySelDerivation` class (10 tests)
- Add `TestOandaDataValidation` class (22 tests)
- Add `TestOandaCSVParsing` class (33+ tests, condicional)

### C. Dependencies

```python
# oanda.py
import polars as pl
import json
import hashlib
from typing import ClassVar, Set, List, Dict, Any
from datetime import datetime
import logging

# test_oanda.py
import pytest
import polars as pl
import json
import hashlib
from datetime import datetime
```

### D. Logging Guidelines

```python
# Error (datos inválidos - skip)
logger.warning(
    f"[OANDA] Skipping order {order_id}: "
    f"reason description"
)

# Info (decisión de negocio - skip)
logger.info(
    f"[OANDA] Skipping order {order_id}: "
    f"business rule description"
)

# Debug (flujo normal)
logger.debug(
    f"[OANDA] Processing order {order_id}: "
    f"{state} trade, {units} units"
)
```

### E. Contact

**Questions or Issues:**
- Check documentation: `PLAN_ANALISIS_VALIDACIONES_OANDA.md`
- Review examples: `EJEMPLOS_CAMBIOS_CODIGO.md`
- Review changes log: `CAMBIOS_IMPLEMENTADOS.md`

---

**Última Actualización:** 2026-01-14
**Versión:** 1.0
**Status:** READY FOR IMPLEMENTATION - START WITH FASE 1 (CRÍTICO)
