# Plan: Análisis de Validaciones Críticas Faltantes - Oanda Normalizer

## Resumen Ejecutivo

Tras analizar exhaustivamente las implementaciones legacy (1,710 líneas en 3 archivos) vs nueva (802 líneas), se identificaron **20+ categorías de validaciones**, de las cuales **1 es CRÍTICA** (buy/sell field faltante en hash con ~0% match rate) y **6-7 son HIGH/MEDIUM priority** para integridad de datos.

**Hallazgo clave**: La nueva implementación es **arquitecturalmente superior** con **strong test coverage** (430 líneas de tests, 31 test cases). Sin embargo, existe un **issue crítico con hash computation**:

### Issue Crítico: Buy/Sell Field Missing in Hash

- **Hash match rate actual**: **~0%** ⚠️⚠️⚠️ (CRÍTICO - peor de todos los brokers)
- **Causa**: Legacy añade campo 'buy/sell' al order object antes de hashear, nueva implementación NO
- **Impacto**: TODOS los trades se duplicarían en re-imports
- **Integridad de datos**: 100% correcta para datos de trading

### Estado Actual: CRÍTICO ⚠️⚠️⚠️

- **Hash match rate**: ~0% (vs objetivo 95%+)
- **Data integrity**: 100% correcta
- **Test coverage**: 430 líneas (31 tests) ✅
- **Scope**: JSON API only (forex, CFDs)

**Diferencias con otros brokers:**
- ⚠️⚠️⚠️ **Issue CRÍTICO con hash** - 0% match vs 42-100% de otros brokers
- ✅ **Strong test coverage** - 430 líneas de tests comprehensivos
- ⚠️ **No CSV support** - Legacy tiene 3 CSV parsers (636 líneas)
- ⚠️ **Complex BUY/SELL derivation** - 4-branch logic basado en realizedPL

---

## Comparación de Implementaciones

### Nueva Implementación (802 líneas totales)

| Archivo | Líneas | Propósito |
|---------|--------|-----------|
| oanda.py | 308 | Main interpreter (JSON API) |
| detector.py | 50 | Format detection |
| __init__.py | 14 | Module exports |
| test_oanda.py | 430 | Comprehensive tests (31 tests) |
| **TOTAL** | **802** | **Production-ready con tests** |

### Legacy Implementación (1,710 líneas totales)

| Archivo | Líneas | Scope |
|---------|--------|-------|
| oanda_export.py (v1) | 624 | API v1 REST integration (OUT OF SCOPE: 237 lines auth/sync) |
| oandav2_export.py (v2) | 450 | API v2 REST integration (OUT OF SCOPE: 164 lines auth/sync) |
| brokers_oanda.py | 636 | CSV normalization - 3 formats (IN SCOPE) |
| **TOTAL** | **1,710** | **Mixed concerns** |

**Scope Analysis:**
- **API normalization (IN SCOPE):** ~640 lines (oanda_export 423-564, oandav2_export 313-451)
- **CSV normalization (IN SCOPE):** 636 lines (3 different CSV formats)
- **OUT OF SCOPE:** ~434 lines (API auth, sync service, DB operations)

**Reducción efectiva:** 1,276 líneas (legacy IN SCOPE) → 308 líneas (new API only) = 76% reduction ✅

---

## Validaciones Identificadas

### CATEGORÍA 1: CRÍTICO - Hash Computation Issues ⚠️⚠️⚠️

#### 1. Buy/Sell Field Missing in Hash (CRÍTICO)

**Criticidad:** CRÍTICA (afecta 100% de trades)
**Estado:** FALTANTE EN NUEVA IMPLEMENTACIÓN

**Ubicación Legacy:**
- oanda_export.py:366-373, 444-445 (añade 'buy/sell' antes de hash)
- oandav2_export.py:266-273, 327-328 (idéntica lógica)

**Ubicación Nueva:**
- oanda.py:123-124 (hashea SIN 'buy/sell' field)

**Problema:**
La nueva implementación calcula hash sobre el raw order object SIN añadir el campo 'buy/sell'. El legacy añade este campo ANTES de hashear, resultando en hashes completamente diferentes.

**Evidencia:**

**Legacy (oanda_export.py):**
```python
# Lines 366-373: Añade 'buy/sell' al trade object
if float(trade['realizedPL'])>0 and float(trade['price']) > float(trade['averageClosePrice']):
    trade['buy/sell']='SELL'
elif float(trade['realizedPL'])>0 and float(trade['price']) < float(trade['averageClosePrice']):
    trade['buy/sell']='BUY'
elif float(trade['realizedPL'])<0 and float(trade['price']) > float(trade['averageClosePrice']):
    trade['buy/sell']='BUY'
elif float(trade['realizedPL'])<0 and float(trade['price']) < float(trade['averageClosePrice']):
    trade['buy/sell']='SELL'

# Line 444-445: Hash computation DESPUÉS de añadir 'buy/sell'
njson = json.dumps(order)  # order ahora incluye 'buy/sell'
njson = hashlib.md5(njson.encode('utf-8')).hexdigest()
```

**Nueva (oanda.py):**
```python
# Lines 123-124: Hash computation SIN 'buy/sell'
order_json = json.dumps(order)  # order NO tiene 'buy/sell' añadido
base_hash = hashlib.md5(order_json.encode('utf-8')).hexdigest()
```

**Impacto:**
- Hash match rate: **~0%** (vs objetivo 95%+)
- TODOS los trades se duplicarían en re-import
- Data integrity: 100% ✓ (datos reales son correctos)
- Riesgo: Crítico - duplicación masiva de trades

**Solución Recomendada:**

```python
# oanda.py líneas ~105-145
def _derive_buy_sell_legacy(cls, order: dict) -> str:
    """
    Derive buy/sell using legacy 4-branch logic.

    Legacy logic (oanda_export.py:366-373):
    - If profit > 0 and entry > exit → SELL
    - If profit > 0 and entry < exit → BUY
    - If profit < 0 and entry > exit → BUY
    - If profit < 0 and entry < exit → SELL

    Args:
        order: Raw OANDA order with realizedPL, price, averageClosePrice

    Returns:
        "BUY" or "SELL"
    """
    try:
        realized_pl = float(order.get('realizedPL', 0))
        entry_price = float(order.get('price', 0))
        exit_price = float(order.get('averageClosePrice', 0))

        if realized_pl > 0 and entry_price > exit_price:
            return 'SELL'
        elif realized_pl > 0 and entry_price < exit_price:
            return 'BUY'
        elif realized_pl < 0 and entry_price > exit_price:
            return 'BUY'
        elif realized_pl < 0 and entry_price < exit_price:
            return 'SELL'
        else:
            # Fallback: use initialUnits sign
            initial_units = float(order.get('initialUnits', 0))
            return 'BUY' if initial_units > 0 else 'SELL'
    except (ValueError, TypeError):
        # Fallback: use initialUnits sign
        initial_units = float(order.get('initialUnits', 0))
        return 'BUY' if initial_units > 0 else 'SELL'

# In parse_json_content, BEFORE hashing (línea ~120):
for row_idx, order in enumerate(orders):
    # Create mutable copy for hash computation
    order_for_hash = dict(order)

    # Add buy/sell field using legacy logic (CRITICAL for hash match)
    buy_sell = cls._derive_buy_sell_legacy(order)
    order_for_hash['buy/sell'] = buy_sell

    # Compute hash with buy/sell included
    order_json = json.dumps(order_for_hash)
    base_hash = hashlib.md5(order_json.encode('utf-8')).hexdigest()

    # Continue with rest of logic...
```

**Archivo:** `oanda.py` líneas 105-145
**Estimado:** 2-3 días (incluye complex testing de 4-branch logic)

**Tests Requeridos:**
```python
def test_buy_sell_derivation_profit_sell():
    """Profit > 0, entry > exit → SELL"""
    order = {
        'realizedPL': 100.0,
        'price': 1.40,
        'averageClosePrice': 1.38,
        'initialUnits': '-100000',
    }
    assert _derive_buy_sell_legacy(order) == 'SELL'

def test_buy_sell_derivation_profit_buy():
    """Profit > 0, entry < exit → BUY"""
    order = {
        'realizedPL': 100.0,
        'price': 1.38,
        'averageClosePrice': 1.40,
        'initialUnits': '100000',
    }
    assert _derive_buy_sell_legacy(order) == 'BUY'

def test_buy_sell_derivation_loss_buy():
    """Loss < 0, entry > exit → BUY"""
    order = {
        'realizedPL': -100.0,
        'price': 1.40,
        'averageClosePrice': 1.38,
        'initialUnits': '100000',
    }
    assert _derive_buy_sell_legacy(order) == 'BUY'

def test_buy_sell_derivation_loss_sell():
    """Loss < 0, entry < exit → SELL"""
    order = {
        'realizedPL': -100.0,
        'price': 1.38,
        'averageClosePrice': 1.40,
        'initialUnits': '-100000',
    }
    assert _derive_buy_sell_legacy(order) == 'SELL'

def test_hash_includes_buy_sell_field():
    """Verifica que hash incluye buy/sell field"""
    order = {
        'id': '82041',
        'realizedPL': 100.0,
        'price': 1.40,
        'averageClosePrice': 1.38,
    }

    # Hash con buy/sell
    order_with_buy_sell = dict(order)
    order_with_buy_sell['buy/sell'] = 'SELL'
    hash_with = hashlib.md5(json.dumps(order_with_buy_sell).encode()).hexdigest()

    # Hash sin buy/sell
    hash_without = hashlib.md5(json.dumps(order).encode()).hexdigest()

    # Deben ser DIFERENTES
    assert hash_with != hash_without
```

---

### CATEGORÍA 2: HIGH - Data Validation Issues

#### 2. Symbol Empty Validation (HIGH)

**Estado:** FALTANTE

**Ubicación Legacy:**
- brokers_oanda.py:135 (CSV Format 1)
- brokers_oanda.py:360 (CSV Format 2)

```python
# CSV Format 1 (line 135)
if not n['price']:
    continue

# CSV Format 2 (line 360)
if ... or not n['symbol'] or ...:
    continue
```

**Estado Nuevo:** Sin validación de symbol vacío

**Implementación:**
```python
# oanda.py después de línea 164
instrument = order.get("instrument", "")
if not instrument or not instrument.strip():
    logger.warning(f"[OANDA] Skipping order {order.get('id', 'unknown')}: empty instrument")
    continue
```

**Archivo:** `oanda.py` línea ~100
**Estimado:** 0.25 días

---

#### 3. Price Validation (HIGH)

**Estado:** FALTANTE

**Ubicación Legacy:**
- brokers_oanda.py:135 (CSV Format 1): `if not n['price']`
- brokers_oanda.py:373-374 (CSV Format 2): float validation

**Implementación:**
```python
# oanda.py después de línea 114
entry_price = float(order.get("price", 0) or 0)
exit_price = float(order.get("averageClosePrice", 0) or 0) if state == "CLOSED" else 0

if entry_price <= 0:
    logger.warning(f"[OANDA] Skipping order {order.get('id', 'unknown')}: zero or negative entry price")
    continue

if state == "CLOSED" and exit_price <= 0:
    logger.warning(f"[OANDA] Skipping order {order.get('id', 'unknown')}: zero or negative exit price")
    continue
```

**Archivo:** `oanda.py` línea ~114
**Estimado:** 0.25 días

---

#### 4. Realized P&L Zero Skip (MEDIUM-HIGH)

**Estado:** FALTANTE

**Ubicación Legacy:**
- oanda_export.py:364-365 (API v1)
- oandav2_export.py:264-265 (API v2)
- brokers_oanda.py:459-460 (CSV)

```python
if "realizedPL" in order and float(order['realizedPL']) == 0:
    continue
```

**Implementación:**
```python
# oanda.py después de línea 95
realized_pl = float(order.get("realizedPL", 0) or 0)
if realized_pl == 0:
    logger.info(f"[OANDA] Skipping order {order.get('id', 'unknown')}: zero realized P&L")
    continue
```

**Archivo:** `oanda.py` línea ~95
**Estimado:** 0.25 días

---

#### 5. Status FILLED Validation (MEDIUM)

**Estado:** FALTANTE

**Ubicación Legacy:**
- oanda_export.py:461-462 (API v1)
- oandav2_export.py:345-346 (API v2)

```python
if 'status' in order and order['status'] != 'FILLED':
    continue
```

**Implementación:**
```python
# oanda.py después de línea 95
status = order.get("status", "")
if status and status != "FILLED":
    logger.info(f"[OANDA] Skipping order {order.get('id', 'unknown')}: status {status} (not FILLED)")
    continue
```

**Archivo:** `oanda.py` línea ~95
**Estimado:** 0.25 días

---

#### 6. Quantity Validation (MEDIUM)

**Estado:** FALTANTE

**Ubicación Legacy:**
- brokers_oanda.py:163 (CSV Format 1): float check
- brokers_oanda.py:540 (CSV Format 3): numeric string check

**Implementación:**
```python
# oanda.py después de línea 111
initial_units = float(order.get("initialUnits", 0) or 0)
if initial_units == 0:
    logger.warning(f"[OANDA] Skipping order {order.get('id', 'unknown')}: zero initial units")
    continue
```

**Archivo:** `oanda.py` línea ~111
**Estimado:** 0.25 días

---

#### 7. Average Close Price Validation (MEDIUM)

**Estado:** FALTANTE

**Ubicación Legacy:**
- oanda_export.py:362-363 (API v1)

```python
if not 'averageClosePrice' in trade:
    continue
```

**Implementación:**
```python
# oanda.py línea ~147
if state == "CLOSED":
    avg_close_price = order.get("averageClosePrice")
    if not avg_close_price:
        logger.warning(f"[OANDA] Skipping closed order {order.get('id', 'unknown')}: missing averageClosePrice")
        continue
```

**Archivo:** `oanda.py` línea ~147
**Estimado:** 0.25 días

---

### CATEGORÍA 3: CONDITIONAL - CSV Support

#### 8. CSV Parser Support (CONDICIONAL)

**Estado:** NO IMPLEMENTADO

**Legacy Scope:**
- `brokers_oanda.py` - 636 líneas con 3 CSV formats:
  1. **Format 1 (getcsv)** - Lines 1-253: Consecutive item format
  2. **Format 2 (getcsv2)** - Lines 257-489: Open/close on same row
  3. **Format 3 (getcsv3)** - Lines 491-612: Japanese format

**Decisión Requerida:** Ejecutar SQL query

```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*) OVER (), 2) as percentage,
    MAX(created_at) as last_used
FROM data_sources
WHERE broker_id = 'oanda'
  AND created_at > NOW() - INTERVAL '12 months'
GROUP BY source_type
ORDER BY count DESC;
```

**Criterio:**
- Si CSV percentage > 5%: **IMPLEMENTAR** (8-12 días)
- Si CSV percentage < 5%: **OMITIR**

**Complejidad:** ALTA (3 formats, auto-detection, ~636 líneas)
**Estimado:** 8-12 días (solo si necesario)

---

### CATEGORÍA 4: OUT OF SCOPE (Correctamente Excluido)

#### 9-14. API Authentication & Sync Service ✅

- **Credential Validation** (oanda_export.py:238-277, oandav2_export.py:165-214)
- **Login/Session** (oanda_export.py:278-294, oandav2_export.py:215-233)
- **API Rate Limiting** (implicit in pagination logic)
- **Pagination Logic** (oanda_export.py:329, oandav2_export.py:240)
- **File Saving/Archiving** (oanda_export.py:565-625)
- **Database Deduplication** (brokers_oanda.py:614-637)

**Conclusión:** OUT OF SCOPE - Pertenece a sync service y p05_write stage

---

### CATEGORÍA 5: ALREADY IMPLEMENTED ✅

#### 15. ✅ JSON Validation
**Estado:** IMPLEMENTADO (parse_json_content validates JSON structure)

#### 16. ✅ Timestamp Conversion
**Estado:** IMPLEMENTADO (oanda.py:187-205)
- Converts Unix seconds.nanoseconds → milliseconds datetime

#### 17. ✅ Symbol Normalization
**Estado:** IMPLEMENTADO (oanda.py:237)
- Removes underscores, converts to uppercase

#### 18. ✅ Trade Splitting
**Estado:** IMPLEMENTADO (oanda.py:146-169)
- Splits closed trades into open/close executions

#### 19. ✅ Asset Type Classification
**Estado:** IMPLEMENTADO (oanda.py:265)
- All classified as "forex"

#### 20. ✅ Commission/Fee Handling
**Estado:** IMPLEMENTADO (oanda.py:252, 255, 258)
- Commission: 0.0 (spread-based pricing)
- Fees: 0.0
- Swap: financing on exit only

---

## Plan de Implementación por Fases

### FASE 1: Critical Hash Fix (2-3 días) ⚠️⚠️⚠️ URGENTE

**Objetivo:** Corregir hash match rate de ~0% a 95%+

**Tareas:**
1. Implementar `_derive_buy_sell_legacy()` con 4-branch logic (1 día)
2. Modificar hash computation para incluir buy/sell field (0.5 días)
3. Testing comprehensivo de 4 branches + fallback (0.5 días)
4. Validación con datos reales (0.5 días)
5. Integration testing open + closed trades (0.5 días)

**Archivos:**
- `oanda.py:105-145` (buy/sell derivation + hash fix)

**Métricas de Éxito:**
- [ ] Hash match rate >= 95% (vs actual ~0%)
- [ ] Buy/sell derivation correcta para las 4 branches
- [ ] Open trades correctamente hasheados
- [ ] Closed trades (both executions) con mismo hash
- [ ] Data integrity mantenida (100%)

**Complejidad:** ALTA
**Riesgo:** ALTO (4-branch logic compleja, critical para deduplication)
**Estimado:** 2-3 días

---

### FASE 2: Data Validation (1-1.5 días)

**Objetivo:** Prevenir datos inválidos

**Tareas:**
1. Symbol validation (0.25 días)
2. Price validation (entry + exit) (0.25 días)
3. Realized P&L zero skip (0.25 días)
4. Status FILLED validation (0.25 días)
5. Quantity validation (0.25 días)
6. Average close price validation (0.25 días)

**Archivos:** `oanda.py:95-150`

**Métricas de Éxito:**
- [ ] Zero symbols vacíos
- [ ] Zero prices zero/negativo
- [ ] Zero P&L trades skipped
- [ ] Non-FILLED status skipped
- [ ] Zero quantities rejected
- [ ] Closed trades sin averageClosePrice rejected
- [ ] Rejection rate < 0.1%

**Complejidad:** BAJA
**Riesgo:** BAJO
**Estimado:** 1-1.5 días

---

### FASE 3: CSV Support (8-12 días - SI NECESARIO) 🎯

**BLOQUEADOR:** Ejecutar SQL query primero

**Criterio:**
- Si CSV > 5%: **IMPLEMENTAR**
- Si CSV < 5%: **OMITIR**

**Complejidad:** MUY ALTA (3 formats, auto-detection, 636 líneas legacy)
**Riesgo:** MEDIO
**Estimado:** 8-12 días (solo si necesario)

**Scope:**
- Format 1 (getcsv): Consecutive item format - 3 días
- Format 2 (getcsv2): Open/close same row - 3 días
- Format 3 (getcsv3): Japanese format - 2 días
- Auto-detection logic - 1 día
- Tests comprehensivos - 2 días
- Documentation - 1 día

---

## Matriz de Priorización

| # | Validación | Criticidad | Estado | Complejidad | Fase | Días |
|---|------------|-----------|--------|-------------|------|------|
| 1 | Buy/Sell Hash Field | ⚠️⚠️⚠️ CRÍTICO | MISSING | ALTA | 1 | 2-3 |
| 2 | Symbol Validation | ⭐⭐⭐ ALTA | MISSING | BAJA | 2 | 0.25 |
| 3 | Price Validation | ⭐⭐⭐ ALTA | MISSING | BAJA | 2 | 0.25 |
| 4 | Realized P&L Zero | ⭐⭐ MEDIA-ALTA | MISSING | BAJA | 2 | 0.25 |
| 5 | Status FILLED | ⭐⭐ MEDIA | MISSING | BAJA | 2 | 0.25 |
| 6 | Quantity Validation | ⭐⭐ MEDIA | MISSING | BAJA | 2 | 0.25 |
| 7 | Avg Close Price | ⭐⭐ MEDIA | MISSING | BAJA | 2 | 0.25 |
| 8 | CSV Support (3 formats) | 🎯 CONDICIONAL | NOT IMPL | MUY ALTA | 3 | 8-12 |
| 9-14 | Out of Scope | ✅ N/A | CORRECTO | - | - | 0 |
| 15-20 | Ya Implementadas | ✅ DONE | DONE | - | - | 0 |

**Total Fase 1 (Hash Fix):** 2-3 días
**Total Fase 2 (Validations):** 1-1.5 días
**Total Fase 1-2:** 3-4.5 días
**Total Fase 3 (Condicional):** 8-12 días adicionales si necesario

---

## Comparación con Otros Brokers

| Broker | Legacy Lines | New Lines | Reduction | Hash Match | Critical Issues | Phase 1-2 Days |
|--------|--------------|-----------|-----------|------------|----------------|----------------|
| **Oanda** | **1,276*** | **308** | **76%** ✅ | **~0%** ⚠️⚠️⚠️ | **7** | **3-4.5** |
| Deribit | 326 | 516 | -58% | 74.27% ⚠️ | 5 | 2-3.5 |
| Charles Schwab | 3,196 | 1,087 | 66% | 42% ⚠️ | 6 | 3.5-5.5 |
| OKX | 170 | 362 | -113% | 95-100% ✅ | 5 | 1.25-1.75 |
| KuCoin | 2,015 | 404 | 80% | 100% ✅ | 4-5 | 2.5-3.5 |

*Nota: Legacy IN SCOPE = 1,276 lines (API normalization 640 + CSV 636)

**Observaciones Oanda:**
- ⚠️⚠️⚠️ **Issue CRÍTICO más severo** - 0% hash match (peor de todos)
- ✅ **Mejor reducción de código** - 76% reduction (mejor que todos)
- ⚠️ **Complex BUY/SELL logic** - 4-branch derivation basado en P&L
- ✅ **Strong test coverage** - 430 líneas (31 tests)
- ⚠️ **3 CSV formats** - Más complejo que otros (OKX: 1, Deribit: 2)
- ✅ **Expansion de código es positiva** - Incluye tests + mejor estructura

---

## Archivos Críticos

### 1. `brokers/oanda/oanda.py` (PRINCIPAL)
**Líneas Actuales:** 308
**Estimadas Post-Cambios:** ~380-420 (Fases 1-2), ~1,000-1,200 (si Fase 3)

**Cambios:**
- Líneas 105-145: Buy/sell derivation + hash fix (Fase 1 - CRÍTICO)
- Líneas 95-150: Data validations (Fase 2)

### 2. `tests/brokers/test_oanda.py` (TESTS)
**Líneas Actuales:** 430
**Estimadas Post-Cambios:** 550-600

**Tests Nuevos Requeridos:**
- test_buy_sell_derivation_profit_sell()
- test_buy_sell_derivation_profit_buy()
- test_buy_sell_derivation_loss_buy()
- test_buy_sell_derivation_loss_sell()
- test_hash_includes_buy_sell_field()
- test_symbol_empty_validation()
- test_price_zero_validation()
- test_realized_pl_zero_skip()
- test_status_not_filled_skip()
- test_quantity_zero_validation()
- test_avg_close_price_missing_validation()

### 3. Legacy Files (REFERENCIA SOLO)
- `old_code_from_legacy/oanda_export.py` (624 líneas - API v1)
- `old_code_from_legacy/oandav2_export.py` (450 líneas - API v2)
- `old_code_from_legacy/brokers_oanda.py` (636 líneas - CSV parsers)

---

## Decisiones Pendientes

### Decisión 1: Buy/Sell Derivation Strategy

**Pregunta:** ¿Usar complex 4-branch P&L logic o simple initialUnits sign?

**Opciones:**
- **A: 4-branch P&L logic (RECOMENDADO para hash compatibility)**
  - Expected: 95%+ hash match rate
  - Requiere: preservar exact legacy formula
  - Esfuerzo: 2-3 días
  - Riesgo: ALTO (complex logic)

- **B: Simple initialUnits sign (NO RECOMENDADO)**
  - Expected: 0% hash match rate
  - No code changes en hash
  - Todos los trades se duplicarían

**Recomendación:** **Opción A** (implementar 4-branch logic)

### Decisión 2: CSV Support Necessity

**Pregunta:** ¿Implementar 3 CSV parsers (636 líneas)?

**Data Requerida:** Ejecutar SQL query

**Criterio:**
- Si CSV > 5%: **IMPLEMENTAR** (8-12 días)
- Si CSV < 5%: **OMITIR**

**Recomendación:** **Query data first, then decide**

---

## Conclusión

Análisis identifica **20+ categorías de validaciones**:
- **1 CRÍTICA (Fase 1):** 2-3 días - Buy/sell hash field (0% match rate)
- **6 HIGH/MEDIUM (Fase 2):** 1-1.5 días - Data validations
- **1 CONDICIONAL (Fase 3):** 8-12 días si necesario - CSV support (3 formats)
- **6 OUT OF SCOPE:** Sin acción (correctamente excluidas)
- **6 YA IMPLEMENTADAS:** Sin acción

**Hallazgo Clave:** Nueva implementación es **excelente arquitecturalmente** (76% code reduction, strong tests) pero tiene el **issue de hash MÁS CRÍTICO** de todos los brokers analizados:
- 0% hash match rate (vs 42-100% de otros)
- Requiere complex 4-branch BUY/SELL derivation
- TODOS los trades se duplicarían sin fix
- Data integrity 100% correcta (solo hash afectado)

**Diferencia vs Otros Brokers:**
- Oanda: 0% hash match (buy/sell field faltante)
- Deribit: 74.27% hash match (expired options suffix)
- Schwab: 42% hash match (closingPrice volatility)
- OKX: 95-100% hash match (sin issues)

**RECOMENDACIÓN FINAL:**
1. **Implementar Fase 1 INMEDIATAMENTE** (2-3 días) - Hash fix CRÍTICO
2. **Implementar Fase 2** (1-1.5 días) - Data validations
3. **Ejecutar SQL query antes de Fase 3** - Decisión sobre CSV
4. **NO sobre-ingenierizar** - La implementación actual es sólida

Total estimado crítico: **3-4.5 días**
Total con CSV condicional: **11-16.5 días** (solo si datos muestran necesidad)

---

## Verificación

### Tests a Ejecutar Post-Implementación

**Unit Tests:**
```bash
pytest tests/brokers/test_oanda.py -v
```

**Hash Match Rate Verification:**
```python
def test_hash_match_rate_legacy_compatibility():
    # Load sample trades
    # Process with new code
    # Compare hashes with legacy
    # Assert match_rate >= 0.95
```

**Manual Verification:**
```bash
# 1. Verify buy/sell derivation
grep "buy/sell" logs/oanda.log | head -20

# 2. Verify hash match rate
# Compare file_row hashes with legacy data
# Target: >= 95%

# 3. Check validation warnings
grep "Skipping order" logs/oanda.log

# 4. Verify rejection rate
# Calculate: rejected_orders / total_orders
# Target: < 0.1%
```

---

**Fecha de Análisis:** 2026-01-14
**Broker ID:** oanda
**Formato:** JSON API (forex, CFDs)
**Assets:** forex (primary)
