# Ejemplos de Cambios de Código - Oanda Normalizer

## Índice

1. [CRÍTICO: Buy/Sell Field in Hash](#1-crítico-buysell-field-in-hash)
2. [Symbol Validation](#2-symbol-validation)
3. [Price Validation](#3-price-validation)
4. [Realized P&L Zero Skip](#4-realized-pl-zero-skip)
5. [Status FILLED Validation](#5-status-filled-validation)
6. [Quantity Validation](#6-quantity-validation)
7. [Average Close Price Validation](#7-average-close-price-validation)

---

## 1. CRÍTICO: Buy/Sell Field in Hash

### Problema

**Hash match rate actual: ~0%** ⚠️⚠️⚠️

El legacy añade campo 'buy/sell' al order object ANTES de calcular el hash MD5. La nueva implementación NO añade este campo, resultando en hashes completamente diferentes y 100% de trades duplicados en re-imports.

### Código Legacy (oanda_export.py)

```python
# Lines 366-373: Complex 4-branch BUY/SELL derivation
if float(trade['realizedPL'])>0 and float(trade['price']) > float(trade['averageClosePrice']):
    trade['buy/sell']='SELL'
elif float(trade['realizedPL'])>0 and float(trade['price']) < float(trade['averageClosePrice']):
    trade['buy/sell']='BUY'
elif float(trade['realizedPL'])<0 and float(trade['price']) > float(trade['averageClosePrice']):
    trade['buy/sell']='BUY'
elif float(trade['realizedPL'])<0 and float(trade['price']) < float(trade['averageClosePrice']):
    trade['buy/sell']='SELL'

# Lines 444-445: Hash computation DESPUÉS de añadir 'buy/sell'
njson = json.dumps(order)  # order ahora incluye 'buy/sell'
njson = hashlib.md5(njson.encode('utf-8')).hexdigest()
```

### Código Actual (oanda.py:123-124) ❌

```python
# Lines 123-124: Hash computation SIN 'buy/sell' field
order_json = json.dumps(order)  # order NO tiene 'buy/sell' añadido
base_hash = hashlib.md5(order_json.encode('utf-8')).hexdigest()
```

**Resultado:** 0% hash match rate

### Código Propuesto ✅

```python
# oanda.py - Add method at class level (~line 74)
@classmethod
def _derive_buy_sell_legacy(cls, order: dict) -> str:
    """
    Derive buy/sell using legacy 4-branch logic.

    Legacy logic (oanda_export.py:366-373):
    - If profit > 0 and entry > exit → SELL
    - If profit > 0 and entry < exit → BUY
    - If profit < 0 and entry > exit → BUY
    - If profit < 0 and entry < exit → SELL

    Args:
        order: Raw OANDA order with realizedPL, price, averageClosePrice

    Returns:
        "BUY" or "SELL"
    """
    try:
        realized_pl = float(order.get('realizedPL', 0))
        entry_price = float(order.get('price', 0))
        exit_price = float(order.get('averageClosePrice', 0))

        if realized_pl > 0 and entry_price > exit_price:
            return 'SELL'
        elif realized_pl > 0 and entry_price < exit_price:
            return 'BUY'
        elif realized_pl < 0 and entry_price > exit_price:
            return 'BUY'
        elif realized_pl < 0 and entry_price < exit_price:
            return 'SELL'
        else:
            # Fallback: use initialUnits sign
            initial_units = float(order.get('initialUnits', 0))
            return 'BUY' if initial_units > 0 else 'SELL'
    except (ValueError, TypeError):
        # Fallback: use initialUnits sign
        initial_units = float(order.get('initialUnits', 0))
        return 'BUY' if initial_units > 0 else 'SELL'


# oanda.py - Modify parse_json_content (~line 120-145)
# BEFORE hashing
for row_idx, order in enumerate(orders):
    # ... existing validations ...

    # Create mutable copy for hash computation
    order_for_hash = dict(order)

    # CRITICAL: Add buy/sell field using legacy logic (for hash compatibility)
    buy_sell = cls._derive_buy_sell_legacy(order)
    order_for_hash['buy/sell'] = buy_sell

    # Compute hash with buy/sell included
    order_json = json.dumps(order_for_hash)
    base_hash = hashlib.md5(order_json.encode('utf-8')).hexdigest()

    # Continue with rest of logic...
    trade_id = str(order.get("id", ""))
    instrument = str(order.get("instrument", ""))
    # ... rest of existing code ...
```

### Tests Requeridos

```python
# tests/brokers/test_oanda.py

def test_derive_buy_sell_profit_sell():
    """Profit > 0, entry > exit → SELL"""
    order = {
        'id': '82041',
        'realizedPL': 100.0,
        'price': 1.40,
        'averageClosePrice': 1.38,
        'initialUnits': '-100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'SELL'


def test_derive_buy_sell_profit_buy():
    """Profit > 0, entry < exit → BUY"""
    order = {
        'id': '82042',
        'realizedPL': 100.0,
        'price': 1.38,
        'averageClosePrice': 1.40,
        'initialUnits': '100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'BUY'


def test_derive_buy_sell_loss_buy():
    """Loss < 0, entry > exit → BUY"""
    order = {
        'id': '82043',
        'realizedPL': -100.0,
        'price': 1.40,
        'averageClosePrice': 1.38,
        'initialUnits': '100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'BUY'


def test_derive_buy_sell_loss_sell():
    """Loss < 0, entry < exit → SELL"""
    order = {
        'id': '82044',
        'realizedPL': -100.0,
        'price': 1.38,
        'averageClosePrice': 1.40,
        'initialUnits': '-100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'SELL'


def test_derive_buy_sell_fallback_buy():
    """Fallback to initialUnits when price comparison inconclusive"""
    order = {
        'id': '82045',
        'realizedPL': 0.0,  # Edge case
        'price': 1.40,
        'averageClosePrice': 1.40,  # Same price
        'initialUnits': '100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'BUY'


def test_derive_buy_sell_fallback_sell():
    """Fallback to initialUnits when price comparison inconclusive"""
    order = {
        'id': '82046',
        'realizedPL': 0.0,
        'price': 1.40,
        'averageClosePrice': 1.40,
        'initialUnits': '-100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'SELL'


def test_derive_buy_sell_invalid_data():
    """Handle invalid/missing data gracefully"""
    order = {
        'id': '82047',
        'realizedPL': 'invalid',
        'initialUnits': '100000',
    }
    # Should fallback to initialUnits
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'BUY'


def test_hash_includes_buy_sell_field():
    """Verifica que hash incluye buy/sell field"""
    order = {
        'id': '82041',
        'realizedPL': 100.0,
        'price': 1.40,
        'averageClosePrice': 1.38,
        'instrument': 'USD_CAD',
        'initialUnits': '-100000',
    }

    # Hash con buy/sell (CORRECT - legacy compatible)
    order_with_buy_sell = dict(order)
    order_with_buy_sell['buy/sell'] = 'SELL'
    hash_with = hashlib.md5(
        json.dumps(order_with_buy_sell).encode()
    ).hexdigest()

    # Hash sin buy/sell (INCORRECT - nuevo actual)
    hash_without = hashlib.md5(
        json.dumps(order).encode()
    ).hexdigest()

    # Deben ser DIFERENTES
    assert hash_with != hash_without

    # El hash correcto debe incluir buy/sell
    assert 'buy/sell' in json.dumps(order_with_buy_sell)


def test_hash_match_rate_integration():
    """Integration test: Verifica hash match rate >= 95%"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82041",
                "instrument": "USD_CAD",
                "price": "1.40",
                "averageClosePrice": "1.38",
                "openTime": "1767371240.327566168",
                "closeTime": "1767371340.123456789",
                "initialUnits": "-100000",
                "realizedPL": "100.0",
                "financing": "0.5",
                "state": "CLOSED"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)

    # Verify hash was computed with buy/sell field
    # (In real test, compare against known legacy hashes)
    assert df is not None
    assert len(df) == 2  # Entry + exit
    assert '_file_row_hash' in df.columns

    # Both executions should have same hash (same order)
    hashes = df['_file_row_hash'].to_list()
    assert hashes[0] == hashes[1]  # Open and close share hash
```

### Verificación de Éxito

```python
# Verificación manual del hash
import json
import hashlib

order = {
    'id': '82041',
    'realizedPL': 100.0,
    'price': 1.40,
    'averageClosePrice': 1.38,
}

# Legacy approach (CORRECT)
order_legacy = dict(order)
order_legacy['buy/sell'] = 'SELL'  # Derived from 4-branch logic
hash_legacy = hashlib.md5(json.dumps(order_legacy).encode()).hexdigest()

# New approach (INCORRECT - actual)
hash_new = hashlib.md5(json.dumps(order).encode()).hexdigest()

print(f"Legacy hash: {hash_legacy}")
print(f"New hash:    {hash_new}")
print(f"Match:       {hash_legacy == hash_new}")  # False → 0% match rate

# After fix (CORRECT)
order_fixed = dict(order)
buy_sell = OandaInterpreter._derive_buy_sell_legacy(order)  # Returns 'SELL'
order_fixed['buy/sell'] = buy_sell
hash_fixed = hashlib.md5(json.dumps(order_fixed).encode()).hexdigest()

print(f"Fixed hash:  {hash_fixed}")
print(f"Match:       {hash_legacy == hash_fixed}")  # True → 100% match rate
```

**Resultado Esperado:** Hash match rate >= 95%

---

## 2. Symbol Validation

### Problema

Símbolos vacíos pueden ingresar al sistema sin validación.

### Código Legacy (brokers_oanda.py)

```python
# Line 135 (CSV Format 1)
if not n['price']:
    continue

# Line 360 (CSV Format 2)
if ... or not n['symbol'] or ...:
    continue
```

### Código Actual ❌

Sin validación de symbol vacío.

### Código Propuesto ✅

```python
# oanda.py - In parse_json_content (~line 100)
for row_idx, order in enumerate(orders):
    trade_id = str(order.get("id", ""))
    instrument = order.get("instrument", "")

    # Symbol validation
    if not instrument or not instrument.strip():
        logger.warning(
            f"[OANDA] Skipping order {trade_id or 'unknown'}: "
            f"empty instrument"
        )
        continue

    state = order.get("state", "")
    # ... rest of code ...
```

### Tests Requeridos

```python
def test_empty_instrument_validation():
    """Empty instrument should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82041",
                "instrument": "",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped


def test_whitespace_instrument_validation():
    """Whitespace-only instrument should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82042",
                "instrument": "   ",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped


def test_missing_instrument_validation():
    """Missing instrument field should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82043",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped
```

---

## 3. Price Validation

### Problema

Precios zero o negativos causan errores en cálculos posteriores.

### Código Legacy

```python
# brokers_oanda.py:135 (CSV Format 1)
if not n['price']:
    continue

# brokers_oanda.py:373-374 (CSV Format 2)
n['price'] = float(line[2]) if line[2] else 0.0
```

### Código Actual ❌

Sin validación de price <= 0.

### Código Propuesto ✅

```python
# oanda.py - In parse_json_content (~line 114)
for row_idx, order in enumerate(orders):
    # ... existing validations ...

    # Entry price validation
    entry_price = float(order.get("price", 0) or 0)
    if entry_price <= 0:
        logger.warning(
            f"[OANDA] Skipping order {trade_id or 'unknown'}: "
            f"zero or negative entry price ({entry_price})"
        )
        continue

    # Exit price validation (for CLOSED trades)
    if state == "CLOSED":
        exit_price = float(order.get("averageClosePrice", 0) or 0)
        if exit_price <= 0:
            logger.warning(
                f"[OANDA] Skipping order {trade_id or 'unknown'}: "
                f"zero or negative exit price ({exit_price})"
            )
            continue

    # ... rest of code ...
```

### Tests Requeridos

```python
def test_zero_entry_price_validation():
    """Zero entry price should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82041",
                "instrument": "USD_CAD",
                "price": "0",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped


def test_negative_entry_price_validation():
    """Negative entry price should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82042",
                "instrument": "USD_CAD",
                "price": "-1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped


def test_zero_exit_price_validation():
    """Zero exit price for CLOSED trade should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82043",
                "instrument": "USD_CAD",
                "price": "1.40",
                "averageClosePrice": "0",
                "openTime": "1767371240.327566168",
                "closeTime": "1767371340.123456789",
                "initialUnits": "100000",
                "realizedPL": "100.0",
                "state": "CLOSED"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped


def test_missing_price_validation():
    """Missing price field should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82044",
                "instrument": "USD_CAD",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped
```

---

## 4. Realized P&L Zero Skip

### Problema

Trades con realized P&L zero contaminan datos y no representan posiciones reales cerradas.

### Código Legacy

```python
# oanda_export.py:364-365 (API v1)
if "realizedPL" in order and float(order['realizedPL']) == 0:
    continue

# oandav2_export.py:264-265 (API v2)
if "realizedPL" in order and float(order['realizedPL']) == 0:
    continue

# brokers_oanda.py:459-460 (CSV)
if float(n['njson']['realizedPL']) == 0:
    continue
```

### Código Actual ❌

Sin validación de realized P&L zero.

### Código Propuesto ✅

```python
# oanda.py - In parse_json_content (~line 95)
for row_idx, order in enumerate(orders):
    # ... existing code ...

    # Realized P&L zero skip
    realized_pl = float(order.get("realizedPL", 0) or 0)
    if realized_pl == 0:
        logger.info(
            f"[OANDA] Skipping order {order.get('id', 'unknown')}: "
            f"zero realized P&L"
        )
        continue

    # ... rest of code ...
```

### Tests Requeridos

```python
def test_zero_realized_pl_skip():
    """Orders with zero realized P&L should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82041",
                "instrument": "USD_CAD",
                "price": "1.40",
                "averageClosePrice": "1.40",
                "openTime": "1767371240.327566168",
                "closeTime": "1767371340.123456789",
                "initialUnits": "100000",
                "realizedPL": "0",
                "state": "CLOSED"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped


def test_positive_realized_pl_accepted():
    """Orders with positive realized P&L should be accepted"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82042",
                "instrument": "USD_CAD",
                "price": "1.40",
                "averageClosePrice": "1.38",
                "openTime": "1767371240.327566168",
                "closeTime": "1767371340.123456789",
                "initialUnits": "-100000",
                "realizedPL": "100.0",
                "financing": "0.5",
                "state": "CLOSED"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 2  # Entry + exit


def test_negative_realized_pl_accepted():
    """Orders with negative realized P&L should be accepted"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82043",
                "instrument": "USD_CAD",
                "price": "1.38",
                "averageClosePrice": "1.40",
                "openTime": "1767371240.327566168",
                "closeTime": "1767371340.123456789",
                "initialUnits": "-100000",
                "realizedPL": "-100.0",
                "financing": "0.5",
                "state": "CLOSED"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 2  # Entry + exit
```

---

## 5. Status FILLED Validation

### Problema

Órdenes con status != "FILLED" no representan trades ejecutados.

### Código Legacy

```python
# oanda_export.py:461-462 (API v1)
if 'status' in order and order['status'] != 'FILLED':
    continue

# oandav2_export.py:345-346 (API v2)
if 'status' in order and order['status'] != 'FILLED':
    continue
```

### Código Actual ❌

Sin validación de status field.

### Código Propuesto ✅

```python
# oanda.py - In parse_json_content (~line 95)
for row_idx, order in enumerate(orders):
    # ... existing validations ...

    # Status FILLED validation
    status = order.get("status", "")
    if status and status != "FILLED":
        logger.info(
            f"[OANDA] Skipping order {order.get('id', 'unknown')}: "
            f"status {status} (not FILLED)"
        )
        continue

    # ... rest of code ...
```

### Tests Requeridos

```python
def test_status_pending_skip():
    """Orders with status=PENDING should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82041",
                "instrument": "USD_CAD",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN",
                "status": "PENDING"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped


def test_status_cancelled_skip():
    """Orders with status=CANCELLED should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82042",
                "instrument": "USD_CAD",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "CLOSED",
                "status": "CANCELLED"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped


def test_status_filled_accepted():
    """Orders with status=FILLED should be accepted"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82043",
                "instrument": "USD_CAD",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN",
                "status": "FILLED"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 1  # Entry only


def test_missing_status_accepted():
    """Orders without status field should be accepted"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82044",
                "instrument": "USD_CAD",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 1  # Entry only
```

---

## 6. Quantity Validation

### Problema

Cantidad zero causa errores en cálculos posteriores.

### Código Legacy

```python
# brokers_oanda.py:163 (CSV Format 1)
if not is_float(line[7]):  # quantity validation
    continue

# brokers_oanda.py:540 (CSV Format 3)
if not str(line[4]).isnumeric():  # numeric string check
    continue
```

### Código Actual ❌

Sin validación de initialUnits == 0.

### Código Propuesto ✅

```python
# oanda.py - In parse_json_content (~line 111)
for row_idx, order in enumerate(orders):
    # ... existing validations ...

    # Quantity validation
    initial_units = float(order.get("initialUnits", 0) or 0)
    if initial_units == 0:
        logger.warning(
            f"[OANDA] Skipping order {trade_id or 'unknown'}: "
            f"zero initial units"
        )
        continue

    # Determine direction: positive = BUY, negative = SELL
    is_buy = initial_units > 0
    units = abs(initial_units)

    # ... rest of code ...
```

### Tests Requeridos

```python
def test_zero_quantity_validation():
    """Zero initialUnits should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82041",
                "instrument": "USD_CAD",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "0",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped


def test_missing_quantity_validation():
    """Missing initialUnits should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82042",
                "instrument": "USD_CAD",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 0  # Should be skipped


def test_positive_quantity_accepted():
    """Positive initialUnits should be accepted (BUY)"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82043",
                "instrument": "USD_CAD",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 1  # Entry only
    assert df["side"][0] == "BUY"


def test_negative_quantity_accepted():
    """Negative initialUnits should be accepted (SELL)"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82044",
                "instrument": "USD_CAD",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "-100000",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 1  # Entry only
    assert df["side"][0] == "SELL"
```

---

## 7. Average Close Price Validation

### Problema

Trades cerrados sin averageClosePrice no pueden calcular P&L correctamente.

### Código Legacy

```python
# oanda_export.py:362-363 (API v1)
if not 'averageClosePrice' in trade:
    continue
```

### Código Actual ❌

Sin validación de averageClosePrice para CLOSED trades.

### Código Propuesto ✅

```python
# oanda.py - In parse_json_content (~line 147)
for row_idx, order in enumerate(orders):
    # ... existing code ...

    # Create ENTRY execution
    entry_record = {
        # ... existing fields ...
    }
    records.append(entry_record)

    # For CLOSED trades, create EXIT execution
    if state == "CLOSED":
        # Average close price validation
        avg_close_price = order.get("averageClosePrice")
        if not avg_close_price:
            logger.warning(
                f"[OANDA] Skipping closed order {trade_id or 'unknown'}: "
                f"missing averageClosePrice"
            )
            continue

        exit_price = float(avg_close_price or 0)
        if exit_price <= 0:
            logger.warning(
                f"[OANDA] Skipping closed order {trade_id or 'unknown'}: "
                f"invalid averageClosePrice ({exit_price})"
            )
            continue

        close_time = order.get("closeTime", "")

        # Exit is opposite direction
        exit_record = {
            # ... existing fields ...
        }
        records.append(exit_record)
```

### Tests Requeridos

```python
def test_missing_average_close_price_validation():
    """CLOSED trade without averageClosePrice should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82041",
                "instrument": "USD_CAD",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "closeTime": "1767371340.123456789",
                "initialUnits": "100000",
                "realizedPL": "100.0",
                "state": "CLOSED"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    # Should create entry but skip exit → only 1 record
    # OR skip entire order → 0 records
    # Depends on implementation choice
    assert len(df) <= 1


def test_zero_average_close_price_validation():
    """CLOSED trade with zero averageClosePrice should be skipped"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82042",
                "instrument": "USD_CAD",
                "price": "1.40",
                "averageClosePrice": "0",
                "openTime": "1767371240.327566168",
                "closeTime": "1767371340.123456789",
                "initialUnits": "100000",
                "realizedPL": "100.0",
                "state": "CLOSED"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) <= 1  # Should skip exit


def test_valid_average_close_price_accepted():
    """CLOSED trade with valid averageClosePrice should create 2 executions"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82043",
                "instrument": "USD_CAD",
                "price": "1.40",
                "averageClosePrice": "1.38",
                "openTime": "1767371240.327566168",
                "closeTime": "1767371340.123456789",
                "initialUnits": "-100000",
                "realizedPL": "100.0",
                "financing": "0.5",
                "state": "CLOSED"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 2  # Entry + exit
    assert df["execution_type"].to_list() == ["OPEN", "CLOSE"]


def test_open_trade_without_average_close_price():
    """OPEN trade doesn't need averageClosePrice"""
    json_content = '''
    {
        "orders": [
            {
                "id": "82044",
                "instrument": "USD_CAD",
                "price": "1.40",
                "openTime": "1767371240.327566168",
                "initialUnits": "100000",
                "state": "OPEN"
            }
        ]
    }
    '''

    df = OandaInterpreter.parse_json_content(json_content)
    assert len(df) == 1  # Entry only
    assert df["execution_type"][0] == "OPEN"
```

---

## Resumen de Impacto

| Validación | Criticidad | Líneas Código | Tests | Effort |
|------------|-----------|---------------|-------|--------|
| 1. Buy/Sell Hash | ⚠️⚠️⚠️ CRÍTICO | ~50 | 10 | 2-3 días |
| 2. Symbol | ⭐⭐⭐ ALTA | ~7 | 3 | 0.25 días |
| 3. Price | ⭐⭐⭐ ALTA | ~15 | 4 | 0.25 días |
| 4. Realized P&L | ⭐⭐ MEDIA-ALTA | ~7 | 3 | 0.25 días |
| 5. Status FILLED | ⭐⭐ MEDIA | ~7 | 4 | 0.25 días |
| 6. Quantity | ⭐⭐ MEDIA | ~7 | 4 | 0.25 días |
| 7. Avg Close Price | ⭐⭐ MEDIA | ~15 | 4 | 0.25 días |
| **TOTAL** | | **~108** | **32** | **3.5-4.5 días** |

**Total Tests:** 32 nuevos tests (actuales: 31, post-cambios: ~63)

**Total Líneas oanda.py:** 308 (actual) → ~416 (post-cambios) = +108 líneas (+35%)

**Total Líneas test_oanda.py:** 430 (actual) → ~600 (post-cambios) = +170 líneas (+40%)

---

## Verificación Final

### Comando de Tests

```bash
# Run all Oanda tests
pytest tests/brokers/test_oanda.py -v

# Run specific test categories
pytest tests/brokers/test_oanda.py::TestOandaBuySelDerivation -v
pytest tests/brokers/test_oanda.py::TestOandaDataValidation -v
pytest tests/brokers/test_oanda.py::TestOandaHashComputation -v

# Run with coverage
pytest tests/brokers/test_oanda.py --cov=pipeline.p01_normalize.brokers.oanda --cov-report=term-missing
```

### Métricas de Éxito

- [ ] Hash match rate >= 95% (vs actual ~0%)
- [ ] Buy/sell derivation correcta para las 4 branches
- [ ] Rejection rate < 0.1%
- [ ] Zero symbols vacíos procesados
- [ ] Zero prices zero/negativo procesados
- [ ] Zero P&L trades correctamente skipped
- [ ] Non-FILLED status correctamente skipped
- [ ] Zero quantities correctamente skipped
- [ ] Closed trades sin averageClosePrice correctamente skipped
- [ ] All 63 tests passing
- [ ] Code coverage >= 90%

---

**Última Actualización:** 2026-01-14
**Broker:** oanda
**Versión:** 1.0
