# Tracking de Implementación - Oanda Normalizer

Este documento rastrea el progreso de implementación de validaciones críticas identificadas en el análisis del Oanda normalizer.

**Fecha de Inicio:** 2026-01-14
**Broker ID:** oanda
**Hash Match Rate Actual:** ~0% ⚠️⚠️⚠️ (EL PEOR DE TODOS)
**Hash Match Rate Objetivo:** 95%+ ✅

---

## FASE 1: Critical Hash Fix (2-3 días) ⚠️⚠️⚠️ URGENTE

**Objetivo:** Corregir hash match rate de ~0% a 95%+

### ⚠️⚠️⚠️ 1. Buy/Sell Field Missing in Hash (CRÍTICO)

**Criticidad:** CRÍTICA (afecta 100% de trades - el peor de todos los brokers)
**Estado:** ❌ PENDIENTE
**Archivo:** `oanda.py` líneas 105-145
**Estimado:** 2-3 días

**Problema:**
- Legacy añade campo 'buy/sell' al order object ANTES de hashear usando complex 4-branch logic
- Nueva hashea order SIN 'buy/sell' field
- Result: 0% hash match rate (vs 42-100% de otros brokers)
- Impacto: TODOS los trades se duplicarían en re-import

**Complex Legacy Logic (4 Branches):**
```python
# oanda_export.py:366-373 - DEBE preservarse EXACTAMENTE
if float(trade['realizedPL'])>0 and float(trade['price']) > float(trade['averageClosePrice']):
    trade['buy/sell']='SELL'   # Profitable short
elif float(trade['realizedPL'])>0 and float(trade['price']) < float(trade['averageClosePrice']):
    trade['buy/sell']='BUY'    # Profitable long
elif float(trade['realizedPL'])<0 and float(trade['price']) > float(trade['averageClosePrice']):
    trade['buy/sell']='BUY'    # Losing long
elif float(trade['realizedPL'])<0 and float(trade['price']) < float(trade['averageClosePrice']):
    trade['buy/sell']='SELL'   # Losing short
```

**Subtareas:**
- [ ] Implementar método `_derive_buy_sell_legacy()` con 4-branch logic (0.5 días)
- [ ] Añadir 'buy/sell' al order object ANTES de hash computation (0.5 días)
- [ ] Modificar hash computation para incluir buy/sell field (0.5 días)
- [ ] Agregar test: test_buy_sell_derivation_profit_sell() (0.25 días)
- [ ] Agregar test: test_buy_sell_derivation_profit_buy() (0.25 días)
- [ ] Agregar test: test_buy_sell_derivation_loss_buy() (0.25 días)
- [ ] Agregar test: test_buy_sell_derivation_loss_sell() (0.25 días)
- [ ] Agregar test: test_hash_includes_buy_sell_field() (0.25 días)
- [ ] Validar con datos reales (0.5 días)

**Tests a Agregar:**
```python
def test_buy_sell_derivation_profit_sell():
    """Profit > 0, entry > exit → SELL"""
    order = {
        'realizedPL': 100.0,
        'price': 1.40,
        'averageClosePrice': 1.38,
        'initialUnits': '-100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'SELL'

def test_buy_sell_derivation_profit_buy():
    """Profit > 0, entry < exit → BUY"""
    order = {
        'realizedPL': 100.0,
        'price': 1.38,
        'averageClosePrice': 1.40,
        'initialUnits': '100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'BUY'

def test_buy_sell_derivation_loss_buy():
    """Loss < 0, entry > exit → BUY"""
    order = {
        'realizedPL': -100.0,
        'price': 1.40,
        'averageClosePrice': 1.38,
        'initialUnits': '100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'BUY'

def test_buy_sell_derivation_loss_sell():
    """Loss < 0, entry < exit → SELL"""
    order = {
        'realizedPL': -100.0,
        'price': 1.38,
        'averageClosePrice': 1.40,
        'initialUnits': '-100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order) == 'SELL'

def test_buy_sell_derivation_fallback():
    """Fallback to initialUnits sign when P&L logic fails"""
    # No realizedPL or averageClosePrice
    order_buy = {
        'initialUnits': '100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order_buy) == 'BUY'

    order_sell = {
        'initialUnits': '-100000',
    }
    assert OandaInterpreter._derive_buy_sell_legacy(order_sell) == 'SELL'

def test_hash_includes_buy_sell_field():
    """Verifica que hash incluye buy/sell field"""
    order = {
        'id': '82041',
        'instrument': 'USD_CAD',
        'realizedPL': 100.0,
        'price': 1.40,
        'averageClosePrice': 1.38,
    }

    # Hash CON buy/sell
    order_with = dict(order)
    order_with['buy/sell'] = 'SELL'
    hash_with = hashlib.md5(json.dumps(order_with).encode()).hexdigest()

    # Hash SIN buy/sell
    hash_without = hashlib.md5(json.dumps(order).encode()).hexdigest()

    # Deben ser DIFERENTES
    assert hash_with != hash_without

def test_hash_same_for_both_executions():
    """Verifica que open y close executions tienen mismo hash"""
    # Parse a closed trade
    order = {
        'id': '82041',
        'instrument': 'USD_CAD',
        'price': '1.37287',
        'openTime': '1767371240.327566168',
        'initialUnits': '-100000',
        'state': 'CLOSED',
        'financing': '-5.50',
        'averageClosePrice': '1.37353',
        'closeTime': '1767386431.471204958',
        'realizedPL': '66.00',
    }

    df = OandaInterpreter.parse_json_content(json.dumps({
        'orders': [order]
    }))

    # Should have 2 executions with SAME file_row hash
    assert df.height == 2
    result = df.collect()
    assert result["file_row"][0] == result["file_row"][1]
```

**Métricas de Éxito:**
- [ ] Hash match rate >= 95% (vs actual ~0%)
- [ ] Buy/sell derivation correcta para las 4 branches
- [ ] Fallback a initialUnits sign funciona correctamente
- [ ] Open y close executions del mismo trade tienen mismo hash
- [ ] Data integrity mantenida (100%)
- [ ] Tests passing: 8 new tests
- [ ] No regressions en tests existentes (31 tests)

**Complejidad:** ALTA (4-branch logic compleja)
**Riesgo:** ALTO (critical para deduplication, formula compleja)

---

## FASE 2: Data Validation (1-1.5 días)

**Objetivo:** Prevenir datos inválidos

### ⭐⭐⭐ 2. Symbol Validation (HIGH)

**Estado:** ❌ PENDIENTE
**Archivo:** `oanda.py` línea ~100
**Estimado:** 0.25 días

**Implementación:**
```python
# oanda.py después de línea 100 (dentro parse_json_content)
instrument = order.get("instrument", "")
if not instrument or not instrument.strip():
    logger.warning(f"[OANDA] Skipping order {order.get('id', 'unknown')}: empty instrument")
    continue
```

**Subtareas:**
- [ ] Agregar validación de instrument vacío
- [ ] Agregar test: test_symbol_empty_validation()
- [ ] Verificar rejection rate < 0.1%

**Test a Agregar:**
```python
def test_symbol_empty_validation():
    """Verifica que symbols vacíos son rechazados"""
    order = {
        'id': '82041',
        'instrument': '',  # Empty instrument
        'price': '1.37287',
        'openTime': '1767371240.327566168',
        'initialUnits': '-100000',
        'state': 'CLOSED',
    }
    df = OandaInterpreter.parse_json_content(json.dumps({
        'orders': [order]
    }))
    assert df.height == 0  # Should be rejected

def test_symbol_whitespace_validation():
    """Verifica que symbols con solo whitespace son rechazados"""
    order = {
        'id': '82041',
        'instrument': '   ',  # Whitespace only
        'price': '1.37287',
        'openTime': '1767371240.327566168',
        'initialUnits': '-100000',
        'state': 'CLOSED',
    }
    df = OandaInterpreter.parse_json_content(json.dumps({
        'orders': [order]
    }))
    assert df.height == 0
```

**Métricas de Éxito:**
- [ ] Zero symbols vacíos en output
- [ ] Warning log generado para cada rechazo
- [ ] Rejection rate < 0.1%

---

### ⭐⭐⭐ 3. Price Validation (HIGH)

**Estado:** ❌ PENDIENTE
**Archivo:** `oanda.py` línea ~114
**Estimado:** 0.25 días

**Implementación:**
```python
# oanda.py después de línea 114
entry_price = float(order.get("price", 0) or 0)
if entry_price <= 0:
    logger.warning(f"[OANDA] Skipping order {order.get('id', 'unknown')}: zero or negative entry price {entry_price}")
    continue

# For closed trades, validate exit price
if state == "CLOSED":
    exit_price = float(order.get("averageClosePrice", 0) or 0)
    if exit_price <= 0:
        logger.warning(f"[OANDA] Skipping closed order {order.get('id', 'unknown')}: zero or negative exit price {exit_price}")
        continue
```

**Subtareas:**
- [ ] Agregar validación de entry price <= 0
- [ ] Agregar validación de exit price <= 0 (para closed trades)
- [ ] Agregar test: test_price_zero_validation()
- [ ] Agregar test: test_price_negative_validation()
- [ ] Agregar test: test_exit_price_zero_validation()
- [ ] Verificar rejection rate < 0.1%

**Tests a Agregar:**
```python
def test_price_zero_validation():
    """Verifica que entry prices zero son rechazados"""
    order = {
        'id': '82041',
        'instrument': 'USD_CAD',
        'price': 0.0,  # Zero price
        'openTime': '1767371240.327566168',
        'initialUnits': '-100000',
        'state': 'CLOSED',
    }
    df = OandaInterpreter.parse_json_content(json.dumps({'orders': [order]}))
    assert df.height == 0

def test_price_negative_validation():
    """Verifica que prices negativos son rechazados"""
    order = {
        'id': '82041',
        'instrument': 'USD_CAD',
        'price': -1.37287,  # Negative price
        'openTime': '1767371240.327566168',
        'initialUnits': '-100000',
        'state': 'CLOSED',
    }
    df = OandaInterpreter.parse_json_content(json.dumps({'orders': [order]}))
    assert df.height == 0

def test_exit_price_zero_validation():
    """Verifica que exit prices zero son rechazados para closed trades"""
    order = {
        'id': '82041',
        'instrument': 'USD_CAD',
        'price': '1.37287',
        'openTime': '1767371240.327566168',
        'initialUnits': '-100000',
        'state': 'CLOSED',
        'averageClosePrice': 0.0,  # Zero exit price
        'closeTime': '1767386431.471204958',
    }
    df = OandaInterpreter.parse_json_content(json.dumps({'orders': [order]}))
    assert df.height == 0  # Should reject entire trade
```

**Métricas de Éxito:**
- [ ] Zero entry prices zero/negativos en output
- [ ] Zero exit prices zero/negativos en output
- [ ] Warning log generado para cada rechazo
- [ ] Rejection rate < 0.1%

---

### ⭐⭐ 4. Realized P&L Zero Skip (MEDIUM-HIGH)

**Estado:** ❌ PENDIENTE
**Archivo:** `oanda.py` línea ~95
**Estimado:** 0.25 días

**Implementación:**
```python
# oanda.py después de línea 95 (early in loop)
realized_pl = float(order.get("realizedPL", 0) or 0)
if realized_pl == 0:
    logger.info(f"[OANDA] Skipping order {order.get('id', 'unknown')}: zero realized P&L")
    continue
```

**Subtareas:**
- [ ] Agregar validación de realized P&L == 0
- [ ] Agregar test: test_realized_pl_zero_skip()
- [ ] Verificar rejection rate < 0.1%

**Test a Agregar:**
```python
def test_realized_pl_zero_skip():
    """Verifica que trades con realizedPL=0 son skipped"""
    order = {
        'id': '82041',
        'instrument': 'USD_CAD',
        'price': '1.37287',
        'openTime': '1767371240.327566168',
        'initialUnits': '-100000',
        'state': 'CLOSED',
        'realizedPL': 0.0,  # Zero P&L
        'averageClosePrice': '1.37287',  # Same as entry
        'closeTime': '1767386431.471204958',
    }
    df = OandaInterpreter.parse_json_content(json.dumps({'orders': [order]}))
    assert df.height == 0  # Should be skipped
```

**Métricas de Éxito:**
- [ ] Zero P&L trades skipped
- [ ] Info log generado para cada skip
- [ ] Rejection rate < 0.1%

---

### ⭐⭐ 5. Status FILLED Validation (MEDIUM)

**Estado:** ❌ PENDIENTE
**Archivo:** `oanda.py` línea ~95
**Estimado:** 0.25 días

**Implementación:**
```python
# oanda.py después de línea 95
status = order.get("status", "")
if status and status != "FILLED":
    logger.info(f"[OANDA] Skipping order {order.get('id', 'unknown')}: status {status} (not FILLED)")
    continue
```

**Subtareas:**
- [ ] Agregar validación de status != "FILLED"
- [ ] Agregar test: test_status_not_filled_skip()
- [ ] Verificar rejection rate < 0.1%

**Test a Agregar:**
```python
def test_status_not_filled_skip():
    """Verifica que orders con status != FILLED son skipped"""
    statuses = ['PENDING', 'CANCELLED', 'REJECTED']

    for status_val in statuses:
        order = {
            'id': '82041',
            'instrument': 'USD_CAD',
            'price': '1.37287',
            'openTime': '1767371240.327566168',
            'initialUnits': '-100000',
            'state': 'CLOSED',
            'status': status_val,  # Non-FILLED status
        }
        df = OandaInterpreter.parse_json_content(json.dumps({'orders': [order]}))
        assert df.height == 0, f"Status {status_val} should be skipped"
```

**Métricas de Éxito:**
- [ ] Non-FILLED status skipped
- [ ] Info log generado para cada skip
- [ ] Rejection rate < 0.1%

---

### ⭐⭐ 6. Quantity Validation (MEDIUM)

**Estado:** ❌ PENDIENTE
**Archivo:** `oanda.py` línea ~111
**Estimado:** 0.25 días

**Implementación:**
```python
# oanda.py después de línea 111
initial_units = float(order.get("initialUnits", 0) or 0)
if initial_units == 0:
    logger.warning(f"[OANDA] Skipping order {order.get('id', 'unknown')}: zero initial units")
    continue
```

**Subtareas:**
- [ ] Agregar validación de initialUnits == 0
- [ ] Agregar test: test_quantity_zero_validation()
- [ ] Verificar rejection rate < 0.1%

**Test a Agregar:**
```python
def test_quantity_zero_validation():
    """Verifica que quantities zero son rechazados"""
    order = {
        'id': '82041',
        'instrument': 'USD_CAD',
        'price': '1.37287',
        'openTime': '1767371240.327566168',
        'initialUnits': 0,  # Zero quantity
        'state': 'CLOSED',
    }
    df = OandaInterpreter.parse_json_content(json.dumps({'orders': [order]}))
    assert df.height == 0
```

**Métricas de Éxito:**
- [ ] Zero quantities rechazados
- [ ] Warning log generado para cada rechazo
- [ ] Rejection rate < 0.1%

---

### ⭐⭐ 7. Average Close Price Validation (MEDIUM)

**Estado:** ❌ PENDIENTE
**Archivo:** `oanda.py` línea ~147
**Estimado:** 0.25 días

**Implementación:**
```python
# oanda.py línea ~147 (before creating exit record)
if state == "CLOSED":
    avg_close_price = order.get("averageClosePrice")
    if not avg_close_price:
        logger.warning(f"[OANDA] Skipping closed order {order.get('id', 'unknown')}: missing averageClosePrice")
        continue
```

**Subtareas:**
- [ ] Agregar validación de averageClosePrice missing
- [ ] Agregar test: test_avg_close_price_missing_validation()
- [ ] Verificar rejection rate < 0.1%

**Test a Agregar:**
```python
def test_avg_close_price_missing_validation():
    """Verifica que closed trades sin averageClosePrice son rechazados"""
    order = {
        'id': '82041',
        'instrument': 'USD_CAD',
        'price': '1.37287',
        'openTime': '1767371240.327566168',
        'initialUnits': '-100000',
        'state': 'CLOSED',
        # averageClosePrice MISSING
        'closeTime': '1767386431.471204958',
    }
    df = OandaInterpreter.parse_json_content(json.dumps({'orders': [order]}))
    assert df.height == 0
```

**Métricas de Éxito:**
- [ ] Closed trades sin averageClosePrice rechazados
- [ ] Warning log generado para cada rechazo
- [ ] Rejection rate < 0.1%

---

## FASE 3: CSV Support (8-12 días - SI NECESARIO) 🎯

**Objetivo:** Implementar soporte para 3 formatos CSV legacy

### 🎯 8. CSV Parser Support (CONDICIONAL)

**Estado:** ⏸️ BLOQUEADO (requiere SQL query)
**Complejidad:** MUY ALTA (3 formats, auto-detection, ~636 líneas)
**Estimado:** 8-12 días (solo si necesario)

**Decisión Requerida:** Ejecutar SQL query para determinar uso

```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*) OVER (), 2) as percentage,
    MAX(created_at) as last_used
FROM data_sources
WHERE broker_id = 'oanda'
  AND created_at > NOW() - INTERVAL '12 months'
GROUP BY source_type
ORDER BY count DESC;
```

**Criterio:**
- Si CSV percentage > 5%: **IMPLEMENTAR** (continuar con Fase 3)
- Si CSV percentage < 5%: **OMITIR** (skip Fase 3)

**Legacy CSV Features (Scope si se implementa):**

**CSV Format 1 (getcsv) - Lines 1-253:**
- [ ] Format detection (detect "DATE" or "transaction" header)
- [ ] Transaction type validation (must be ORDER_FILL)
- [ ] Side mapping (BUY/SELL, COVER/SHORT)
- [ ] Symbol normalization (remove '/', add '$')
- [ ] Forex detection via '/' in symbol
- [ ] Price decimal calculation
- [ ] Commission/fee extraction (spread cost)
- [ ] Pip value lookup by currency pair
- [ ] Date parsing (flexible format)

**CSV Format 2 (getcsv2) - Lines 257-489:**
- [ ] Format detection (detect "Open Date" and "Close Date")
- [ ] Open/close on same row processing
- [ ] Creates 2 executions per row (entry + exit)
- [ ] Price field priority (entry price → exit price)
- [ ] Symbol numeric check (reject if numeric)
- [ ] Commission field variants (commission vs commissions)
- [ ] Financing/swap field extraction
- [ ] Dividend handling
- [ ] Symbol suffix with row index (e.g., "USDCAD|1")

**CSV Format 3 (getcsv3) - Lines 491-612:**
- [ ] Japanese format detection (detect '売買' character)
- [ ] Different field names (quantity, not amount)
- [ ] Action mapping (BUY/SELL/COVER/SHORT/B/S)
- [ ] Character encoding handling
- [ ] Quantity numeric string validation
- [ ] Symbol formatting (remove special chars)

**Auto-Detection Logic (Lines 70-76):**
- [ ] Implement format detection hierarchy:
  1. If '売買' in content → Format 3 (Japanese)
  2. If 'Open Date' and 'Close Date' → Format 2
  3. Default → Format 1
- [ ] Fallback from Format 1 to Format 2 if no results

**Subtareas (solo si se implementa):**
- [ ] Crear CSVFormat1Interpreter class (3 días)
- [ ] Crear CSVFormat2Interpreter class (3 días)
- [ ] Crear CSVFormat3Interpreter class (2 días)
- [ ] Agregar format detection en detector.py (0.5 días)
- [ ] Implement auto-detection hierarchy (0.5 días)
- [ ] Tests comprehensivos para 3 formats (2 días)
- [ ] Documentación completa (0.5 días)
- [ ] Integration testing (0.5 días)

**Métricas de Éxito (si se implementa):**
- [ ] CSV format 1 correctamente parseado
- [ ] CSV format 2 correctamente parseado
- [ ] CSV format 3 correctamente parseado
- [ ] Auto-detection funciona correctamente
- [ ] Hash match rate >= 95% para CSVs
- [ ] Tests passing para 3 formats
- [ ] Documentación actualizada

**Estado:** ⏸️ ESPERANDO DECISIÓN (ejecutar SQL query primero)

---

## Validaciones OUT OF SCOPE ✅

Estas validaciones pertenecen a otros pipeline stages y están correctamente excluidas:

### ✅ 9. Credential Validation
**Estado:** OUT OF SCOPE (sync service)
**Ubicación Legacy:** oanda_export.py:238-277, oandav2_export.py:165-214
**Sin acción requerida**

### ✅ 10. Login/Session Management
**Estado:** OUT OF SCOPE (sync service)
**Ubicación Legacy:** oanda_export.py:278-294, oandav2_export.py:215-233
**Sin acción requerida**

### ✅ 11. API Rate Limiting
**Estado:** OUT OF SCOPE (sync service)
**Ubicación Legacy:** Implicit in pagination logic
**Sin acción requerida**

### ✅ 12. Pagination Logic
**Estado:** OUT OF SCOPE (sync service)
**Ubicación Legacy:** oanda_export.py:329, oandav2_export.py:240
**Sin acción requerida**

### ✅ 13. File Saving/Archiving
**Estado:** OUT OF SCOPE (sync service)
**Ubicación Legacy:** oanda_export.py:565-625
**Sin acción requerida**

### ✅ 14. Database Deduplication
**Estado:** OUT OF SCOPE (p05_write stage)
**Ubicación Legacy:** brokers_oanda.py:614-637
**Sin acción requerida**

---

## Validaciones Ya Implementadas ✅

Estas validaciones ya están implementadas correctamente:

### ✅ 15. JSON Validation
**Estado:** COMPLETO (parse_json_content validates JSON structure)
**Sin cambios requeridos**

### ✅ 16. Timestamp Conversion
**Estado:** COMPLETO (oanda.py:187-205)
**Sin cambios requeridos**

### ✅ 17. Symbol Normalization
**Estado:** COMPLETO (oanda.py:237)
**Sin cambios requeridos**

### ✅ 18. Trade Splitting
**Estado:** COMPLETO (oanda.py:146-169)
**Sin cambios requeridos**

### ✅ 19. Asset Type Classification
**Estado:** COMPLETO (oanda.py:265)
**Sin cambios requeridos**

### ✅ 20. Commission/Fee Handling
**Estado:** COMPLETO (oanda.py:252-258)
**Sin cambios requeridos**

---

## Resumen de Progreso

### Fase 1: Critical Hash Fix
- **Total:** 1 validación
- **Completadas:** 0
- **Pendientes:** 1
- **Progreso:** 0%

### Fase 2: Data Validation
- **Total:** 6 validaciones
- **Completadas:** 0
- **Pendientes:** 6
- **Progreso:** 0%

### Fase 3: CSV Support
- **Total:** 1 validación condicional
- **Bloqueadas:** 1 (requiere SQL query)
- **Progreso:** 0%

### Out of Scope + Ya Implementadas
- **Total:** 12 validaciones
- **Completadas:** 12
- **Progreso:** 100%

---

## Métricas de Calidad

### Objetivos
- [ ] Hash match rate >= 95% (actual: ~0%)
- [ ] Data rejection rate < 0.1%
- [ ] Zero symbols vacíos en output
- [ ] Zero prices zero/negativos en output
- [ ] Zero quantities zero en output
- [ ] Zero P&L trades skipped
- [ ] Non-FILLED status skipped
- [ ] Test coverage >= 95%
- [ ] All tests passing

### Status Actual
- ⚠️⚠️⚠️ Hash match rate: ~0% (objetivo: 95%) - **CRÍTICO**
- ⚠️ Data rejection rate: unknown (objetivo: < 0.1%)
- ✅ Test coverage: 430 líneas (comprehensive)
- ✅ Tests passing: 31/31 (100%)

---

## Próximos Pasos

### Inmediato (Esta Semana)
1. **Fase 1 (URGENTE):** Implementar buy/sell hash fix (2-3 días)
   - Esta es la prioridad #1 más alta de todos los brokers
   - 0% hash match rate es el peor escenario posible

2. **Fase 2:** Implementar data validations (1-1.5 días)
   - Symbol, price, quantity, P&L, status validations

### Corto Plazo (Próximas 2 Semanas)
3. **Decisión CSV:** Ejecutar SQL query para determinar necessity
4. **Fase 3 (Condicional):** Implementar CSV support si es necesario (8-12 días)

### Validación Final
5. Ejecutar integration tests con datos reales
6. Validar hash match rate >= 95%
7. Validar rejection rate < 0.1%
8. Code review completo
9. Deploy a producción

---

**Última Actualización:** 2026-01-14
**Próxima Revisión:** TBD
**Responsable:** Development Team
