
import itertools
import traceback
import time
import json
import re
import hashlib
from datetime import datetime, timedelta
import requests
from flask_restful import Resource, request
from app.helpers import Getters

from dateutil.parser import parse
from ..security_service import verify_role
from ..responses import ErrorResponses
from ..import_params import ImportParams
from ..trade.trades_regroups import TradeRegroups
from ..server_info import ServerInfo
from ..event.events import UserEventList as EventUser
from .brokers_export import BrokerExport
from app.models import UserEvents

class KucoinExport(BrokerExport,ServerInfo, Resource):
    """Handles the import of trading data from Kucoin exchange. Supports multiple account types (Futures, Spot, and Margin)."""
    
    def __init__(self):
        BrokerExport.__init__(self)
    
    @verify_role('broker export')
    def post(self):
        """
        Processes and handles the POST request for importing Kucoin trading data.
        This method handles the import of trading data from Kucoin exchange, supporting multiple
        account types (Futures, Spot, and Margin). It performs authentication, data retrieval,
        and storage of trading information.
        The process includes:
        1. Initializing synchronization
        2. Validating credentials
        3. Saving account information
        4. Retrieving orders from different account types (Future/Spot/Margin)
        5. Interpreting and saving orders
        6. Updating configurations and sync status
        The method can also handle AWS Lambda execution if configured.
        Returns:
            dict: Response containing sync status and any relevant messages
            - On success: Returns sync completion status
            - On failure: Returns error response with 500 status code
        Raises:
            Exception: Any unexpected errors during the import process
        Progress Updates:
            - Pre-Data: 1%
            - Credentials Check: 5%
            - Account Saving: 8-9%
            - Server Connection: 15%
            - Login: 16%
            - Order Retrieval: 18%
        """
        
        try:
            version = {}
            self.orders_execid = []
            error_future = False
            error_spot = False
            error_margin = False
            self.data = request.json
            self._update_progress('Saving Pre-Data', '1%')
            # ini syncronization
            new_event_id=self.init_sync()
            if not isinstance(new_event_id,UserEvents) and not self.is_lambda():
                return new_event_id
            for credential in self.credentials:
                self._update_progress('Checking Credentials', '5%')
                self.credential = credential
                self.not_error = self.validate()
                if self.not_error is not True:
                    _end = self.end_sync()
                    if _end is not True:
                        return _end
                self.account_pusher = f'Account #{self.api_key_information}'
                self.symbols = []
                self.time_out_lambda=False
                self.executed_lambda = False
                if self.condition_lambda() and not self.aws_autosync:
                    try:
                        self.executed_lambda = True
                        response = self.call_lambda()
                    except Exception as err:
                        self.not_error = traceback.format_exc()
                        self.executed_lambda = False
                
                if self.executed_lambda is False:
                    self._update_progress('Connecting to the Kucoin server', '10%')
                    self.user.users_configs.save()
                    if self.future1:
                        self._update_progress('Login future', '15%')
                        self.not_error = self.login_future()
                        if self.not_error is not True:
                            self.future1 = False
                            error_future = self.not_error
                        else:
                            error_future = True
                            self._update_progress('Saving Account future', '16%')
                            self.save_account(self.api_key_information,category='Perpetual')
                            self._update_progress('Retrieving Future orders future', '18%')
                            self.not_error=self.get_kucoin_futures_orders()
                            if self.not_error is not True:
                                _end = self.end_sync(code=7)
                                if _end is not True:
                                    return _end
                            if len( self.all_orders) > 0:
                                self.save_files(file_type='json')
                                self.interpret_orders()
                                self._update_progress('Saving Future orders', '19%')
                                self.save_orders()
                            self.not_error = self.save_params("Perpetual")
                    if self.spot:
                        self._update_progress('Login Spot', '15%')  
                        self.not_error = self.login_spot()
                        if self.not_error is not True:
                            self.spot = False
                            error_spot = self.not_error
                        else:
                            error_spot = True
                            self._update_progress('Saving Account Spot', '16%')
                            self.save_account(self.api_key_information,category='Spot')
                            self._update_progress('Retrieving Spot orders', '18%')
                            self.not_error=self.get_kucoin_orders()
                            if self.not_error is not True:
                                _end = self.end_sync(code=7)
                                if _end is not True:
                                    return _end
                            if len(self.orders) > 0:
                                self.save_files(file_type='json')
                                self.interpret_orders()
                                self._update_progress('Saving Spot orders', '19%')
                                self.save_orders()
                            self.not_error = self.save_params("Spot")

                        #self.not_error=self.get_kucoin_margin_orders()
                    if self.margin1:
                        self._update_progress('Login Margin', '15%')
                        self.not_error = self.login_margin()
                        if self.not_error is not True:
                            self.margin1 = False
                            error_margin = self.not_error
                        else:
                            error_margin = True
                            self._update_progress('Saving Account Margin', '9%')
                            self.save_account(self.api_key_information,category='Margin')
                            self._update_progress('Retrieving Margin orders', '18%')
                            self.not_error=self.get_kucoin_margin_orders()
                            if self.not_error is not True:
                                _end = self.end_sync(code=7)
                                if _end is not True:
                                    return _end
                            if len(self.orders) > 0:
                                self.save_files(file_type='json')
                                self.interpret_orders()
                                self._update_progress('Saving Margin orders', '19%')
                                self.save_orders()
                            self.not_error = self.save_params("Margin")
                    if error_future is not True and error_spot is not True and error_margin is not True:
                        self.not_error = error_future
                        _end = self.end_sync(code=1)
                        if _end is not True:
                            return _end
                        
                    self._update_progress('Saving data from Kucoin', '16%')
                    self.save_configs()
                    
            return self.end_sync(event_id=new_event_id)

        except Exception as err:
            #UserParams.append_param(self.user.id,'IMPORT_UPLOAD',False,True)
            self.unlock_portfolios()
            self._update_progress('warning', 'Imported Fail', True)
            if self.is_lambda() and self.can_call_lambda():      
                self.save_params_lambda((int(time.time())-self.star_timestamp), 500, 'Fail')
            return ErrorResponses.error_500(self.data, err, {})

    def validate(self):
        """
        Validates user credentials and trading settings for Kucoin broker integration.
        This method processes and validates user credentials and trading preferences,
        including API keys, account types, and sync settings. It handles both new
        and existing credentials.
        Returns:
            bool or str: Returns True if validation is successful, 
                        'select_type' if no trading type is selected,
                        'Invalid credentials' if credentials are missing,
                        'Wrong credentials' if there's an error with existing credentials
        Raises:
            Exception: Handles various validation errors, falling back to existing credentials
                       if available
        Notes:
            - Updates user configuration for import warnings
            - Supports spot, futures, and margin trading types
            - Handles both test and live accounts
            - Processes API key, secret key, and passphrase
            - Manages new and existing credential scenarios
        """
        
        try:
            #self.perpetual=False
            #self.spot=False
            #self.margin=False
            self.data = request.json
            self.user.users_configs.not_warning_import = self.data.get('not_warning_import',True)
            self.user.users_configs.save()
            self.symbols = self.data.get('symbols',self.symbols)
            self.auto_import = self.data['auto_import'] if 'auto_import' in self.data else False
            #try:
            #    if not self.credential:
            #        self.credential = BrokersConnections.find_by(**{'user_id':self.user.id, "broker_id":self.broker.broker_id, "api_key":self.data['api_key'], "secret_key":self.data['secret_key'], "active": True})
            #except Exception as err:
            #    #print(err)
            #    pass
             
            if self.credential and self.add is False:
                #self.credential = BrokersConnections.find_by(**{'user_id':self.user.id, "broker_id":self.broker.broker_id, 'active': True})
                self.api_key = self.credential.api_key
                self.secret_key = self.credential.secret_key
                self.test_account =  self.credential.test_account
                self.passphrase =  self.credential.passphrase
                self.spot =  self.credential.spot
                self.future1 =  self.credential.future1
                self.margin1= self.credential.margin1
                self.api_key_information=self.credential.api_key
                self.new_sync =  self.credential.new_sync
            else:
                if self.data['api_key'] =='' or self.data['secret_key'] == '' or self.data['test_account']=='':
                    return "Invalid credentials"

                #self.credential = BrokersConnections.find_by(**{'user_id':self.user.id, "broker_id":self.broker.broker_id, 'active': True})
                self.api_key = self.data['api_key'].strip()
                self.api_key_information=self.data['api_key'].strip()
                self.secret_key = self.data['secret_key'].strip()
                self.test_account =  self.data['test_account'] if 'test_account' in self.data and self.data['test_account'] != "" else False
                self.passphrase =  self.data.get('passphrase',None)
                self.spot =  self.data['spot'] if 'spot' in self.data and self.data['spot'] != "" else False
                self.future1 =  self.data['future1'] if 'future1' in self.data and self.data['future1'] != "" else False
                self.margin1= self.data['margin1'] if 'margin1' in self.data and self.data['margin1'] != "" else False
                self.new_sync = True
            if not any([self.spot, self.future1, self.margin1]): 
                return 'select_type'

            return True

        except Exception as err:
            if not self.credential:
                return "Wrong credentials"

            self.api_key = self.credential.api_key
            self.secret_key = self.credential.secret_key
            self.test_account =  self.credential.test_account
            self.passphrase =  self.credential.passphrase
            self.api_key_information=self.credential.account
            self.user.users_configs.not_warning_import = True
            self.user.users_configs.save()
            self.data = None

    def login_spot(self):
        """
        Logs in to Kucoin Spot trading API and retrieves order data.
        This method attempts to authenticate with Kucoin's API using provided credentials 
        and retrieves spot trading orders through a Lambda function endpoint.
        Returns:
            Union[bool, str]: Returns True if login and data retrieval is successful.
                              Returns error message string if an exception occurs.
                              Returns order data string if orders are returned as string.
        Raises:
            Exception: Any exception during the API request is caught and returned as string.
        Notes:
            - Uses AWS Lambda endpoint for API communication
            - Requires api_key, secret_key and passphrase to be set before calling
            - Sets account name as "KucoinSpot" + api_key
        """
        
        #print(self.api_key, self.secret_key, self.passphrase, self.test_account)
        #self.account = "KucoinSpot"+self.user.uuid
        self.account = "KucoinSpot"+self.api_key
        try:
            url = "https://pfdby3vxyfaqggxfwjsxga5fwa0epynu.lambda-url.eu-central-1.on.aws/"
            headers = {
            'Content-Type': 'application/json'
            }
            payload = json.dumps({
                "api_key": self.api_key, 
                "secret_key": self.secret_key,
                "passphrase": self.passphrase,
                "partial": self.partial,
                "last_date": datetime.now().timestamp(),
                "action":"get_orders_spot", #get_orders_spot or get_orders_margin or get_orders_future
            })
            response = requests.request("POST", url, headers=headers, data=payload) 
            if response.status_code == 200:
                data = response.json()
                if 'orders' in data:
                    if isinstance(data['orders'], str):
                        return data['orders']
                return True
        
        except Exception as err:
            return str(err)
            
        return True

    def login_future(self):
        """
        Authenticates and retrieves future orders from Kucoin exchange via AWS Lambda endpoint.
        This method sends a POST request to an AWS Lambda endpoint with the user's API credentials
        to fetch their futures trading orders.
        Returns:
            Union[bool, str]: Returns True if authentication and order retrieval is successful.
                              Returns error message string if orders data contains error message.
                              Returns error string if exception occurs during the process.
        Raises:
            Exception: Any exception that occurs during the API request process is caught
                       and returned as a string.
        Note:
            - The method uses class instance variables for API credentials (api_key, secret_key, passphrase)
            - Sets account name with prefix "KucoinPerpetual" followed by api_key
            - The endpoint expects a specific payload format including api credentials and timestamp
        """
        
        #print(self.api_key, self.secret_key, self.passphrase, self.test_account)
        #self.account = "KucoinFuture"+self.user.uuid
        self.account = "KucoinPerpetual"+self.api_key
        try:
            url = "https://pfdby3vxyfaqggxfwjsxga5fwa0epynu.lambda-url.eu-central-1.on.aws/"
            headers = {
            'Content-Type': 'application/json'
            }
            payload = json.dumps({
                "api_key": self.api_key, 
                "secret_key": self.secret_key,
                "passphrase": self.passphrase,
                "partial": self.partial,
                "last_date": int(datetime.now().timestamp()),
                "action":"get_orders_future", #get_orders_spot or get_orders_margin or get_orders_future
            })
            response = requests.request("POST", url, headers=headers, data=payload)  
            if response.status_code == 200:
                data = response.json()
                if 'orders' in data:
                    if isinstance(data['orders'], str):
                        return data['orders']
                return True
                
        except Exception as err:
            return str(err)
        return True

    def login_margin(self):
        """
        Attempts to login to Kucoin margin trading account and retrieve orders.
        This method makes a POST request to an AWS Lambda endpoint with the user's API credentials
        to authenticate and fetch margin trading orders.
        Returns:
            Union[bool, str, Exception]: 
                - True if login successful but no orders returned
                - String containing orders if orders were returned as string
                - Exception if an error occurred during the request
                - True if none of the above conditions are met
        Raises:
            May raise requests.exceptions.RequestException if the HTTP request fails
        """
        #print(self.api_key, self.secret_key, self.passphrase, self.test_account)
        #self.account = "KucoinSpot"+self.user.uuid
        self.account = "KucoinSpot"+self.api_key
        try:
            url = "https://pfdby3vxyfaqggxfwjsxga5fwa0epynu.lambda-url.eu-central-1.on.aws/"
            headers = {
            'Content-Type': 'application/json'
            }
            payload = json.dumps({
                "api_key": self.api_key, 
                "secret_key": self.secret_key,
                "passphrase": self.passphrase,
                "partial": self.partial,
                "last_date": datetime.now().timestamp(),
                "action":"get_orders_margin", #get_orders_spot or get_orders_margin or get_orders_future
            })
            response = requests.request("POST", url, headers=headers, data=payload)
            #print('response', response.status_code, response.json())
            if response.status_code == 200:
                data = response.json()
                if 'orders' in data:
                    if isinstance(data['orders'], str):
                        return data['orders']
                return True
        
        except Exception as err:
            return err
            
        return True
    
    def get_kucoin_orders(self):
        """
        Retrieves Kucoin spot trading orders for a given timeframe.
        This method fetches spot trading orders from Kucoin exchange by making API calls to a Lambda endpoint.
        It handles pagination and date filtering to get historical orders up to a specified end date.
        The method updates order data incrementally in weekly chunks (604800 seconds) and stores them
        in the instance's orders and all_orders attributes.
        Returns:
            bool: True if orders were successfully retrieved
            str: Error message if the API request failed
        Side effects:
            - Updates self.orders with retrieved order data
            - Updates self.all_orders with retrieved order data 
            - Updates self.last_date with timestamp of most recent order
            - Updates user events through EventUser class
            - Shows progress updates through self._update_progress()
        Attributes used:
            self.api_key: Kucoin API key
            self.secret_key: Kucoin secret key
            self.passphrase: Kucoin API passphrase
            self.partial: Boolean flag for partial data retrieval
            self.params: Dictionary containing broker information
        Note:
            - Uses a Lambda function endpoint for actual API communication
            - Implements retry logic with max 12 attempts for empty responses
            - Handles both full historical and incremental (partial) data retrieval
            - Processes timestamps in milliseconds from Kucoin API
        """
        
        order={}
        orders= []
        #print('Spot')
        get_orders_event = EventUser.post_user_events(100, new_user=self.user_id)
        self.orders=[]
        present_date = datetime.now().timestamp() + 86400
        j = 1
        attempts=0
        paginated = True
        i = 1
        self.params['broker']='Kucoin'
        last_date = self.get_last_date_a(category='Spot', user_account=self.api_key_information)  - (86400/24)  # una hora menos

        end_at = 1672631574   if self.partial is False and last_date <= 0 else last_date
        attempts = 0
        while present_date > end_at:
            j += 1
            self._update_progress(f"Retrieving Spot orders ({datetime.fromtimestamp(present_date).strftime('%Y-%m-%d')})", '18%')

            url = "https://pfdby3vxyfaqggxfwjsxga5fwa0epynu.lambda-url.eu-central-1.on.aws/"
            headers = {
            'Content-Type': 'application/json'
            }
            payload = json.dumps({
                "api_key": self.api_key, 
                "secret_key": self.secret_key,
                "passphrase": self.passphrase,
                "partial": self.partial,
                "last_date": present_date,
                "action":"get_orders_spot", #get_orders_spot or get_orders_margin or get_orders_future
            })
            response = requests.request("POST", url, headers=headers, data=payload) 
            if response.status_code == 200:            
                present_date = present_date + 604800
                data = response.json()
                if 'orders' in data:
                    if isinstance(data['orders'], str):
                        return data['orders']
                    if len(data['orders']) > 0:
                        if self.partial is True and 'created_at' in data['orders'][0]:
                            data['orders'] = [order for order in data['orders'] if order['created_at']/1000 >= last_date]
                        orders.extend(data['orders'] )
                        attempts = 0
                    else:
                        attempts += 1
                else:
                    attempts += 1
            else:
                return response.text
            if attempts > 12:  # 12 attempts of 7 days
                break
        self.orders = orders
        self.all_orders = self.orders
        try:
            last_date = self.all_orders[0]['created_at']
            self.last_date = last_date/1000
        except Exception:
            pass
        
        if get_orders_event:
            EventUser.update_user_events(get_orders_event)
        return True

    def get_kucoin_margin_orders(self):
        """
        Retrieves margin orders from Kucoin exchange for a specified time period.
        This method fetches isolated margin trade orders from Kucoin exchange. It retrieves orders
        in weekly chunks starting from a specified end date up to the present date. Orders are 
        paginated with 500 orders per page.
        Returns:
            bool: True if orders were successfully retrieved, 'API KEY not exists' if API authentication fails
        Attributes modified:
            self.margin (bool): Set to True if margin orders are found
            self.all_orders (list): Appended with retrieved margin orders
            self.last_date (int): Updated with timestamp of most recent order
        Side effects:
            - Updates progress indicator with current retrieval status
            - Updates user events through EventUser
            - Introduces delay between API calls (0.333s per page, 1s on error)
        Notes:
            - Orders are retrieved in descending chronological order
            - Timestamps are converted from milliseconds to seconds where needed
            - Orders are enriched with 'category' and formatted creation date
        """
        order={}
        margin_orders=[]
        #print('Margin')
        get_orders_event = EventUser.post_user_events(100, new_user=self.user_id)
        orders = []
        present_date = int(datetime.now().timestamp())
        # for status in ['active', 'done']:
        #self.pusher_title =  'Finding active orders'
        #self.pusher_message = '18%'
        #self.send_pusher()
        j = 1
        k = 1
        attempts = 0
        for status in ['done']:
            if not self.exclude_future_kucoin():
                self.params['broker']='Kucoin'
            last_date = self.get_last_date_a(category='Margin', user_account=self.api_key_information)  - (86400/24)  # una hora menos

            end_at = 1673236374  if self.partial is False or (not last_date or last_date<=0) else last_date
            while present_date > end_at:
                orders = []
                self._update_progress(f"Retrieving Margin orders ({datetime.fromtimestamp(present_date).strftime('%Y-%m-%d')})", '18%')
                url = "https://pfdby3vxyfaqggxfwjsxga5fwa0epynu.lambda-url.eu-central-1.on.aws/"
                headers = {
                'Content-Type': 'application/json'
                }
                payload = json.dumps({
                    "api_key": self.api_key, 
                    "secret_key": self.secret_key,
                    "passphrase": self.passphrase,
                    "partial": self.partial,
                    "last_date": present_date,
                    "action":"get_orders_margin", #get_orders_spot or get_orders_margin or get_orders_future
                })
                response = requests.request("POST", url, headers=headers, data=payload)
                print('response', response.status_code, response.json()) 
                if response.status_code == 200:            
                    present_date =present_date - 604800
                    data = response.json()
                    if 'orders' in data:
                        if isinstance(data['orders'], str):
                            return data['orders']
                        if len(data['orders']) > 0:
                            attempts = 0
                            if self.partial is True:
                                data['orders'] = [order for order in data['orders'] if order['created_at']/1000 >= last_date]
                                #_data = [order for order in data['orders'] if order['created_at'] >= last_date*1000]
                            margin_orders.extend(data['orders'])
                        else:
                            attempts += 1                        
                    else:
                        attempts += 1
                else:
                    return response.text
                if attempts > 2:  #2 attempts of 7 days
                    break
        self.all_orders.extend(margin_orders)
        self.orders= self.all_orders
        try:
            last_date = self.all_orders[0]['created_at']
            self.last_date = last_date/1000
        except Exception:
            pass

        if get_orders_event:
            EventUser.update_user_events(get_orders_event)
        
        return True

    def get_kucoin_futures_orders(self):
        """
        Retrieves Kucoin futures orders for the authenticated user.
        This method fetches futures trading orders from Kucoin exchange by making requests to
        a Lambda endpoint. It processes orders in weekly chunks, working backwards from the
        present date until either reaching the end date or hitting the maximum attempts.
        Returns:
            bool: True if orders were successfully retrieved
            str: Error message if the API request fails
        Attributes modified:
            self.all_orders: Extended with retrieved future orders
            self.orders: Set to all retrieved orders
            self.last_date: Updated with timestamp of most recent order
        Notes:
            - Uses Lambda endpoint for actual API calls to Kucoin
            - Processes orders in 7-day (604800 seconds) chunks
            - Makes up to 12 attempts to retrieve orders
            - Handles partial syncs by filtering orders after last sync date
            - Updates progress via self._update_progress method
        """
        
        order={}
        future_orders=[]
        #print('Futures')
        get_orders_event = EventUser.post_user_events(100, new_user=self.user_id)
        orders = []
        present_date = int(datetime.now().timestamp()) + 86400
        # for status in ['active', 'done']:
        #self.pusher_title =  'Finding active orders'
        #self.pusher_message = '18%'
        #self.send_pusher()
        j = 1
        k = 1
        attempts = 0
        for status in ['done']:
            if not self.exclude_future_kucoin():
                self.params['broker']='FutureKucoin'
            last_date = self.get_last_date_a(category='Perpetual', user_account=self.api_key_information)  - (86400/24)  # una hora menos

            end_at = 1673236374  if self.partial is False or (not last_date or last_date<=0) else last_date
            while present_date > end_at:
                orders = []
                self._update_progress(f"Retrieving Future orders ({datetime.fromtimestamp(present_date).strftime('%Y-%m-%d')})", '18%')
                url = "https://pfdby3vxyfaqggxfwjsxga5fwa0epynu.lambda-url.eu-central-1.on.aws/"
                headers = {
                'Content-Type': 'application/json'
                }
                payload = json.dumps({
                    "api_key": self.api_key, 
                    "secret_key": self.secret_key,
                    "passphrase": self.passphrase,
                    "partial": self.partial,
                    "last_date": present_date,
                    "action":"get_orders_future", #get_orders_spot or get_orders_margin or get_orders_future
                })
                response = requests.request("POST", url, headers=headers, data=payload)  
                if response.status_code == 200:            
                    present_date =present_date - 604800
                    data = response.json()
                    if 'orders' in data:
                        if isinstance(data['orders'], str):
                            return data['orders']
                        if len(data['orders']) > 0:
                            attempts = 0
                            if self.partial is True:
                                data['orders'] = [order for order in data['orders'] if order['created_at']/1000 >= last_date]
                                #_data = [order for order in data['orders'] if order['created_at'] >= last_date*1000]
                            future_orders.extend(data['orders'])
                        else:
                            attempts += 1                        
                    else:
                        attempts += 1
                else:
                    return response.text
                if attempts > 12:  # 12 attempts of 7 days
                    break
        self.all_orders.extend(future_orders)
        self.orders= self.all_orders
        try:
            last_date = self.all_orders[0]['created_at']
            self.last_date = last_date/1000
        except Exception:
            pass

        if get_orders_event:
            EventUser.update_user_events(get_orders_event)
        
        return True

    def get_kucoin_futures_orders_microservice(self):
        """
        Retrieves Kucoin futures orders through a microservice.
        This method connects to a microservice endpoint to fetch futures trading orders from Kucoin.
        It handles both full and partial data retrieval based on the instance configuration.
        The method performs the following steps:
        1. Updates progress status
        2. Constructs API request with authentication credentials
        3. Fetches data from microservice
        4. Processes response and extracts orders
        5. Handles S3 stored data if applicable
        6. Filters orders by date if partial retrieval is enabled
        7. Extends internal orders list with retrieved data
        Returns:
            bool: True if orders were successfully retrieved, otherwise returns error response text
        Raises:
            No explicit exceptions are raised, but may return API error responses as text
        Notes:
            - Uses instance variables for configuration (broker, API credentials, etc)
            - Interacts with S3 storage through microservice
            - Updates progress through internal _update_progress method
            - Records events through EventUser class
        """

        self._update_progress('Retrieving Futures Transactions', '18%')
        get_orders_event = EventUser.post_user_events(100, new_user=self.user_id)

        futures_orders = None 
        url = self.url_microservice()
        headers = {
        'Content-Type': 'application/json'
        }
        start_time = 0
        if self.partial is True:
            start_time = self.get_last_date_a(category='Perpetual', user_account=self.api_key_information)
            start_time = max(start_time, 0)
        end_time = int(time.time() * 1000)


        account = {
            "broker": self.broker.broker_key.lower(),
            "credentials": {
                "user": self.api_key,
                "password": self.secret_key,
                "passphrase": self.passphrase,
            },
            "categories": [
                "futures",
            ]
        }
        if start_time > 0:
            account["date_range"] = {
                "start": start_time,
                "end": end_time
            }
        payload = {
            "users": [
                {
                    "id": f"{self.user.uuid}-futures",
                    "accounts": [account]
                }
            ]
        }

        if not self.exclude_future_kucoin():
            self.params['broker']='FutureKucoin'
        response = requests.request("POST", url, headers=headers, json=payload)

        #print(response.json())
        if response.status_code == 200:
            if 'body' in response.json()['result']:
                body = response.json()['result']['body']
                #print('body',body)
                if body['final_path'] is not None and body['final_path'] != '':
                    file_path = body['final_path']
                    order = self.s3_to_json_microservice(s3_url=file_path)
                    futures_orders = order['orders']
                else:
                    futures_orders = []
        else:
            return response.text
        #self.all_orders = futures_orders
        for order in futures_orders:
            order['category'] = 'Perpetual'
            order['created_at'] = order['createdAt'] if 'createdAt' in order else order['created_at']
            order['created_at_formated'] = datetime.utcfromtimestamp(order['created_at']/1000).strftime('%Y-%m-%d %H:%M:%S')
            if 'createdAt' in order:
                del order['createdAt']
            
        if self.partial is True:
            futures_orders = [order for order in futures_orders if order['created_at']/1000 >= start_time]
        self.orders.extend(futures_orders if futures_orders is not None else [])
        self.all_orders = self.orders
        if get_orders_event:
            EventUser.update_user_events(get_orders_event)

        return True

    def exclude_future_kucoin(self):
        """
        Check if the operation should be excluded based on credential creation date.
        This method determines whether to exclude operations based on a cutoff date
        and credential creation time. It's specifically used to handle Kucoin-related
        operations.
        Returns:
            bool: True if the operation should be excluded, False otherwise.
                  Returns True if:
                  - self.add is True, or
                  - credential doesn't exist, or
                  - credential creation date is after cutoff date (2025-07-17)
                  Returns False if credential exists and was created before cutoff date.
        """
        cutoff_date = datetime.strptime('2025-07-17 00:00:00', '%Y-%m-%d %H:%M:%S')

        if not self.add and (self.credential and self.credential.created_at <= cutoff_date):
            return False
        return True

    def interpret_orders(self):
        """
        Interprets and processes trading orders from Kucoin exchange data.
        This method processes raw order data and transforms it into a standardized format for further use.
        It handles both spot and futures trading orders from Kucoin, performs data validation,
        and applies necessary transformations to ensure consistency.
        Key operations:
        - Validates and transforms date/time formats
        - Standardizes order actions (BUY/SELL)
        - Calculates proper order quantities and prices
        - Handles symbol formatting
        - Manages portfolio assignments
        - Validates against existing orders to prevent duplicates
        - Processes fees and pip values
        - Creates standardized trade records
        Returns:
            bool: True if orders were successfully interpreted and processed
        Side effects:
            - Populates self.out_result with processed order data
            - Updates progress through self._update_progress
            - Creates and updates user events
            - Modifies order data in place during processing
        Raises:
            No explicit exceptions are raised, but catches and handles various exceptions during processing
        """
        
        interpreter_orders_event = EventUser.post_user_events(101, new_user=self.user_id)

        result_b = []
        b=0
        new_date_time = []
        pip_value_order = dict()
        verify_njson_len = 0
        date_min = ''
        i=0
        try:
            date_min = min([parse(row['created_at_formated'].split(' ')[0]) for (index, row) in enumerate(self.all_orders)
                                        if 'created_at_formated' in row]) - timedelta(30)
        except Exception:
            #print(err)
            pass
        if self.future1 and not self.exclude_future_kucoin():
            self.params['broker']='FutureKucoin'
        else:
            self.params['broker']='Kucoin'
        verify_njson_len = 0
        orders_filerow = []
        orders_filerow = ImportParams.filerow_orders(self.params['get_session_userid'],
                                                         var_sync='orderId',
                                                         var_file='transaction id',
                                                         broker=self.params['broker']
                                                )
        
        for order in self.all_orders:
            # if order['channel'] == 'WEB': # ESTO LO HICE PARA QUE CUADRARA CON LA COMPETENCIA
            #     continue
            i+=1
            if i%100 == 0 or i == 1 or i == len( self.all_orders):
                self._update_progress(f"Reading orders: {i}/{len( self.all_orders)}", '19%')

            original_file_row = json.loads(json.dumps(order))
            njson = json.dumps(order)
            njson = hashlib.md5(njson.encode('utf-8')).hexdigest()
            multiplicator = 1
            try:
            #if self.future:
                multiplicator = float(order['value'])/ (float(order['price'])* float(order['size']))
            except Exception:
                pass
            
            action = order['buy/sell'].upper() if 'buy/sell' in order else order['side'].upper() if 'side' in order else order['transaction category'].upper() if 'transaction category' in order else ''
            action = 'BUY' if action == 'BUY' or action=='COVER' else 'SELL' if action == 'SELL' or action == 'SHORT' else ''
            order['symbol']  = order['symbol'].replace('-', '').replace('XBTUSDT', 'BTCUSDT')
            last_symbol = order['symbol'][-1:]
            if last_symbol == 'M':
                order['symbol'] = order['symbol'][:-1]
            order['date'] = order['created_at']
            order['date'] = datetime.utcfromtimestamp(order['date']/1000).strftime('%Y-%m-%d %H:%M:%S')
            order['type'] = order['tradeType'].upper() if 'tradeType' in order else order['types'].upper() if 'types' in order else 'TRADE'
            try:
            #if not self.future:
                if not 'dealFunds' in order:
                    order['price'] = float(float(order['dealValue']) / float(order['dealSize']) ) if float(order['dealSize']) != 0 else 0
                else:
                    order['price'] = float(float(order['dealFunds']) / float(order['dealSize']) ) if float(order['dealSize']) != 0 else 0
                order['quantity'] =  abs(float(order['dealSize'])) if 'dealSize' in order and ImportParams.isfloat(order['dealSize']) else ''
                qty =  str(order['quantity']).split('.')
                try:
                    decimal = qty[1][:4]
                except Exception:
                    decimal = qty
                qty[1] = decimal
                qty[1] = re.findall('[0-9]+',  qty[1])[0]
                order['quantity']  = float(".".join(qty))
            except:
                order['price'] = order['price']
                order['quantity'] =  order['size']


            #if not order['symbol'] or not order['date'] or not order['price'] or not action or order['type'] != "TRADE":
             #   continue
            #if order['type'].lower() != 'trade' and order['type'].lower() != 'busttrade':
             #   continue
            pip_value = 1
            if 'time' in order:
                date_split = order['date'].strip()
                d = date_split
                t = order['time'].replace('.',':') if 'time' in order else ''
                date_time = (d+' '+t).strip()
            else:
                date_time = order['date']
            original_file_row['date_tz'] = date_time
            self.any_error, date, time = ImportParams.get_param_datetime(date_time,b)
            new_date_time = ImportParams.convert_date(date, time, self.date_format,True,'UTC')

            order['date'] = new_date_time[0]
            order['time'] = new_date_time[1]
            fp = str(order['price']).replace(',', '').replace('$','')
            decimal = fp[::-1].find('.')
            decimal = decimal if decimal > 1 else 2
            price = round(float(fp),decimal)
            #price = round(float(order['price'].replace(',', '')),6)

            sm = len(order['symbol'])
            type_trade = 'crypto'
            option = 'CRYPTO'
            strike = ''
            expire = ''

            if order['category'] in self.params['user_portfolios']:
                user_portfolio = self.params['user_portfolios'][order['category']]
            else:
                user_portfolio = self.params['user_portfolio']  
            #value / (precio *size)
            pip_value = self.convert_usdt(order, pip_value_order, date )
            #################### VERIFY FILE ROW ######################
            try:
                order_id = ''
                portfolio_ = self.params['user_portfolio']
                if 'orderId' in order and order['orderId']:
                    order_id = order['orderId']
                valid_file_row = ImportParams.validate_filerow(orders_filerow,
                                                               njson=njson,
                                                               order_id=order_id,
                                                               date_tz=original_file_row['date_tz'],
                                                               price=fp,
                                                               option=option,
                                                               action=action,
                                                               quantity=order['quantity'],
                                                               strike=strike,
                                                               expire=expire,
                                                               portfolio=portfolio_
                                                              )
                if valid_file_row:
                    verify_njson_len = verify_njson_len + 1
                    continue
            except Exception:
                #print(traceback.format_exc())
                pass
            ##########################################################

            if 'size' in order:
                order['size'] = 1
            if self.future1:
                order['size'] = multiplicator
            order['trade_notes'] = order['trade_notes'] if 'trade_notes' in order else ''
            order['type_stock'] = type_trade
            order['type_option'] = option
            order['action'] = action
            order['price'] = fp
            order['shares'] = order['quantity']
            # print(order['shares'])
            order['fees'] = order['fee'] if 'fee' in order else '0.00'
            order['njson'] = njson
            order['decimal'] = decimal
            # print(order['decimal'])
            order['expire'] = expire
            order['strike'] = strike
            order['pip_value'] = pip_value
            order['original_file_row'] = original_file_row
            order['broker'] = self.params['broker']
            order['userid'] = self.params['get_session_userid']
            order['portfolio'] = user_portfolio
            order['app_broker'] = 1
            data_item = ImportParams.get_result_append(order)
            self.out_result.append(data_item)
            b = b + 1
        if interpreter_orders_event:
            EventUser.update_user_events(interpreter_orders_event)
        return True

    def save_orders(self):
        """
        Save and process trading orders with specified grouping rules.
        This method handles the regrouping and database storage of trading orders based on user configurations.
        It supports different grouping strategies ('split' or 'spread') and maintains error handling throughout
        the process.
        Returns:
            bool or dict: Returns True if orders are successfully processed and saved.
                         Returns a dictionary with error information if processing fails, containing:
                         - 'any_error': Error code if present
                         - 'rows': Number of trades processed
                         - 'ordens': Number of executions
                         - 'return_total': Total return value
                         - 'error_issue': Description of error if present
                         - 'broker': Broker identifier
                         - 'error_code': Error code
                         - 'mgs': Status or error message
        Side Effects:
            - Updates user status in database
            - Saves processed orders to database
            - Creates event logs
            - Saves transaction files
        Raises:
            May raise exceptions during database operations or file handling
        """
        regroup_orders_event = EventUser.post_user_events(102, new_user=self.user_id)
        
        import_broker = self.params['broker']
        regroup_value = [group['code'] for group in self.user.users_configs.trade_grouping if group['active'] == True][0]
        error_mgs = "There was an error. Our support team will review the issue and get back to you"
        # print('antes del regroup', len(self.out_result))
        spread = False
        self.orders = self.out_result
        self.save_files(file_type='out_result')
        output_group = {}
        if regroup_value == 'split':
            output_group = TradeRegroups.regroup_trades_closed(self.out_result,import_broker,self.user.id)
        elif regroup_value=='spread':
            spread = True
            output_group = TradeRegroups.regroup_trades(self.out_result, import_broker, self.user.id, spread=spread, user=self.user)
        else:
            output_group['error'] = "No grouping selected"

        if 'error' in output_group:
            self.out['mgs'] = error_mgs
            return self.out

        # self.orders = output_group
        # self.save_files(file_type='output_group')
        any_error, rows, return_total, error_issue, executions, self.upload_id = TradeRegroups.save_to_database(
                                                                                    output_group,
                                                                                    import_broker,
                                                                                    self.user.id,
                                                                                    archive = self.filename_csv,
                                                                                    upload_id = self.upload_id)
        self.out_result = output_group
        """
            rows : total trades imported
            executions: total executions imported
        """
        # print('despues del regroup', executions)
        self.rows = rows
        self.out = {'any_error': any_error, 'rows': rows, 'ordens': executions,
                'return_total': return_total, 'error_issue': error_issue,
                'broker': import_broker, 'error_code': any_error}
        if any_error and any_error != 21:
            self.out['mgs'] = error_mgs
            return self.out
        else:
            if rows > 0:
                try:
                    self.user.fk_user_status_id = 1
                    self.user.save()
                except Exception:
                    pass
                mgs = f'{str(rows)} Trades,{str(executions)} Executions'
            else:
                if self.last_date == 0 or self.last_date == None:
                    mgs = "No new trades found."
                else:
                    mgs = "No new trades to import"
        self.out['mgs'] = mgs
        self.save_file(tmp_route = self.tmp_route_csv, archive = self.filename_csv ,data = self.all_orders, to = self.to_csv, f_type= 'transactions_csv', content_type = 'text/csv',  portfolio =self.portfolio,  executions=executions , trades=rows, fk_broker_id = self.broker.broker_id, broker = self.broker.broker_key)
        #UserParams.append_param(self.user.id,'IMPORT_UPLOAD',False,replace=True)
        if regroup_orders_event:
            event_success = EventUser.update_user_events(regroup_orders_event)
        
        return True
    
    def handling_errors(self, err, syscode, httpcode):
        """
            System error codes
            200001:	Order creation for this pair suspended
            200002:	Order cancel for this pair suspended
            200003:	Number of orders breached the limit
            200009:	Please complete the KYC verification before you trade XX
            200004:	Balance insufficient
            400001:	Any of KC-API-KEY, KC-API-SIGN, KC-API-TIMESTAMP, KC-API-PASSPHRASE is missing in your request header
            400002:	KC-API-TIMESTAMP Invalid
            400003:	KC-API-KEY not exists
            400004:	KC-API-PASSPHRASE error
            400005:	Signature error
            400006:	The requested ip address is not in the api whitelist
            400007:	Access Denied
            404000:	Url Not Found
            400100:	Parameter Error
            400200:	Forbidden to place an order
            400500:	Your located country/region is currently not supported for the trading of this token
            400700:	Transaction restricted, there's a risk problem in your account
            400800:	Leverage order failed
            411100:	User are frozen
            500000:	Internal Server Error
            900001:	symbol not exists
        """
        #print('handling_errors')
        syscodes = {
            '400005':'Invalid Credentials',
            '400001':'Invalid Credentials',
            '400003':'Invalid Api Key',
            '400004':'Invalid Passphrase',
            '500000':"Kucoin's server is temporarily down. Try again in a few minutes",
            '400000':"Bad request",
        }
        self.not_error = syscodes.get(syscode, 'Unhandled error')

        HandlingErrorsLog(
            **{
            'user_id':self.user.id,
            'error':str(err),
            'syscode':syscode,
            'httpcode':httpcode,
            'data':{"api_key": self.api_key, "secret_key": self.secret_key},
            'path':'{}.{}.{}'.format(__name__,type(self).__name__, sys._getframe().f_code.co_name),
            'exception':Getters.get_full_class_name(err),
            'origin':getattr(request, 'origin',  'object has no attribute "origin"') ,
            }
        ).save()
        # return ErrorResponses.error_400(message=self.not_error, status='warning')
        return eval('ErrorResponses.error_{}'.format(httpcode))(message=self.not_error, status='warning')

