# Plan: Análisis de Validaciones Críticas Faltantes - KuCoin Normalizer

## Resumen Ejecutivo

Tras analizar exhaustivamente las implementaciones legacy (2,015 líneas en 3 archivos) vs nueva (404 líneas), se identificaron **24 categorías de validaciones**, de las cuales **4-5 son CRÍTICAS** para integridad de datos.

**Hallazgo clave**: La nueva implementación es **arquitecturalmente superior** con **80% menos código** (2,015→404 líneas). La mayoría de "validaciones faltantes" son en realidad:
- Funcionalidad de sync service (API auth, rate limiting, Lambda calls, incremental sync)
- Lógica que pertenece a p03_group (trade regrouping strategies)
- Cálculos que pertenecen a p04_calculate (fee currency conversion, pip values, forex base currency)

**Match Rate Actual**: **100%** hash compatibility (MD5 formula legacy-compatible con key ordering)

**Scope Actual**: Solo Futures (no Spot/Margin) - JSON API only (no CSV parsers)

---

## Validaciones Críticas Identificadas

### CATEGORÍA 1: INTEGRIDAD DE DATOS [CRITICIDAD: ALTA]

#### ⭐⭐⭐ 1. Side Validation Estricta (CRÍTICO)
**Ubicación Legacy:** `brokers_kucoin.py:130-131`
```python
if n['action'] not in ['BUY', 'SELL'] or not n['symbol'] or not n['price']:
    continue
```

**Ubicación Nueva:** `kucoin.py:124-125`
```python
side_raw = order.get("side", "").lower()
side = "BUY" if side_raw == "buy" else "SELL"  # PROBLEMA: Default silencioso!
```

**Estado Nuevo:** PERMISIVO - Convierte sides inválidos a SELL silenciosamente

**Impacto si falta:**
- Sides inválidos/vacíos se convierten a SELL sin warning
- Posiciones pueden calcularse incorrectamente
- No hay trazabilidad de datos corruptos
- Violaciones de data integrity

**Implementación:**
```python
# Líneas 123-130 (reemplazar lógica actual)
side_raw = order.get("side", "").lower()

# Validación estricta
if not side_raw or side_raw not in ("buy", "sell"):
    logger.warning(f"Skipping order {order.get('tradeId')}: invalid side '{side_raw}'")
    continue

side = "BUY" if side_raw == "buy" else "SELL"
```

**Archivo:** `kucoin.py` líneas 123-130
**Estimado:** 0.5 días

**Tests Requeridos:**
```python
def test_side_validation_rejects_invalid():
    """Verifica que sides inválidos son rechazados"""
    content = {"orders": [{"tradeId": "test1", "side": "INVALID", ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_side_validation_rejects_empty():
    """Verifica que sides vacíos son rechazados"""
    content = {"orders": [{"tradeId": "test1", "side": "", ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado
```

---

#### ⭐⭐⭐ 2. Required Fields Validation (CRÍTICO)
**Ubicación Legacy:** `brokers_kucoin.py:130-131`, `kucoin_export.py:859-862`
```python
if n['action'] not in ['BUY', 'SELL'] or not n['symbol'] or not n['price']:
    continue
```

**Estado Nuevo:** Sin validación explícita de campos requeridos

**Impacto si falta:**
- tradeId vacíos causan hash collisions
- symbol vacíos causan grouping errors
- price/size zero/missing causan calculation errors
- Datos incompletos pasan a stages siguientes

**Implementación:**
```python
# Después de línea 122 en parse_json_content()
trade_id = order.get("tradeId", "")
symbol = order.get("symbol", "")
side = order.get("side", "")
price = order.get("price", 0)
size = order.get("size", 0)

# Validar campos requeridos
if not trade_id:
    logger.warning(f"Skipping order: missing tradeId")
    continue

if not symbol:
    logger.warning(f"Skipping order {trade_id}: missing symbol")
    continue

if not side:
    logger.warning(f"Skipping order {trade_id}: missing side")
    continue

# Validar price > 0
try:
    price_float = float(price or 0)
    if price_float <= 0:
        logger.warning(f"Skipping order {trade_id}: invalid price {price_float}")
        continue
except (ValueError, TypeError):
    logger.warning(f"Skipping order {trade_id}: non-numeric price")
    continue

# Validar size > 0
try:
    size_float = float(size or 0)
    if size_float <= 0:
        logger.warning(f"Skipping order {trade_id}: invalid size {size_float}")
        continue
except (ValueError, TypeError):
    logger.warning(f"Skipping order {trade_id}: non-numeric size")
    continue
```

**Archivo:** `kucoin.py` después de línea 122
**Estimado:** 0.5 días

**Tests Requeridos:**
```python
def test_required_fields_missing_trade_id():
    content = {"orders": [{"tradeId": "", "symbol": "BTCUSDT", ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    assert len(result) == 0

def test_required_fields_missing_symbol():
    content = {"orders": [{"tradeId": "test1", "symbol": "", ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    assert len(result) == 0

def test_required_fields_zero_price():
    content = {"orders": [{"tradeId": "test1", "price": 0, ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    assert len(result) == 0

def test_required_fields_zero_size():
    content = {"orders": [{"tradeId": "test1", "size": 0, ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    assert len(result) == 0
```

---

#### ⭐⭐ 3. Type Filter Validation (MEDIO-ALTO)
**Ubicación Legacy:** `brokers_kucoin.py:634-637`
```python
if not order['symbol'] or not order['date'] or not order['price'] or not action or order['type'] != "TRADE":
    continue
if order['type'].lower() != 'trade' and order['type'].lower() != 'busttrade':
    continue
```

**Estado Nuevo:** Sin filtro de tipo de orden

**Impacto si falta:**
- Órdenes no-trade (TRANSFER, LIQUIDATION, FUNDING) contaminan reportes
- Cálculos P&L incorrectos con eventos no-trade
- Datos irrelevantes en análisis

**Implementación:**
```python
# Después de validación de campos requeridos
trade_type = order.get("tradeType", "").upper()
if trade_type and trade_type not in ("TRADE", "BUSTTRADE"):
    logger.debug(f"Skipping order {trade_id}: non-trade type '{trade_type}'")
    continue
```

**Archivo:** `kucoin.py` después de validación de campos
**Estimado:** 0.25 días

**Tests Requeridos:**
```python
def test_type_filter_rejects_transfer():
    content = {"orders": [{"tradeId": "test1", "tradeType": "TRANSFER", ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    assert len(result) == 0

def test_type_filter_accepts_trade():
    content = {"orders": [{"tradeId": "test1", "tradeType": "TRADE", ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    assert len(result) == 1

def test_type_filter_accepts_busttrade():
    content = {"orders": [{"tradeId": "test1", "tradeType": "BUSTTRADE", ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    assert len(result) == 1
```

---

#### ⭐⭐ 4. Status Filter Validation (MEDIO-ALTO - Condicional)
**Ubicación Legacy:** `brokers_kucoin.py:92-96`
```python
if 'status' in n and 'deal' in n['status']:
    n['status'] = 'COMPLETED'
if 'status' in n and n['status'].upper() not in ['COMPLETED']:
    continue
```

**Estado Nuevo:** Sin filtro de status

**Impacto si falta:**
- Órdenes PENDING/CANCELLED pueden pasar como ejecutadas
- Duplicación de datos si orden se completa después
- Data integrity issues

**Nota Importante:** La API JSON probablemente **solo devuelve órdenes completadas**. **Verificar con data real antes de implementar.**

**Implementación condicional:**
```python
# Solo si API devuelve campo status
status = order.get("status", "").upper()
if status and status not in ("COMPLETED", "DONE", "FILLED"):
    logger.debug(f"Skipping order {trade_id}: incomplete status '{status}'")
    continue
```

**Archivo:** `kucoin.py` después de validación de tipo
**Estimado:** 0.25 días (si necesario)

**Decisión Requerida:**
1. Revisar 10-20 sample orders del JSON actual
2. Buscar campo `status` o similar
3. Verificar si hay órdenes con status != COMPLETED

**Acción:**
- Si API solo devuelve completed: **OMITIR validación**
- Si API puede devolver incomplete: **IMPLEMENTAR**

---

### CATEGORÍA 2: CALIDAD DE DATOS [CRITICIDAD: MEDIA]

#### ⭐ 5. Multiplier Validation (MEDIO)
**Ubicación Legacy:** `kucoin_export.py:850-855`
```python
multiplicator = 1
try:
    multiplicator = float(order['value'])/ (float(order['price'])* float(order['size']))
except Exception:
    pass
```

**Ubicación Nueva:** `kucoin.py:300-306`
```python
pl.when((pl.col("price") > 0) & (pl.col("size") > 0))
.then(pl.col("value") / (pl.col("price") * pl.col("size")))
.otherwise(pl.lit(1.0))
.alias("multiplier"),
```

**Estado Nuevo:** Validación parcial - solo verifica divisores zero, no valores resultado

**Impacto si falta:**
- Multipliers inválidos (0, infinito, negativos) pasan sin warning
- Cálculos posteriores pueden fallar
- Anomalías no detectadas

**Implementación:**
```python
# Reemplazar líneas 300-306
pl.when((pl.col("price") > 0) & (pl.col("size") > 0) & (pl.col("value") > 0))
.then(
    pl.when(
        (pl.col("value") / (pl.col("price") * pl.col("size"))).is_between(0.01, 1000000)
    )
    .then(pl.col("value") / (pl.col("price") * pl.col("size")))
    .otherwise(pl.lit(1.0))  # Fallback para valores anómalos
)
.otherwise(pl.lit(1.0))
.alias("multiplier"),
```

**Archivo:** `kucoin.py` líneas 300-306
**Estimado:** 0.5 días

**Tests Requeridos:**
```python
def test_multiplier_validation_normal():
    """Multiplier en rango normal"""
    # value=50000, price=50000, size=1 → multiplier=1
    df = pl.DataFrame({"value": [50000], "price": [50000], "size": [1]})
    result = normalize(df)
    assert result["multiplier"][0] == 1.0

def test_multiplier_validation_anomalous():
    """Multiplier anómalo → fallback a 1.0"""
    # value=999999999, price=1, size=1 → multiplier=999999999 (fuera de rango)
    df = pl.DataFrame({"value": [999999999], "price": [1], "size": [1]})
    result = normalize(df)
    assert result["multiplier"][0] == 1.0  # Fallback
```

---

#### ⭐ 6. Fee Absolute Value (YA IMPLEMENTADO ✅)
**Ubicación Legacy:** `brokers_kucoin.py:183`
**Ubicación Nueva:** `kucoin.py:281`
```python
pl.col("fee").abs().alias("fees"),
```

**Estado:** **YA IMPLEMENTADO** ✅

**Notas:** La implementación actual ya usa `.abs()` correctamente.

---

#### 💰 7. Quantity Absolute Value + Decimal Limit (BAJO)
**Ubicación Legacy:** `kucoin_export.py:872-880`
```python
order['quantity'] = abs(float(order['dealSize']))
qty = str(order['quantity']).split('.')
decimal = qty[1][:4]  # Limitar a 4 decimales
order['quantity'] = float(".".join(qty))
```

**Ubicación Nueva:** `kucoin.py:262`
```python
pl.col("size").alias("quantity"),
```

**Estado Nuevo:** Sin abs() ni límite de decimales

**Impacto si falta:**
- Quantities negativas pueden causar confusión (aunque side indica dirección)
- Exceso de decimales puede causar problemas de precisión
- Mejor normalizar consistentemente

**Implementación:**
```python
# Línea 262
pl.col("size").abs().round(8).alias("quantity"),  # 8 decimales para crypto
```

**Archivo:** `kucoin.py` línea 262
**Estimado:** 0.25 días

**Tests Requeridos:**
```python
def test_quantity_absolute_value_negative():
    content = {"orders": [{"tradeId": "test1", "size": "-1.5", ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    normalized = normalize(result)
    assert normalized["quantity"][0] == 1.5  # Convertido a positivo

def test_quantity_decimal_limit():
    content = {"orders": [{"tradeId": "test1", "size": "1.123456789", ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    normalized = normalize(result)
    assert normalized["quantity"][0] == 1.12345679  # Redondeado a 8 decimales
```

---

#### 💰 8. Price Rounding (BAJO)
**Ubicación Legacy:** `brokers_kucoin.py:141-142`
```python
price = round(float(fp), 6) if vfp else 0.00
```

**Ubicación Nueva:** `kucoin.py:265`
```python
pl.col("price").alias("price"),
```

**Estado Nuevo:** Sin rounding

**Impacto si falta:**
- Precisión excesiva puede causar problemas de storage
- 6-8 decimales es suficiente para crypto
- Consistencia con legacy

**Implementación:**
```python
# Línea 265
pl.col("price").round(8).alias("price"),  # 8 decimales para crypto
```

**Archivo:** `kucoin.py` línea 265
**Estimado:** 0.25 días

**Tests Requeridos:**
```python
def test_price_rounding():
    content = {"orders": [{"tradeId": "test1", "price": "50000.123456789", ...}]}
    result = KucoinInterpreter.parse_json_content(content)
    normalized = normalize(result)
    assert normalized["price"][0] == 50000.12345679  # Redondeado a 8 decimales
```

---

### CATEGORÍA 3: VALIDACIONES CONDICIONALES [CRITICIDAD: CONDICIONAL]

#### 🎯 9. Spot/Margin Support (Condicional)
**Ubicación Legacy:**
- `kucoin_export.py:116-161` - Spot/Margin login y retrieval
- `brokers_kucoin.py` - CSV parser para Spot (762 líneas)

**Estado Nuevo:** Solo Futures support

**Decisión Necesaria:** ¿Usuarios operan KuCoin Spot o Margin?

**Query:**
```sql
SELECT
    category,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    MAX(created_at) as last_order
FROM kucoin_orders_raw
WHERE created_at > NOW() - INTERVAL '6 months'
GROUP BY category;
```

**Criterio:**
- Si (Spot count > 0 O Margin count > 0) Y user_count > 5: **IMPLEMENTAR** (3-5 días)
- Si ambos = 0: **OMITIR**

**Tareas (si implementar):**
1. Añadir category detection en can_handle()
2. Implementar parsing para Spot (no multiplier, multiplier=1)
3. Implementar parsing para Margin (cross/isolated modes)
4. Symbol transformation diferente por tipo
5. Tests para cada categoría

**Complejidad:** MEDIA-ALTA
**Riesgo:** MEDIO
**Estimado:** 3-5 días (solo si necesario)

---

#### 🎯 10. CSV Support (Condicional)
**Ubicación Legacy:**
- `brokers_kucoin.py` - 4 parsers CSV (getcsv, getcsv2, getcsv3, getcsv4)
- Formatos: standard, contracts, German locale, API format

**Estado Nuevo:** Solo JSON API support

**Decisión Necesaria:** ¿Usuarios suben CSVs manualmente?

**Query:**
```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*)) OVER (), 2) as percentage
FROM data_sources
WHERE broker_id = 'kucoin'
  AND created_at > NOW() - INTERVAL '6 months'
GROUP BY source_type;
```

**Criterio:**
- Si CSV percentage > 5%: **IMPLEMENTAR** (5-7 días)
- Si CSV percentage < 5%: **OMITIR**

**Tareas (si implementar):**
1. Implementar detector para 4 CSV formats
2. Parser 1: Standard format (TIME/CONTRACTS columns)
3. Parser 2: German locale (ZEIT/KONTRAKTE)
4. Parser 3: API export format (created_at_formated field)
5. Parser 4: Generic format
6. Tests para todos formatos

**Complejidad:** ALTA
**Riesgo:** MEDIO-ALTO
**Estimado:** 5-7 días (solo si necesario)

---

### CATEGORÍA 4: OUT OF SCOPE [NO IMPLEMENTAR]

#### ✅ 11. Credential Validation
**Ubicación Legacy:** `kucoin_export.py:231-232`
```python
if self.data['api_key'] == '' or self.data['secret_key'] == '':
    return "Invalid credentials"
```

**Conclusión:** **OUT OF SCOPE** - Pertenece a **sync service**

**Razón:** Normalizer procesa data ya sincronizada, no hace API calls

---

#### ✅ 12. Incremental Sync / Date Filtering
**Ubicación Legacy:** `kucoin_export.py:466-467`
```python
if self.partial is True:
    data['orders'] = [order for order in data['orders'] if order['created_at']/1000 >= last_date]
```

**Conclusión:** **OUT OF SCOPE** - Pertenece a **sync service**

**Razón:** Normalizer no gestiona sync state

---

#### ✅ 13. Retry Logic / Rate Limiting
**Ubicación Legacy:** `kucoin_export.py:476-477, 565-566`
```python
if attempts > 12:  # 12 attempts of 7 days
    break
```

**Conclusión:** **OUT OF SCOPE** - Pertenece a **sync service**

**Razón:** Normalizer no hace API calls

---

#### ✅ 14. Trade Regrouping
**Ubicación Legacy:** `kucoin_export.py:1006-1017`
```python
if regroup_value == 'split':
    output_group = TradeRegroups.regroup_trades_closed(...)
elif regroup_value == 'spread':
    output_group = TradeRegroups.regroup_trades(...)
```

**Conclusión:** **OUT OF SCOPE** - Pertenece a **p03_group** stage

**Razón:** Pipeline separation of concerns

---

#### ✅ 15. Fee Currency Conversion / Pip Value Lookup
**Ubicación Legacy:** `brokers_kucoin.py:160-167`
```python
if 'fee currency' in n and n['fee currency'].upper() != 'USDT':
    pip_fee = ImportParams.pip_value_crypto_(date=n['date'], currency=n['fee currency'])
```

**Conclusión:** **OUT OF SCOPE** - Pertenece a **p04_calculate** stage

**Razón:** Currency conversions y P&L calculations no van en normalizer

---

#### ✅ 16. Forex Base Currency Conversion
**Ubicación Legacy:** `brokers_kucoin.py:169-178`
```python
if return_is_base_currency:
    pip_value_params = float(ImportParams.value_pip_forex(n['date'], sym))
```

**Conclusión:** **OUT OF SCOPE** - Pertenece a **p04_calculate** stage

**Razón:** User-specific conversions no van en normalizer

---

## Validaciones Ya Implementadas ✅

### 17. ✅ Column Detection
**Estado:** YA IMPLEMENTADO (línea 85)
```python
def can_handle(cls, df: pl.DataFrame, metadata: dict) -> bool:
    required = {"tradeId", "symbol", "side", "price", "size", "fee", "createdAt"}
    return required.issubset(set(df.columns))
```

### 18. ✅ JSON Structure Validation
**Estado:** YA IMPLEMENTADO (líneas 108-110)

### 19. ✅ Empty Array Handling
**Estado:** YA IMPLEMENTADO (líneas 116-118)

### 20. ✅ Timestamp Parsing with Error Handling
**Estado:** YA IMPLEMENTADO (líneas 131-135)

### 21. ✅ Hash Computation (Legacy Compatible)
**Estado:** YA IMPLEMENTADO (líneas 137-167)
- MD5 hash con legacy key order (27 keys)
- 100% compatible con legacy system

### 22. ✅ Symbol Transformation
**Estado:** YA IMPLEMENTADO (líneas 246-256)
- Uppercase, remove M, add # prefix, XBTUSDT→BTCUSDT

### 23. ✅ Timezone Conversion
**Estado:** YA IMPLEMENTADO (líneas 268-275)
- UTC → America/New_York → naive

### 24. ✅ Side Normalization (Parcial)
**Estado:** IMPLEMENTADO PERO PERMISIVO
- **PROBLEMA:** Default silencioso a SELL (ver validación #1)

---

## Plan de Implementación por Fases

### FASE 1: Validaciones Críticas (1.5-2 días)

**Objetivo:** Prevenir datos inválidos que causen errores

**Tareas:**
1. ⭐⭐⭐ Side validation estricta (0.5 días)
2. ⭐⭐⭐ Required fields validation (0.5 días)
3. ⭐⭐ Type filter validation (0.25 días)
4. ⭐⭐ Status filter validation (0.25 días - verificar primero con data)

**Archivos:** `brokers/kucoin/kucoin.py` líneas 122-200

**Métricas de Éxito:**
- Sin sides inválidos en output
- Sin records con campos requeridos vacíos/zero
- Sin órdenes non-trade type contaminando datos
- Warnings logged para todos rechazos
- Match rate mantenido (100%)

**Complejidad:** BAJA
**Riesgo:** BAJO
**Estimado:** 1.5-2 días

---

### FASE 2: Calidad de Datos (1-1.5 días)

**Objetivo:** Mejorar robustez y consistencia con legacy

**Tareas:**
1. ⭐ Multiplier validation (0.5 días)
2. 💰 Quantity absolute value + decimal limit (0.25 días)
3. 💰 Price rounding (0.25 días)

**Archivos:** `brokers/kucoin/kucoin.py` líneas 262, 265, 300-306

**Métricas de Éxito:**
- Multipliers válidos (0.01-1000000 range)
- Quantities siempre positivas, max 8 decimales
- Prices redondeados a 8 decimales
- Consistencia con legacy format

**Complejidad:** BAJA
**Riesgo:** BAJO
**Estimado:** 1-1.5 días

---

### FASE 3: Spot/Margin Support (3-5 días si necesario)

**Objetivo:** Soporte para KuCoin Spot y Margin trading

**BLOQUEADOR - Decisión Requerida:**
⚠️ **Ejecutar SQL query primero**

**Query:**
```sql
SELECT
    category,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    MAX(created_at) as last_order
FROM kucoin_orders_raw
WHERE created_at > NOW() - INTERVAL '6 months'
GROUP BY category;
```

**Criterio:**
- Si (Spot count > 0 O Margin count > 0) Y user_count > 5: **IMPLEMENTAR**
- Si ambos = 0: **OMITIR**

**Complejidad:** MEDIA-ALTA
**Riesgo:** MEDIO
**Estimado:** 3-5 días (solo si necesario)

---

### FASE 4: CSV Support (5-7 días si necesario)

**Objetivo:** Soporte para uploads manuales de CSV

**BLOQUEADOR - Decisión Requerida:**
⚠️ **Ejecutar SQL query primero**

**Query:**
```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*)) OVER (), 2) as percentage
FROM data_sources
WHERE broker_id = 'kucoin'
  AND created_at > NOW() - INTERVAL '6 months'
GROUP BY source_type;
```

**Criterio:**
- Si CSV percentage > 5%: **IMPLEMENTAR**
- Si CSV percentage < 5%: **OMITIR**

**Complejidad:** ALTA
**Riesgo:** MEDIO-ALTO
**Estimado:** 5-7 días (solo si necesario)

---

## Matriz de Priorización

| # | Validación | Criticidad | Complejidad | Fase | Días |
|---|------------|-----------|-------------|------|------|
| 1 | Side Validation Estricta | ⭐⭐⭐ ALTA | BAJA | 1 | 0.5 |
| 2 | Required Fields | ⭐⭐⭐ ALTA | BAJA | 1 | 0.5 |
| 3 | Type Filter | ⭐⭐ MEDIA-ALTA | BAJA | 1 | 0.25 |
| 4 | Status Filter | ⭐⭐ MEDIA-ALTA | BAJA | 1 | 0.25 |
| 5 | Multiplier Validation | ⭐ MEDIA | BAJA | 2 | 0.5 |
| 6 | Fee Absolute | ✅ DONE | - | - | 0 |
| 7 | Qty Abs + Decimal | 💰 BAJA | BAJA | 2 | 0.25 |
| 8 | Price Rounding | 💰 BAJA | BAJA | 2 | 0.25 |
| 9 | Spot/Margin Support | 🎯 CONDICIONAL | MEDIA-ALTA | 3 | 3-5 |
| 10 | CSV Support | 🎯 CONDICIONAL | ALTA | 4 | 5-7 |
| 11-16 | Out of Scope | ✅ N/A | - | - | 0 |
| 17-24 | Ya Implementadas | ✅ DONE | - | - | 0 |

**Total Fase 1 (Crítico):** 1.5-2 días
**Total Fase 2 (Calidad):** 1-1.5 días
**Total Fase 1-2:** 2.5-3.5 días
**Total Fase 3-4 (Condicional):** 8-12 días adicionales si necesario

---

## Archivos Críticos

### 1. `brokers/kucoin/kucoin.py` (PRINCIPAL)
**Líneas Actuales:** 340
**Estimadas Post-Cambios:** ~420-480 (Fases 1-2), ~700-900 (si Fases 3-4)

**Cambios:**
- Líneas 122-200: Validaciones críticas en parse_json_content()
- Líneas 262-306: Calidad de datos en normalize()
- Si Fase 3: Spot/Margin logic
- Si Fase 4: CSV parsing methods

---

### 2. `brokers/kucoin/detector.py` (MODIFICAR SI FASE 3-4)
**Líneas Actuales:** 50
**Si Fase 3:** Añadir Spot/Margin detectors
**Si Fase 4:** Añadir CSV format detectors (4 variants)

---

### 3. `tests/brokers/test_kucoin.py` (MODIFICAR)
**Líneas Actuales:** 536
**Añadir:** ~100-150 líneas de tests (Fases 1-2), ~300+ (si Fases 3-4)

---

### 4. Legacy Files (REFERENCIA SOLO)
- `old_code_from_legacy/kucoin_export.py` (1,117 líneas)
- `old_code_from_legacy/brokers_kucoin.py` (762 líneas)
- `old_code_from_legacy/brokers_futurekucoin.py` (136 líneas)

---

## Comparación con Otros Brokers

| Broker | Legacy Lines | New Lines | Reduction | Critical Validations | Phase 1-2 Days |
|--------|--------------|-----------|-----------|---------------------|----------------|
| **KuCoin** | **2,015** | **404** | **80%** | **4-5** | **2.5-3.5** |
| Kraken | 787 | 361 | 54% | 3-4 | 2 |
| Coinbase | 1,725 | 324 | 81% | 2 | 2-3 |
| Bybit | 1,584 | 384 | 76% | 2 | 3-5 |

**Observaciones KuCoin:**
- **Máxima reducción de código** (80%) - legacy muy complejo (multi-account, 4 CSV variants)
- **Más validaciones críticas** (4-5) debido a:
  - Side validation permisiva
  - Sin required fields check
  - Sin type/status filtering
- **Esfuerzo similar a otros** (2.5-3.5 días Fase 1-2)
- **Correcta exclusión de scope:** Credentials, sync, rate limiting, regrouping, currency conversion

---

## Decisiones Pendientes

### Decisión 1: Status Filter
**Pregunta:** ¿API JSON devuelve campo `status`? ¿Devuelve órdenes incomplete?

**Verificación:**
1. Revisar 10-20 sample orders del JSON actual
2. Buscar campo `status` o similar
3. Verificar si hay órdenes con status != COMPLETED

**Acción:**
- Si API solo devuelve completed: **OMITIR validación**
- Si API puede devolver incomplete: **IMPLEMENTAR**

---

### Decisión 2: Spot/Margin Support
**Query SQL** (ver Fase 3)

**Criterio:**
- Si (Spot O Margin) Y user_count > 5: **IMPLEMENTAR** (3-5 días)
- Si ambos = 0: **OMITIR**

---

### Decisión 3: CSV Support
**Query SQL** (ver Fase 4)

**Criterio:**
- Si CSV > 5%: **IMPLEMENTAR** (5-7 días)
- Si CSV < 5%: **OMITIR**

---

## Conclusión

Análisis identifica **24 categorías de validaciones**:
- **4 CRÍTICAS (Fase 1):** 1.5-2 días
- **3 CALIDAD (Fase 2):** 1-1.5 días
- **2 CONDICIONALES (Fases 3-4):** 8-12 días si necesario
- **6 OUT OF SCOPE:** Sin acción (correctamente excluidas)
- **8 YA IMPLEMENTADAS:** Sin acción (1 con issue de permisividad)

**Hallazgo Clave:** Nueva implementación es **arquitecturalmente superior**:
- 80% menos código (2,015→404 líneas)
- 100% hash compatibility (legacy MD5 formula con key ordering)
- Correcta separación de concerns (sync, regrouping, calculations excluidos)
- Solo 4-5 validaciones críticas faltantes

**Issue Principal:** Side validation es **demasiado permisiva** (convierte inválidos a SELL silenciosamente) - DEBE corregirse.

**Segundo Issue:** Sin validación de required fields - puede causar hash collisions y grouping errors.

**RECOMENDACIÓN FINAL:**
1. **Implementar Fase 1 inmediatamente** (1.5-2 días) - CRÍTICO
2. **Implementar Fase 2** (1-1.5 días) - Robustez y consistencia
3. **Verificar status field en data** - Antes de implementar filtro
4. **Ejecutar SQL queries antes de Fases 3-4** - Decisiones data-driven
5. **NO sobre-ingenierizar** - Preservar simplicidad arquitectural

Total estimado crítico: **2.5-3.5 días**
Total con condicionales: **10.5-15.5 días** (solo si datos muestran necesidad)
