"""Kraken format detection."""

import polars as pl
from typing import List, Type

from pipeline.p01_normalize.base import BaseInterpreter
from pipeline.p01_normalize.exceptions import UnknownFormatError
from .kraken import KrakenInterpreter

# Register all interpreters for this broker
INTERPRETERS: List[Type[BaseInterpreter]] = [
    KrakenInterpreter,
]


def detect(df: pl.DataFrame, metadata: dict = None) -> BaseInterpreter:
    """
    Select interpreter based on can_handle() checks.

    Args:
        df: DataFrame to check
        metadata: Optional metadata dict

    Returns:
        Interpreter instance

    Raises:
        UnknownFormatError: If no interpreter can handle the data
    """
    metadata = metadata or {}

    # Sort by priority (higher values first)
    sorted_interpreters = sorted(
        INTERPRETERS,
        key=lambda x: x.get_priority(),
        reverse=True
    )

    # Return first interpreter that can handle the data
    for interpreter_cls in sorted_interpreters:
        if interpreter_cls.can_handle(df, metadata):
            return interpreter_cls()

    raise UnknownFormatError(f"No interpreter found for Kraken. Columns: {df.columns}")
