# Archivos de Código - Kraken Broker

## Estructura de Archivos

```
brokers/kraken/
├── kraken.py.original        # Código actual (sin modificar) - 306 líneas
├── detector.py               # Detector de formato - 44 líneas
└── README.md                 # Este archivo
```

---

## kraken.py.original

**Descripción:** Código actual del normalizador Kraken
**Líneas:** 306
**Estado:** REFERENCIA - No modificado

### Validaciones Actuales

**✅ YA IMPLEMENTADAS:**
- Column detection (can_handle)
- JSON structure validation
- Hash computation (MD5 de fill_id) - 100% match rate
- PF_ prefix detection para perpetual futures
- Timezone conversion (UTC → America/New_York)
- Naive datetime storage

**❌ FALTANTES (a implementar):**
- Side validation estricta (CRÍTICO - actualmente demasiado permisivo)
- Required fields validation (fill_id, symbol, side)
- Price/quantity zero validation
- Fee absolute value + rounding (preparación)
- Quantity absolute value

---

## Cambios Propuestos por Fase

### FASE 1: Validaciones Críticas

#### Cambio 1: Side Validation Estricta + Required Fields
**Ubicación:** Líneas 114-125

**Antes:**
```python
# Líneas 116-118 - PROBLEMA: Default silencioso
side = order.get("side", "").upper()
if side not in ("BUY", "SELL"):
    side = "BUY" if side.lower() in ("buy", "b") else "SELL"
```

**Después:**
```python
# Líneas 114-125 (expandido)
# Extraer campos requeridos
fill_id = order.get("fill_id", "")
symbol = order.get("symbol", "")
side = order.get("side", "")

# Validar campos requeridos
if not fill_id:
    logger.warning(f"Skipping order: missing fill_id")
    continue

if not symbol:
    logger.warning(f"Skipping fill {fill_id}: missing symbol")
    continue

if not side:
    logger.warning(f"Skipping fill {fill_id}: missing side")
    continue

# Normalizar side
side = side.upper()

# Validación estricta
if side not in ("BUY", "SELL"):
    logger.warning(f"Skipping fill {fill_id}: invalid side '{side}'")
    continue

# Continuar con ordertxid
ordertxid = str(fill_id)
```

---

#### Cambio 2: Price/Quantity Zero Validation
**Ubicación:** Después de línea 145-146

**Antes:**
```python
# Líneas 145-164 (simplificado)
"price": float(order.get("price", 0) or 0),
"qty": float(order.get("qty", 0) or 0),
```

**Después:**
```python
# Antes de construir el dict, validar price y qty
try:
    price_float = float(order.get("price", 0) or 0)
    if price_float <= 0:
        logger.warning(f"Skipping fill {ordertxid}: invalid price {price_float}")
        continue
except (ValueError, TypeError):
    logger.warning(f"Skipping fill {ordertxid}: non-numeric price")
    continue

try:
    qty_float = float(order.get("qty", 0) or 0)
    if qty_float <= 0:
        logger.warning(f"Skipping fill {ordertxid}: invalid quantity {qty_float}")
        continue
except (ValueError, TypeError):
    logger.warning(f"Skipping fill {ordertxid}: non-numeric quantity")
    continue

# Usar valores validados en dict
"price": price_float,
"qty": qty_float,
```

---

### FASE 2: Calidad de Datos

#### Cambio 3: Fee Absolute + Rounding
**Ubicaciones:** Líneas 157, 205

**Antes:**
```python
# Línea 169
"fee": 0.0,  # Hardcoded
```

**Después:**
```python
# Línea 157 (preparar para cuando API agregue fees)
"fee": abs(round(float(order.get("fee", 0) or 0), 2)),

# Línea 205 en normalize() (cuando se agregue fee al schema)
pl.col("fee").abs().round(2).alias("fees"),
```

---

#### Cambio 4: Quantity Absolute Value
**Ubicación:** Línea 164

**Antes:**
```python
# Línea 164
"qty": float(order.get("qty", 0) or 0),
```

**Después:**
```python
# Línea 164
"qty": abs(float(order.get("qty", 0) or 0)),
```

---

### FASE 3: Features Condicionales (SI NECESARIO)

#### Cambio 5: Spot Support (Condicional)
**Condición:** Si usuarios operan Kraken Spot

**Añadir:**
- Update `can_handle()` para detectar formato Spot
- Implementar parsing diferencial Spot vs Futures
- Symbol transformation para Spot (diferente de Futures)
- Asset class mapping

---

#### Cambio 6: CSV Support (Condicional)
**Condición:** Si >5% de uploads son CSVs

**Añadir:**
- CSV detector en `detector.py`
- `parse_csv()` method en `kraken.py`
- Field mapping CSV→normalized schema
- Header detection logic

---

## detector.py

**Descripción:** Detector de formato para interpreter selection
**Líneas:** 44
**Estado:** REFERENCIA - No modificado (modificar solo si Fase 3-4)

### Funcionalidad Actual
```python
def can_handle(df_sample: "pl.DataFrame") -> bool:
    required_columns = {"fill_id", "symbol", "side", "qty", "price"}
    return required_columns.issubset(set(df_sample.columns))
```

### Modificaciones Condicionales
- **Si Fase 3:** Añadir Spot detector
- **Si Fase 4:** Añadir CSV detector

---

## Implementación

### Para implementar cambios:

1. **Fase 1 (Crítica):**
   ```bash
   # Copiar archivo original
   cp kraken.py.original ../../brokers/kraken/kraken.py

   # Aplicar cambios 1, 2
   # Editar kraken.py según ejemplos arriba
   ```

2. **Fase 2 (Calidad):**
   ```bash
   # Aplicar cambios 3, 4
   ```

3. **Fase 3-4 (Condicional):**
   ```bash
   # Ejecutar queries de decisión primero
   # Solo implementar si datos lo justifican
   ```

4. **Testing:**
   ```bash
   # Tests unitarios
   pytest ../../tests/brokers/test_kraken.py -v

   # Validación QA (si existe script)
   python qa/validate_broker.py --broker-id kraken
   ```

---

## Métricas Actuales

### Performance
- **Hash Match Rate:** 100% (261/261 records)
- **Code Size:** 306 líneas (vs 787 legacy = 61% reducción)
- **Test Coverage:** 406 líneas de tests (ratio 1.33:1)

### Validaciones
- **Column Detection:** ✅ Implementado
- **JSON Validation:** ✅ Implementado
- **Hash Computation:** ✅ 100% compatible
- **PF_ Detection:** ✅ Implementado
- **Timezone Handling:** ✅ Implementado
- **Side Validation:** ⚠️ PERMISIVO (DEBE corregirse)
- **Required Fields:** ❌ Faltante
- **Price/Qty Zero:** ❌ Faltante
- **Fee Absolute:** ❌ Faltante (preparación)
- **Qty Absolute:** ❌ Faltante

---

## Arquitectura

### BaseInterpreter Pattern
```python
class KrakenInterpreter(BaseInterpreter):
    BROKER_ID: ClassVar[str] = "kraken"
    FORMAT_VERSION: ClassVar[str] = "1.0"
    SUPPORTED_ASSETS: ClassVar[Set[str]] = {"perpetual_future"}  # Solo Futures actualmente
```

### Data Flow
```
JSON String
→ parse_json_content() [validation, filtering, hashing]
→ DataFrame [raw parsed data]
→ normalize() [LazyFrame transformations]
→ Normalized DataFrame [matches schema]
```

### Hash Formula (Legacy Compatible)
```python
# MD5 de fill_id solamente
ordertxid = str(order.get("fill_id", ""))
file_row_hash = hashlib.md5(
    json.dumps(ordertxid).encode('utf-8')
).hexdigest()
```

---

## Referencias

- **Plan completo:** `../PLAN_ANALISIS_VALIDACIONES_KRAKEN.md`
- **Log de cambios:** `../CAMBIOS_IMPLEMENTADOS.md`
- **Ejemplos de código:** `../EJEMPLOS_CAMBIOS_CODIGO.md`
- **Código legacy:**
  - `../old_code_from_legacy/kraken_export.py` (662 líneas)
  - `../old_code_from_legacy/brokers_kraken.py` (125 líneas)

---

**Estado Actual:** ✅ CÓDIGO REFERENCIA COMPLETO
**Issue Crítico:** Side validation **demasiado permisiva** (convierte inválidos a SELL silenciosamente)
**Siguiente Paso:** Implementar Fase 1 (validaciones críticas)
