# Plan: Análisis de Validaciones Críticas Faltantes - Kraken Normalizer

## Resumen Ejecutivo

Tras analizar exhaustivamente las implementaciones legacy (787 líneas en 2 archivos) vs nueva (361 líneas), se identificaron **10 categorías de validaciones**, de las cuales **3-4 son CRÍTICAS** para integridad de datos.

**Hallazgo clave**: La nueva implementación es **arquitecturalmente superior** con **54% menos código** (787→361 líneas). Muchas "validaciones faltantes" son en realidad:
- Funcionalidad de sync service (rate limiting)
- Lógica que pertenece a p03_group (VWAP aggregation)
- Cálculos que pertenecen a p04_calculate (pip values)

**Match Rate Actual**: **100%** hash compatibility (261/261 records)

**Scope Actual**: Solo perpetual futures (PF_ prefix) - Spot support condicional

---

## Validaciones Críticas Identificadas

### CATEGORÍA 1: INTEGRIDAD DE DATOS [CRITICIDAD: ALTA]

#### ⭐⭐⭐ 1. Side Validation Estricta (CRÍTICO)
**Ubicación Legacy:** `brokers_kraken.py:67`
```python
if n['action'] not in ['BUY', 'SELL']:
    continue
```

**Ubicación Nueva:** `kraken.py:116-118`
```python
side = order.get("side", "").upper()
if side not in ("BUY", "SELL"):
    side = "BUY" if side.lower() in ("buy", "b") else "SELL"  # PROBLEMA: Default silencioso!
```

**Estado Nuevo:** PERMISIVO - Convierte sides inválidos a SELL silenciosamente

**Impacto si falta:**
- Sides inválidos/vacíos se convierten a SELL sin warning
- Posiciones pueden calcularse incorrectamente
- No hay trazabilidad de datos corruptos
- Violaciones de data integrity

**Implementación:**
```python
# Líneas 116-125 (reemplazar lógica actual)
side = order.get("side", "").upper()

# Validación estricta
if not side or side not in ("BUY", "SELL"):
    logger.warning(f"Skipping fill {ordertxid}: invalid side '{side}'")
    continue
```

**Archivo:** `kraken.py` líneas 116-125
**Estimado:** 0.5 días

**Tests Requeridos:**
```python
def test_side_validation_rejects_invalid():
    """Verifica que sides inválidos son rechazados"""
    content = {"data": [{"fill_id": "test1", "symbol": "PF_XBTUSD", "side": "INVALID", ...}]}
    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_side_validation_rejects_empty():
    """Verifica que sides vacíos son rechazados"""
    content = {"data": [{"fill_id": "test1", "symbol": "PF_XBTUSD", "side": "", ...}]}
    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado
```

---

#### ⭐⭐ 2. Required Fields Validation (MEDIO-ALTO)
**Ubicación Legacy:** `brokers_kraken.py:63,67`
```python
if not n['symbol'] or n['action'] not in ['BUY', 'SELL']:
    continue
```

**Estado Nuevo:** Sin validación explícita de campos requeridos

**Impacto si falta:**
- fill_id vacíos causan hash collisions
- symbol vacíos causan grouping errors
- Datos incompletos pasan a stages siguientes

**Implementación:**
```python
# Después de línea 114 en parse_json_content()
fill_id = order.get("fill_id", "")
symbol = order.get("symbol", "")
side = order.get("side", "")

# Validar campos requeridos
if not fill_id:
    logger.warning(f"Skipping order: missing fill_id")
    continue

if not symbol:
    logger.warning(f"Skipping fill {fill_id}: missing symbol")
    continue

if not side:
    logger.warning(f"Skipping fill {fill_id}: missing side")
    continue
```

**Archivo:** `kraken.py` línea ~114
**Estimado:** 0.5 días

**Tests Requeridos:**
```python
def test_required_fields_missing_fill_id():
    content = {"data": [{"fill_id": "", "symbol": "PF_XBTUSD", ...}]}
    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0

def test_required_fields_missing_symbol():
    content = {"data": [{"fill_id": "test1", "symbol": "", ...}]}
    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0
```

---

#### ⭐ 3. Price/Quantity Zero Validation (MEDIO)
**Ubicación Legacy:** `brokers_kraken.py:78-82`
```python
if 'price' in n and ImportParams.isfloat(n['price']):
    price = str(float(n['price']))
price = float(price) if price else 0
```

**Estado Nuevo:** Sin validación de valores zero/negativos

**Impacto si falta:**
- Precios zero causan errores en P&L calculations
- Quantities zero contaminan reportes
- Datos inválidos no son filtrados

**Implementación:**
```python
# En parse_json_content(), después de líneas 145-146
try:
    price_float = float(order.get("price", 0) or 0)
    if price_float <= 0:
        logger.warning(f"Skipping fill {ordertxid}: invalid price {price_float}")
        continue
except (ValueError, TypeError):
    logger.warning(f"Skipping fill {ordertxid}: non-numeric price")
    continue

try:
    qty_float = float(order.get("qty", 0) or 0)
    if qty_float <= 0:
        logger.warning(f"Skipping fill {ordertxid}: invalid quantity {qty_float}")
        continue
except (ValueError, TypeError):
    logger.warning(f"Skipping fill {ordertxid}: non-numeric quantity")
    continue
```

**Archivo:** `kraken.py` después de líneas 145-146
**Estimado:** 0.5 días

**Tests Requeridos:**
```python
def test_price_quantity_zero_rejected():
    content = {"data": [{"fill_id": "test1", "price": 0, "qty": 10, ...}]}
    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0

def test_price_quantity_negative_rejected():
    content = {"data": [{"fill_id": "test1", "price": 50000, "qty": -10, ...}]}
    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0
```

---

### CATEGORÍA 2: CALIDAD DE DATOS [CRITICIDAD: BAJA]

#### 💰 4. Fee Absolute Value + Rounding (BAJO)
**Ubicación Legacy:** `brokers_kraken.py:91-93`
```python
commission = round(float(commission), 2) if commission else 0.00
```

**Estado Nuevo:** Fee hardcoded a 0.0 (línea 169)

**Impacto si falta:**
- Actualmente no hay impacto (API no devuelve fees)
- Preparar para cuando API agregue fees

**Implementación:**
```python
# Línea 157 en parse_json_content()
"fee": abs(round(float(order.get("fee", 0) or 0), 2)),

# Línea 205 en normalize() - cuando se agregue columna fee
pl.col("fee").abs().round(2).alias("fees"),
```

**Archivo:** `kraken.py` líneas 157, 205
**Estimado:** 0.25 días

**Tests Requeridos:**
```python
def test_fee_absolute_value_negative():
    # Preparar para cuando API agregue fees
    pass

def test_fee_rounding_two_decimals():
    # Preparar para cuando API agregue fees
    pass
```

---

#### 🔢 5. Quantity Absolute Value (BAJO)
**Ubicación Legacy:** `brokers_kraken.py:89`
```python
quantity = abs(float(n['quantity'].replace('-', '')))
```

**Estado Nuevo:** Sin abs() en quantity

**Impacto si falta:**
- Quantities negativas pueden causar confusión
- Mejor normalizar siempre a positivo

**Implementación:**
```python
# Línea 164 en parse_json_content()
"qty": abs(float(order.get("qty", 0) or 0)),
```

**Archivo:** `kraken.py` línea 164
**Estimado:** 0.25 días

**Tests Requeridos:**
```python
def test_quantity_absolute_value_negative():
    content = {"data": [{"fill_id": "test1", "qty": "-10", ...}]}
    result = KrakenInterpreter.parse_json_content(content)
    assert result[0]["qty"] == 10.0  # Convertido a positivo
```

---

### CATEGORÍA 3: VALIDACIONES CONDICIONALES [CRITICIDAD: CONDICIONAL]

#### 🎯 6. Spot Support (Condicional)
**Ubicación Legacy:** `kraken_export.py:165`
```python
# Dual account support: Spot OR Futures
if account_type == 'spot':
    orders = self.get_spot_orders(...)
else:
    orders = self.get_futures_fills(...)
```

**Estado Nuevo:** Solo Futures (PF_ prefix detection en línea 128-133)

**Decisión Necesaria:** ¿Usuarios operan Kraken Spot (además de Futures)?

**Query:**
```sql
SELECT
    CASE
        WHEN symbol LIKE 'PF_%' THEN 'Futures'
        ELSE 'Spot'
    END as account_type,
    COUNT(*) as count
FROM kraken_orders_raw
WHERE created_at > NOW() - INTERVAL '6 months'
GROUP BY account_type;
```

**Si Spot count > 0:** Implementar soporte Spot (2-3 días)
- Añadir detector para formato Spot
- Implementar parsing diferencial
- Symbol transformation diferente

**Si Spot count = 0:** OMITIR

**Estimado:** 2-3 días (si necesario)

---

#### 🎯 7. CSV Support (Condicional)
**Ubicación Legacy:** `brokers_kraken.py` - 125 líneas de CSV parsing

**Estado Nuevo:** Solo JSON API support

**Decisión Necesaria:** ¿Usuarios suben CSVs manualmente?

**Query:**
```sql
SELECT
    source_type,
    COUNT(*) as count
FROM data_sources
WHERE broker_id = 'kraken'
  AND created_at > NOW() - INTERVAL '6 months'
GROUP BY source_type;
```

**Si CSV count > 5% del total:** Implementar CSV parser (3-4 días)
- Añadir can_handle() para CSV detection
- Implementar parse_csv() method
- Field mapping CSV→normalized schema

**Si CSV count < 5%:** OMITIR

**Estimado:** 3-4 días (si necesario)

---

### CATEGORÍA 4: OUT OF SCOPE [NO IMPLEMENTAR]

#### ✅ 8. Order Aggregation / VWAP Calculation
**Ubicación Legacy:** `kraken_export.py:440-481`
```python
# Agrupa múltiples fills en un order
# Calcula VWAP para precio promedio
```

**Conclusión:** **OUT OF SCOPE** - Pertenece a **p03_group** stage

**Razón:** Pipeline separation of concerns
- p01_normalize: Solo validación y transformación básica
- p03_group: Aggregation y VWAP calculations

---

#### ✅ 9. Rate Limiting / Exponential Backoff
**Ubicación Legacy:** `kraken_export.py:312-419`
```python
base_retry_delay = 33  # seconds
max_retries = 10
if error == 'EAPI:Rate limit exceeded':
    consecutive_rate_limits += 1
    wait_time = base_retry_delay * (consecutive_rate_limits + attempt)
    wait_time = min(wait_time, 180)  # Cap at 3 minutes
```

**Conclusión:** **OUT OF SCOPE** - Pertenece a **sync service**

**Razón:** Normalizer procesa data ya sincronizada, no hace API calls

---

#### ✅ 10. Pip Value Calculation (BTC/XBT conversion)
**Ubicación Legacy:** `kraken_export.py:588-597`
```python
if 'BTC' in n['symbol'] or 'XBT' in n['symbol']:
    pip = self.params['btc_pip'] if pip == 0 else pip
```

**Conclusión:** **OUT OF SCOPE** - Pertenece a **p04_calculate** stage

**Razón:** P&L calculations no van en normalizer

---

## Validaciones Ya Implementadas ✅

### 11. ✅ Column Detection
**Estado:** YA IMPLEMENTADO (línea 80)
```python
def can_handle(df_sample: "pl.DataFrame") -> bool:
    required_columns = {"fill_id", "symbol", "side", "qty", "price"}
    return required_columns.issubset(set(df_sample.columns))
```

### 12. ✅ JSON Structure Validation
**Estado:** YA IMPLEMENTADO (líneas 102-110)
```python
if not isinstance(content, dict) or "data" not in content:
    logger.warning("Invalid JSON structure")
    return []
```

### 13. ✅ Hash Computation
**Estado:** 100% COMPATIBLE (líneas 120-124)
```python
ordertxid = str(order.get("fill_id", ""))
file_row_hash = hashlib.md5(json.dumps(ordertxid).encode('utf-8')).hexdigest()
```
- Match rate: 261/261 records (100%)

### 14. ✅ PF_ Prefix Detection
**Estado:** YA IMPLEMENTADO (líneas 128-133)
```python
if symbol.startswith("PF_"):
    asset_class = "perpetual_future"
```

### 15. ✅ Timezone Conversion
**Estado:** YA IMPLEMENTADO (líneas 233-238)
```python
pl.col("_fill_time_dt")
    .dt.convert_time_zone("America/New_York")
    .dt.replace_time_zone(None)
    .alias("date")
```

---

## Plan de Implementación por Fases

### FASE 1: Validaciones Críticas (1.5 días)

**Objetivo:** Prevenir datos inválidos que causen errores

**Tareas:**
1. ✅ Side validation estricta (0.5 días)
2. ✅ Required fields validation (0.5 días)
3. ✅ Price/quantity zero validation (0.5 días)

**Archivos:** `brokers/kraken/kraken.py`

**Métricas de Éxito:**
- Sin sides inválidos en output
- Sin records con campos requeridos vacíos
- Sin precios/quantities zero/negativos
- Warnings logged para todos rechazos
- Match rate ≥ baseline (100%)

**Complejidad:** BAJA
**Riesgo:** BAJO
**Estimado:** 1.5 días

---

### FASE 2: Calidad de Datos (0.5 días)

**Objetivo:** Mejorar robustez y preparar para cambios API

**Tareas:**
1. ✅ Fee absolute value + rounding (0.25 días)
2. ✅ Quantity absolute value (0.25 días)

**Archivos:** `brokers/kraken/kraken.py`

**Métricas de Éxito:**
- Quantities siempre positivas
- Fees preparados para cuando API los agregue

**Complejidad:** BAJA
**Riesgo:** BAJO
**Estimado:** 0.5 días

---

### FASE 3: Spot Support (2-3 días si necesario)

**Objetivo:** Soporte para Kraken Spot trading

**BLOQUEADOR - Decisión Requerida:**
⚠️ **Ejecutar SQL query primero para verificar si usuarios operan Spot**

**Query:**
```sql
SELECT
    CASE
        WHEN symbol LIKE 'PF_%' THEN 'Futures'
        ELSE 'Spot'
    END as account_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count
FROM kraken_orders_raw
WHERE created_at > NOW() - INTERVAL '6 months'
GROUP BY account_type;
```

**Criterio:**
- Si Spot count > 0 Y user_count > 5: **IMPLEMENTAR**
- Si Spot count = 0: **OMITIR**

**Tareas (si implementar):**
1. Añadir detector para formato Spot
2. Implementar parsing diferencial
3. Symbol transformation
4. Asset class mapping
5. Tests

**Complejidad:** MEDIA
**Riesgo:** MEDIO
**Estimado:** 2-3 días (solo si necesario)

---

### FASE 4: CSV Support (3-4 días si necesario)

**Objetivo:** Soporte para uploads manuales de CSV

**BLOQUEADOR - Decisión Requerida:**
⚠️ **Ejecutar SQL query primero para verificar volumen de CSVs**

**Query:**
```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*)) OVER (), 2) as percentage
FROM data_sources
WHERE broker_id = 'kraken'
  AND created_at > NOW() - INTERVAL '6 months'
GROUP BY source_type;
```

**Criterio:**
- Si CSV percentage > 5%: **IMPLEMENTAR**
- Si CSV percentage < 5%: **OMITIR**

**Tareas (si implementar):**
1. Añadir can_handle() para CSV
2. Implementar parse_csv() method
3. Field mapping
4. Header detection
5. Tests

**Complejidad:** MEDIA-ALTA
**Riesgo:** MEDIO
**Estimado:** 3-4 días (solo si necesario)

---

## Matriz de Priorización

| # | Validación | Criticidad | Complejidad | Fase | Días |
|---|------------|-----------|-------------|------|------|
| 1 | Side Validation Estricta | ⭐⭐⭐ ALTA | BAJA | 1 | 0.5 |
| 2 | Required Fields | ⭐⭐ MEDIA-ALTA | BAJA | 1 | 0.5 |
| 3 | Price/Qty Zero | ⭐ MEDIA | BAJA | 1 | 0.5 |
| 4 | Fee Abs + Round | 💰 BAJA | BAJA | 2 | 0.25 |
| 5 | Qty Absolute | 💰 BAJA | BAJA | 2 | 0.25 |
| 6 | Spot Support | 🎯 CONDICIONAL | MEDIA | 3 | 2-3 |
| 7 | CSV Support | 🎯 CONDICIONAL | MEDIA-ALTA | 4 | 3-4 |
| 8-10 | Out of Scope | ✅ N/A | - | - | 0 |
| 11-15 | Ya Implementadas | ✅ DONE | - | - | 0 |

**Total Fase 1-2 (Crítico):** 2 días
**Total Fase 3-4 (Condicional):** 5-7 días adicionales si necesario

---

## Archivos Críticos

### 1. `brokers/kraken/kraken.py` (PRINCIPAL)
**Líneas Actuales:** 306
**Estimadas Post-Cambios:** ~360-380 (Fases 1-2), ~500-600 (si Fases 3-4)

**Cambios:**
- Líneas 116-125: Side validation estricta
- Línea ~114: Required fields validation
- Después líneas 145-146: Price/quantity zero validation
- Línea 157: Fee abs + round
- Línea 164: Quantity absolute value
- Si Fase 3: Spot support logic
- Si Fase 4: CSV parsing method

---

### 2. `brokers/kraken/detector.py` (MODIFICAR SI FASE 3-4)
**Líneas Actuales:** 44
**Si Fase 3:** Añadir Spot detector
**Si Fase 4:** Añadir CSV detector

---

### 3. `tests/brokers/test_kraken.py` (MODIFICAR)
**Líneas Actuales:** 406
**Añadir:** ~80-120 líneas de tests (Fases 1-2), ~200+ (si Fases 3-4)

---

### 4. Legacy Files (REFERENCIA SOLO)
- `old_code_from_legacy/kraken_export.py` (662 líneas)
- `old_code_from_legacy/brokers_kraken.py` (125 líneas)

---

## Comparación con Otros Brokers

| Broker | Legacy Lines | New Lines | Reduction | Critical Validations | Phase 1-2 Days |
|--------|--------------|-----------|-----------|---------------------|----------------|
| **Kraken** | **787** | **361** | **54%** | **3-4** | **2** |
| Coinbase | 1,725 | 324 | 81% | 2 | 2-3 |
| Bybit | 1,584 | 384 | 76% | 2 | 3-5 |
| Tastyworks | ~1,200 | ~350 | 71% | 3 | 4-6 |

**Observaciones Kraken:**
- **Menor reducción de código** (54%) porque legacy era más conciso
- **Más validaciones críticas** (3-4 vs 2-3) debido a side validation permisiva
- **Menor esfuerzo Fase 1-2** (2 días) - cambios pequeños pero críticos
- **Mejor separación de concerns** - Rate limiting y VWAP correctamente excluidos

---

## Decisiones Pendientes

### Decisión 1: Spot Support
**Query:**
```sql
SELECT
    CASE
        WHEN symbol LIKE 'PF_%' THEN 'Futures'
        ELSE 'Spot'
    END as account_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count
FROM kraken_orders_raw
WHERE created_at > NOW() - INTERVAL '6 months'
GROUP BY account_type;
```

**Criterio:**
- Si Spot count > 0 Y user_count > 5: **IMPLEMENTAR** (2-3 días)
- Si Spot count = 0: **OMITIR**

---

### Decisión 2: CSV Support
**Query:**
```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*)) OVER (), 2) as percentage
FROM data_sources
WHERE broker_id = 'kraken'
  AND created_at > NOW() - INTERVAL '6 months'
GROUP BY source_type;
```

**Criterio:**
- Si CSV percentage > 5%: **IMPLEMENTAR** (3-4 días)
- Si CSV percentage < 5%: **OMITIR**

---

## Conclusión

Análisis identifica **10 categorías de validaciones**:
- **3 CRÍTICAS (Fase 1):** 1.5 días
- **2 CALIDAD (Fase 2):** 0.5 días
- **2 CONDICIONALES (Fases 3-4):** 5-7 días si necesario
- **3 OUT OF SCOPE:** Sin acción (correctamente excluidas)
- **5 YA IMPLEMENTADAS:** Sin acción

**Hallazgo Clave:** Nueva implementación es **arquitecturalmente superior**:
- 54% menos código (787→361 líneas)
- 100% hash compatibility (261/261)
- Correcta separación de concerns (rate limiting, VWAP, pip calc excluidos)
- Solo 3-4 validaciones críticas faltantes

**Issue Principal:** Side validation es **demasiado permisiva** (convierte inválidos a SELL silenciosamente) - DEBE corregirse.

**RECOMENDACIÓN FINAL:**
1. **Implementar Fase 1 inmediatamente** (1.5 días) - CRÍTICO
2. **Implementar Fase 2** (0.5 días) - Robustez
3. **Ejecutar SQL queries antes de Fases 3-4** - Decisiones data-driven
4. **NO sobre-ingenierizar** - Preservar simplicidad arquitectural

Total estimado crítico: **2 días**
