# Ejemplos de Cambios de Código - Kraken

Este documento muestra ejemplos concretos con código antes/después para cada validación propuesta.

---

## FASE 1: Validaciones Críticas

### Cambio 1: Side Validation Estricta

**Archivo:** `brokers/kraken/kraken.py`
**Líneas:** 116-125
**Criticidad:** 🔴 ALTA

#### Código Actual (ANTES - PROBLEMA)
```python
# Líneas 116-118
side = order.get("side", "").upper()
if side not in ("BUY", "SELL"):
    side = "BUY" if side.lower() in ("buy", "b") else "SELL"  # DEFAULT SILENCIOSO!
```

**Problemas:**
- Sides inválidos ("INVALID", "", None, "HOLD") se convierten a SELL sin warning
- No hay trazabilidad de datos corruptos
- Posiciones se calculan incorrectamente
- Violaciones de data integrity

#### Código Propuesto (DESPUÉS)
```python
# Líneas 116-125 (reemplazar lógica completa)
side = order.get("side", "").upper()

# Validación estricta - rechaza sides inválidos
if not side or side not in ("BUY", "SELL"):
    logger.warning(f"Skipping fill {ordertxid}: invalid side '{side}'")
    continue
```

#### Justificación
- **Legacy:** `brokers_kraken.py:67` rechaza sides inválidos explícitamente
- **Impacto:** Previene corrupción de datos silenciosa
- **Casos rechazados:** side vacío, "INVALID", "HOLD", None, etc.

#### Tests Requeridos
```python
def test_side_validation_rejects_invalid():
    """Verifica que sides inválidos son rechazados"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "INVALID",  # Inválido
            "qty": "1.0",
            "price": "50000.0",
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_side_validation_rejects_empty():
    """Verifica que sides vacíos son rechazados"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "",  # Vacío
            "qty": "1.0",
            "price": "50000.0",
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_side_validation_accepts_valid_buy():
    """Verifica que BUY válido es aceptado"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "BUY",  # Válido
            "qty": "1.0",
            "price": "50000.0",
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 1  # Aceptado
    assert result[0]["side"] == "BUY"

def test_side_validation_accepts_valid_sell():
    """Verifica que SELL válido es aceptado"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "SELL",  # Válido
            "qty": "1.0",
            "price": "50000.0",
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 1  # Aceptado
    assert result[0]["side"] == "SELL"
```

---

### Cambio 2: Required Fields Validation

**Archivo:** `brokers/kraken/kraken.py`
**Línea:** ~114 (después de extraer variables)
**Criticidad:** 🟡 MEDIA-ALTA

#### Código Actual (ANTES)
```python
# Líneas 102-145 (simplificado)
for order in data:
    # Extrae variables sin validar si existen
    ordertxid = str(order.get("fill_id", ""))
    symbol = order.get("symbol", "")
    side = order.get("side", "").upper()
    # ... continúa procesando sin validar
```

#### Código Propuesto (DESPUÉS)
```python
# Después de línea 114 (después de extraer variables básicas)
for order in data:
    # Extraer campos requeridos
    fill_id = order.get("fill_id", "")
    symbol = order.get("symbol", "")
    side = order.get("side", "")

    # Validar campos requeridos antes de continuar
    if not fill_id:
        logger.warning(f"Skipping order: missing fill_id")
        continue

    if not symbol:
        logger.warning(f"Skipping fill {fill_id}: missing symbol")
        continue

    if not side:
        logger.warning(f"Skipping fill {fill_id}: missing side")
        continue

    # Normalizar side
    side = side.upper()
    if side not in ("BUY", "SELL"):
        logger.warning(f"Skipping fill {fill_id}: invalid side '{side}'")
        continue

    # Continuar con processing...
    ordertxid = str(fill_id)
    # ...
```

#### Justificación
- **Legacy:** `brokers_kraken.py:63,67` valida symbol y action
- **Impacto:** Previene hash collisions, grouping errors
- **Casos rechazados:** fill_id vacío, symbol vacío, side vacío

#### Tests Requeridos
```python
def test_required_fields_missing_fill_id():
    """Verifica que orders sin fill_id son rechazados"""
    content = {
        "data": [{
            "fill_id": "",  # Vacío
            "symbol": "PF_XBTUSD",
            "side": "BUY",
            "qty": "1.0",
            "price": "50000.0",
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_required_fields_missing_symbol():
    """Verifica que orders sin symbol son rechazados"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "",  # Vacío
            "side": "BUY",
            "qty": "1.0",
            "price": "50000.0",
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_required_fields_all_present():
    """Verifica que orders con todos los campos son aceptados"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "BUY",
            "qty": "1.0",
            "price": "50000.0",
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 1  # Aceptado
```

---

### Cambio 3: Price/Quantity Zero Validation

**Archivo:** `brokers/kraken/kraken.py`
**Línea:** Después de líneas 145-146
**Criticidad:** 🟡 MEDIA

#### Código Actual (ANTES)
```python
# Líneas 145-146
"price": float(order.get("price", 0) or 0),
"qty": float(order.get("qty", 0) or 0),

# No hay validación de valores zero/negativos
```

#### Código Propuesto (DESPUÉS)
```python
# Después de líneas 145-146, antes de construir el dict
# Validar price > 0
try:
    price_float = float(order.get("price", 0) or 0)
    if price_float <= 0:
        logger.warning(f"Skipping fill {ordertxid}: invalid price {price_float}")
        continue
except (ValueError, TypeError):
    logger.warning(f"Skipping fill {ordertxid}: non-numeric price")
    continue

# Validar qty > 0
try:
    qty_float = float(order.get("qty", 0) or 0)
    if qty_float <= 0:
        logger.warning(f"Skipping fill {ordertxid}: invalid quantity {qty_float}")
        continue
except (ValueError, TypeError):
    logger.warning(f"Skipping fill {ordertxid}: non-numeric quantity")
    continue

# Construir dict con valores validados
parsed_orders.append({
    "_file_row_hash": file_row_hash,
    "_original_file_row": json.dumps(order),
    "_row_index": row_index,
    "fill_id": ordertxid,
    "symbol": symbol,
    "side": side,
    "price": price_float,  # Usar valor validado
    "qty": qty_float,      # Usar valor validado
    # ...
})
```

#### Justificación
- **Legacy:** `brokers_kraken.py:78-82` valida price con isfloat()
- **Impacto:** Previene errores en P&L calculations
- **Casos rechazados:** price ≤ 0, qty ≤ 0, valores non-numeric

#### Tests Requeridos
```python
def test_price_quantity_zero_price_rejected():
    """Verifica que precios zero son rechazados"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "BUY",
            "qty": "1.0",
            "price": "0",  # Zero
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_price_quantity_zero_qty_rejected():
    """Verifica que quantities zero son rechazadas"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "BUY",
            "qty": "0",  # Zero
            "price": "50000.0",
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_price_quantity_negative_price_rejected():
    """Verifica que precios negativos son rechazados"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "BUY",
            "qty": "1.0",
            "price": "-50000.0",  # Negativo
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_price_quantity_non_numeric_rejected():
    """Verifica que valores non-numeric son rechazados"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "BUY",
            "qty": "1.0",
            "price": "invalid",  # Non-numeric
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado
```

---

## FASE 2: Calidad de Datos

### Cambio 4: Fee Absolute Value + Rounding

**Archivo:** `brokers/kraken/kraken.py`
**Líneas:** 157, 205
**Criticidad:** 🟢 BAJA (Preparación para futuro)

#### Código Actual (ANTES)
```python
# Línea 157 (en parse_json_content)
# Fee no se procesa (API no devuelve fees actualmente)

# Línea 169 (hardcoded en data dict)
"fee": 0.0,  # Hardcoded
```

#### Código Propuesto (DESPUÉS)
```python
# Línea 157 (preparar para cuando API agregue fees)
"fee": abs(round(float(order.get("fee", 0) or 0), 2)),

# Línea 205 en normalize() - cuando se agregue columna fee al schema
pl.col("fee").abs().round(2).alias("fees"),
```

#### Justificación
- **Legacy:** `brokers_kraken.py:91-93` aplica abs() y round(2)
- **Impacto:** Preparación para cuando API agregue fees
- **Casos corregidos:** Fees negativos → positivos, precisión 2 decimales

#### Tests Requeridos
```python
def test_fee_absolute_value_negative():
    """Preparar para cuando API agregue fees negativos"""
    # TODO: Implementar cuando API devuelva fees
    pass

def test_fee_rounding_two_decimals():
    """Preparar para verificar redondeo a 2 decimales"""
    # TODO: Implementar cuando API devuelva fees
    pass
```

---

### Cambio 5: Quantity Absolute Value

**Archivo:** `brokers/kraken/kraken.py`
**Línea:** 164
**Criticidad:** 🟢 BAJA

#### Código Actual (ANTES)
```python
# Línea 164
"qty": float(order.get("qty", 0) or 0),
```

#### Código Propuesto (DESPUÉS)
```python
# Línea 164
"qty": abs(float(order.get("qty", 0) or 0)),
```

#### Justificación
- **Legacy:** `brokers_kraken.py:89` aplica abs()
- **Impacto:** Normaliza quantities a positivas
- **Casos corregidos:** Negative quantities → positive

#### Tests Requeridos
```python
def test_quantity_absolute_value_negative():
    """Verifica conversión de quantities negativas a positivas"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "BUY",
            "qty": "-1.5",  # Negativo
            "price": "50000.0",
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 1
    assert result[0]["qty"] == 1.5  # Convertido a positivo

def test_quantity_absolute_value_positive_unchanged():
    """Verifica que quantities positivas no cambian"""
    content = {
        "data": [{
            "fill_id": "test-1",
            "symbol": "PF_XBTUSD",
            "side": "BUY",
            "qty": "1.5",  # Ya positivo
            "price": "50000.0",
            "fill_time": "2025-01-01T00:00:00.000000Z"
        }]
    }

    result = KrakenInterpreter.parse_json_content(content)
    assert len(result) == 1
    assert result[0]["qty"] == 1.5  # Sin cambio
```

---

## Resumen de Cambios por Archivo

### brokers/kraken/kraken.py
- **Línea ~114-125:** Required fields validation + Side validation estricta (Cambios 1, 2)
- **Después línea 145:** Price/qty zero validation (Cambio 3)
- **Línea 157:** Fee abs + round (Cambio 4)
- **Línea 164:** Qty absolute value (Cambio 5)

### tests/brokers/test_kraken.py
- Añadir ~10-15 funciones de test nuevas
- Coverage de Fases 1 y 2

---

## Orden de Implementación

1. ✅ **Cambio 2:** Required fields validation (Fase 1)
2. ✅ **Cambio 1:** Side validation estricta (Fase 1)
3. ✅ **Cambio 3:** Price/qty zero validation (Fase 1)
4. ✅ **Validar Fase 1:** Tests passing, match rate ≥ baseline
5. ✅ **Cambio 4:** Fee abs + round (Fase 2)
6. ✅ **Cambio 5:** Qty absolute value (Fase 2)
7. ✅ **Validar Fase 2:** Tests passing
8. ⚠️ **Decisiones:** Ejecutar queries para Fase 3-4

---

**Nota:** Todos los cambios mantienen compatibilidad con arquitectura existente y usan operaciones type-safe para validación.
