# Plan: Análisis de Validaciones Críticas Faltantes - Interactive Brokers Normalizer

## Resumen Ejecutivo

Tras analizar exhaustivamente ambas implementaciones (legacy: 1,696 líneas en `brokers_ib.py` vs nueva: 513 líneas en `flexquery.py`), se identificaron **23 validaciones críticas** que faltan en la nueva implementación y podrían comprometer la integridad de datos.

## Contexto de la Migración

### Arquitectura Legacy
- Manejo de múltiples formatos (CSV con 5 parsers diferentes, XML, API JSON)
- Conexión directa a IB API con autenticación
- Validación de credenciales y asignación de portfolio
- Lógica de negocio integrada (deduplicación, transformación de símbolos, detección de asset types)

### Arquitectura Nueva
- Enfoque en un solo formato: IB FlexQuery XML
- Arquitectura modular basada en intérpretes
- Separación de responsabilidades (parsing, normalización, validación)
- Sin lógica de conexión API (probablemente movida a otro módulo)

---

## Validaciones Críticas Faltantes

### CATEGORÍA 1: VALIDACIONES DE INTEGRIDAD DE DATOS [CRITICIDAD: ALTA]

#### ❌ 1. Validación de Quantity Zero
**Ubicación Legacy:** `brokers_ib.py:391-393`
```python
if int(float(size.replace(',', '').replace('-', ''))) == 0:
    continue  # Skip records with zero quantity
```

**Descripción:** El código legacy descarta trades con cantidad = 0, que son registros inválidos (probablemente ajustes contables o cancelaciones).

**Impacto si falta:**
- Trades fantasma en la base de datos
- Cálculos de posición incorrectos
- Métricas de volumen infladas artificialmente

**Nivel de Criticidad:** 🔴 **ALTA** - Afecta directamente la integridad de las posiciones

**Recomendación de implementación:**
```python
# En flexquery.py, método normalize(), después de cast de quantity
.filter(pl.col("quantity") > 0)  # Add after line 216
```

**Ubicación sugerida:** `flexquery.py:216` (después del `.abs()` en quantity)

---

#### ❌ 2. Validación de Symbol Vacío
**Ubicación Legacy:** `brokers_ib.py:476-477`
```python
if symbol == "":
    continue  # Skip trades without symbol
```

**Descripción:** Trades sin símbolo son inválidos y deben descartarse.

**Impacto si falta:**
- Trades sin identificación en base de datos
- Errores en búsquedas y agrupaciones
- Impossibilidad de matching con market data

**Nivel de Criticidad:** 🔴 **ALTA** - Datos no utilizables

**Recomendación de implementación:**
```python
# En flexquery.py, después de normalización de symbol
.filter(pl.col("symbol") != "")  # Add after line 178
```

**Ubicación sugerida:** `flexquery.py:178` (después de uppercase + trim)

---

#### ❌ 3. Validación de Price Zero con Fallback
**Ubicación Legacy:** `brokers_ib.py:427-441`
```python
if isfloat(price_val):
    price = float(price_val)
else:
    price = float(n['closePrice']) if 'closePrice' in n and isfloat(n['closePrice']) else 0

if price == 0:
    if n_val['notes'] == 'A':  # Assignment trades
        price = float(n['closePrice']) if 'closePrice' in n else 0
```

**Descripción:** Trades con precio 0 pueden ser válidos en casos especiales (asignaciones de opciones), pero normalmente indican datos corruptos. El legacy tiene fallback a `closePrice`.

**Impacto si falta:**
- Trades con $0 afectan cálculos de P&L
- Imposible calcular valor de posición
- Breaks en reportes financieros

**Nivel de Criticidad:** 🔴 **ALTA** - Afecta cálculos financieros

**Recomendación de implementación:**
```python
# En flexquery.py, añadir validación después de cast de price
.with_columns([
    pl.when(pl.col("price") == 0)
      .then(
          pl.col("fifoPnlRealized").fill_null(0) / pl.col("quantity")  # Calculate from realized P&L
      )
      .otherwise(pl.col("price"))
      .alias("price")
])
.filter(pl.col("price") != 0)  # Discard if still zero
```

**Ubicación sugerida:** `flexquery.py:215` (después de cast price)

---

#### ❌ 4. Validación de Action (BUY/SELL) Inválido
**Ubicación Legacy:** `brokers_ib.py:380-388`
```python
if action == '' or (action != 'BUY' and action != 'SELL'):
    continue  # Skip trades without valid action
```

**Descripción:** Solo se permiten acciones BUY o SELL después de normalización.

**Impacto si falta:**
- Trades con dirección desconocida
- Cálculos de posición erróneos (no sabe si suma o resta)
- Imposible determinar long/short

**Nivel de Criticidad:** 🔴 **ALTA** - Crítico para posiciones

**Recomendación de implementación:**
```python
# En flexquery.py, añadir después de normalización de side
.filter(pl.col("side").is_in(["BUY", "SELL"]))  # Add after line 173
```

**Ubicación sugerida:** `flexquery.py:173` (después de map `buySell`)

---

#### ❌ 5. Validación de Timestamp Inválido
**Ubicación Legacy:** `brokers_ib.py:494-500`
```python
try:
    trade_date = dateutil.parser.parse(datetime_str)
except:
    try:
        trade_date = parse_date_formats(date_val)  # Multiple format fallback
    except:
        continue  # Skip if date cannot be parsed
```

**Descripción:** Trades sin fecha válida deben ser descartados porque no se pueden ordenar ni agrupar temporalmente.

**Impacto si falta:**
- Trades sin timestamp en DB
- Imposible ordenar cronológicamente
- Breaks en cálculos de FIFO/LIFO
- Reportes diarios incorrectos

**Nivel de Criticidad:** 🔴 **ALTA** - Esencial para ordenamiento temporal

**Recomendación de implementación:**
```python
# En flexquery.py, después de parse de datetime
.filter(pl.col("timestamp").is_not_null())  # Add after line 165
```

**Ubicación sugerida:** `flexquery.py:165` (después de `str.to_datetime`)

---

### CATEGORÍA 2: TRANSFORMACIONES DE ASSET TYPES [CRITICIDAD: MEDIA-ALTA]

#### ⚠️ 6. Symbol Transformation - Bitcoin Cleanup
**Ubicación Legacy:** `brokers_ib.py:569-570`
```python
if symbol.startswith("BITCOIN") and tr_type == 'STK':
    symbol = symbol.replace("BITCOIN", "")
```

**Descripción:** IB exporta Bitcoin stocks con prefijo "BITCOIN" que debe eliminarse.

**Impacto si falta:**
- Símbolos incorrectos (e.g., "BITCOINUSD" en vez de "USD")
- No hace match con símbolos estándar de mercado
- Gráficos y market data no funcionan

**Nivel de Criticidad:** 🟠 **MEDIA-ALTA** - Afecta UX pero no integridad crítica

**Recomendación de implementación:**
```python
# En flexquery.py, añadir después de uppercase de symbol
.with_columns([
    pl.when((pl.col("symbol").str.starts_with("BITCOIN")) & (pl.col("asset") == "stocks"))
      .then(pl.col("symbol").str.replace("BITCOIN", ""))
      .otherwise(pl.col("symbol"))
      .alias("symbol")
])
```

**Ubicación sugerida:** `flexquery.py:179` (nueva transformación de symbol)

---

#### ⚠️ 7. Symbol Transformation - Hong Kong Stocks
**Ubicación Legacy:** `brokers_ib.py:478-479`
```python
if symbol.isdigit() and len(symbol) == 4:
    symbol = f"{symbol}.HK"  # Hong Kong stocks are numeric
```

**Descripción:** Símbolos de Hong Kong son numéricos (e.g., "0700") y necesitan sufijo ".HK".

**Impacto si falta:**
- Símbolos numéricos sin contexto de mercado
- Duplicados con otros mercados (e.g., 0700 China vs 0700 en otro mercado)
- No hace match con data providers

**Nivel de Criticidad:** 🟠 **MEDIA-ALTA** - Crítico para usuarios con HK stocks

**Recomendación de implementación:**
```python
# En flexquery.py, añadir transformación condicional
.with_columns([
    pl.when(pl.col("symbol").str.lengths() == 4)
      .then(
          pl.when(pl.col("symbol").str.contains(r"^\d{4}$"))
            .then(pl.col("symbol") + ".HK")
            .otherwise(pl.col("symbol"))
      )
      .otherwise(pl.col("symbol"))
      .alias("symbol")
])
```

**Ubicación sugerida:** `flexquery.py:179` (junto con otras transformaciones de symbol)

---

#### ⚠️ 8. CFD Symbol Formatting
**Ubicación Legacy:** `brokers_ib.py:620`
```python
if tr_type == 'CFD':
    symbol = f"{symbol}|CFD"
```

**Descripción:** CFDs necesitan sufijo "|CFD" para distinguirlos de stocks con mismo symbol.

**Impacto si falta:**
- CFDs confundidos con stocks
- Cálculos de margin incorrectos
- Reportes de asset class mezclados

**Nivel de Criticidad:** 🟠 **MEDIA** - Afecta clasificación pero no integridad financiera

**Recomendación de implementación:**
```python
# En flexquery.py, añadir transformación para CFDs
.with_columns([
    pl.when(pl.col("asset") == "cfd")
      .then(pl.col("symbol") + "|CFD")
      .otherwise(pl.col("symbol"))
      .alias("symbol")
])
```

**Ubicación sugerida:** `flexquery.py:179`

---

#### ⚠️ 9. Forex Symbol Formatting
**Ubicación Legacy:** `brokers_ib.py:773`
```python
symbol = "$" + symbol.replace(".", "")  # e.g., EUR.USD → $EURUSD
```

**Descripción:** Forex pairs necesitan formato estándar `$EURUSD` (sin punto, con prefijo $).

**Impacto si falta:**
- Forex pairs con formato inconsistente (EUR.USD vs EURUSD)
- No hace match con símbolos de platform
- Búsquedas de forex no funcionan

**Nivel de Criticidad:** 🟠 **MEDIA-ALTA** - Crítico para traders de forex

**Recomendación de implementación:**
```python
# En flexquery.py, añadir transformación para forex
.with_columns([
    pl.when(pl.col("asset") == "forex")
      .then(pl.lit("$") + pl.col("symbol").str.replace_all(".", ""))
      .otherwise(pl.col("symbol"))
      .alias("symbol")
])
```

**Ubicación sugerida:** `flexquery.py:179`

---

#### ⚠️ 10. Futures Symbol Formatting
**Ubicación Legacy:** `brokers_ib.py:678-689`
```python
symbol = "/" + symbol  # e.g., ESH24 → /ESH24
```

**Descripción:** Futures necesitan prefijo "/" para distinguirse de stocks.

**Impacto si falta:**
- Futures confundidos con stocks
- Símbolos no coinciden con estándar de mercado
- Búsquedas de futures fallan

**Nivel de Criticidad:** 🟠 **MEDIA** - Importante para traders de futures

**Recomendación de implementación:**
```python
# En flexquery.py, añadir transformación para futures
.with_columns([
    pl.when(pl.col("asset") == "futures")
      .then(pl.lit("/") + pl.col("symbol"))
      .otherwise(pl.col("symbol"))
      .alias("symbol")
])
```

**Ubicación sugerida:** `flexquery.py:179`

---

#### ⚠️ 11. SPXW → SPX Conversion para Options
**Ubicación Legacy:** `brokers_ib.py:1264-1265`
```python
if symbol.startswith("SPXW"):
    symbol = symbol.replace("SPXW", "SPX")
```

**Descripción:** SPXW (weekly SPX options) deben normalizarse a SPX para consistencia.

**Impacto si falta:**
- SPX y SPXW tratados como diferentes underlyings
- Reportes de opciones fragmentados
- Posiciones de SPX no se agrupan correctamente

**Nivel de Criticidad:** 🟡 **MEDIA** - Afecta agrupación pero no crítico

**Recomendación de implementación:**
```python
# En flexquery.py, añadir para options
.with_columns([
    pl.when((pl.col("asset") == "options") & (pl.col("symbol").str.starts_with("SPXW")))
      .then(pl.col("symbol").str.replace("SPXW", "SPX"))
      .otherwise(pl.col("symbol"))
      .alias("symbol")
])
```

**Ubicación sugerida:** `flexquery.py:179`

---

### CATEGORÍA 3: MANEJO DE COMISIONES Y FEES [CRITICIDAD: MEDIA]

#### ⚠️ 12. Swap Detection (High Commission Reclassification)
**Ubicación Legacy:** `brokers_ib.py:840-842`
```python
if commission > 1000:
    swap = commission
    commission = 0
```

**Descripción:** Comisiones muy altas (>$1000) en realidad son swaps/financing costs, no fees de broker.

**Impacto si falta:**
- Costos de financiamiento mezclados con comisiones
- Métricas de comisión infladas artificialmente
- Análisis de costos de trading incorrecto

**Nivel de Criticidad:** 🟠 **MEDIA** - Afecta reportes financieros

**Recomendación de implementación:**
```python
# En flexquery.py, después de calcular commission
.with_columns([
    pl.when(pl.col("commission") > 1000)
      .then(pl.lit(0.0))
      .otherwise(pl.col("commission"))
      .alias("commission"),
    pl.when(pl.col("commission") > 1000)
      .then(pl.col("commission"))
      .otherwise(pl.lit(0.0))
      .alias("swap")
])
```

**Ubicación sugerida:** `flexquery.py:223` (reemplazar el swap hardcoded a 0)

---

#### ⚠️ 13. Commission Currency Conversion con Pip Values
**Ubicación Legacy:** `brokers_ib.py:844-856`
```python
if commission_currency != user_base_currency:
    pip_value = get_daily_pip_value(commission_currency, user_base_currency, trade_date)
    if commission_currency == "USD":
        pip_value = 1  # Override for USD
    commission = commission * pip_value
```

**Descripción:** La nueva implementación usa `fxRateToBase` directamente, pero el legacy tiene lógica especial para pip values de forex.

**Impacto si falta:**
- Comisiones en currencies exóticas mal convertidas
- Pequeñas discrepancias en P&L (centavos acumulados)

**Nivel de Criticidad:** 🟡 **MEDIA-BAJA** - Precisión financiera pero no crítico

**Recomendación:** La nueva implementación parece suficiente con `fxRateToBase`, pero verificar con casos de test de currencies exóticas.

---

### CATEGORÍA 4: VALIDACIONES DE ASSET CLASS [CRITICIDAD: MEDIA]

#### ⚠️ 14. Asset Type Whitelist
**Ubicación Legacy:** `brokers_ib.py:452-453`
```python
allowed_types = ['STK', 'FUT', 'CASH', 'OPT', 'FOP', 'WAR', 'FXCFD', 'CFD', 'BILL', 'EVENT', 'BOND', 'CMDTY']
if asset_type not in allowed_types:
    continue  # Skip unrecognized asset types
```

**Descripción:** Solo se permiten asset types conocidos. Nuevos tipos sin soporte deben ser rechazados explícitamente.

**Impacto si falta:**
- Asset types desconocidos procesados incorrectamente
- Errores silenciosos en nuevos productos de IB
- Datos corruptos por mal mapeo de fields

**Nivel de Criticidad:** 🟠 **MEDIA** - Prevención de errores futuros

**Recomendación de implementación:**
```python
# En flexquery.py, después de mapping de asset
ALLOWED_ASSETS = {"stocks", "options", "futures", "forex", "crypto", "cfd"}
.filter(pl.col("asset").is_in(ALLOWED_ASSETS))
```

**Ubicación sugerida:** `flexquery.py:189` (después de asset mapping)

---

#### ⚠️ 15. IND (Index) → STK Mapping
**Ubicación Legacy:** Implícito en asset mapping
**Nueva Implementación:** `flexquery.py:59` ya lo tiene: `"IND": "stocks"`

**Estado:** ✅ **YA IMPLEMENTADO** correctamente

---

### CATEGORÍA 5: MANEJO DE DUPLICADOS [CRITICIDAD: BAJA-MEDIA]

#### 💡 16. Duplicate Detection con File Row Check
**Ubicación Legacy:** `brokers_ib.py:363-368`
```python
if hashlib.md5(file_row_input.encode('utf-8')).hexdigest() in previousfilerow:
    continue  # Skip already imported trades
```

**Descripción:** El legacy verifica contra `previousfilerow` (trades ya guardados) para evitar reimportaciones.

**Impacto si falta:**
- Duplicados en caso de reimportación del mismo archivo
- Usuarios pueden ver trades duplicados

**Nivel de Criticidad:** 🟡 **MEDIA-BAJA** - Afecta UX pero DB constraints deberían prevenir

**Recomendación:** Verificar que este check se hace en capa superior (orquestador de sync). Si no, añadir:
```python
# En orquestador, antes de llamar normalize()
existing_hashes = fetch_existing_file_row_hashes(user_id, broker)
df = df.filter(~pl.col("file_row_hash").is_in(existing_hashes))
```

**Ubicación sugerida:** No en normalizer, sino en capa de persistencia

---

### CATEGORÍA 6: MANEJO DE CASOS ESPECIALES [CRITICIDAD: BAJA]

#### 💡 17. TRADECANCEL Reversal
**Ubicación Legacy:** `brokers_ib.py:386-388`
```python
if action == 'TRADECANCEL':
    if original_action == 'BUY':
        action = 'SELL'
    else:
        action = 'BUY'
```

**Descripción:** TRADECANCEL invierte la acción original (una compra cancelada se convierte en venta).

**Impacto si falta:**
- Cancelaciones procesadas con dirección incorrecta
- Posiciones no se revierten correctamente

**Nivel de Criticidad:** 🟡 **MEDIA-BAJA** - Poco frecuente pero importante cuando ocurre

**Recomendación de implementación:**
```python
# En flexquery.py, añadir lógica de reversal
.with_columns([
    pl.when(pl.col("buySell") == "TRADECANCEL")
      .then(
          pl.when(pl.col("openCloseIndicator") == "O")  # O=Open, C=Close
            .then(pl.lit("SELL"))
            .otherwise(pl.lit("BUY"))
      )
      .otherwise(pl.col("side"))
      .alias("side")
])
```

**Ubicación sugerida:** `flexquery.py:173` (en mapeo de side)

---

#### 💡 18. Assignment Trades (Notes='A') Special Handling
**Ubicación Legacy:** `brokers_ib.py:430-441`
```python
if n_val['notes'] == 'A':  # Assignment
    price = float(n['closePrice']) if 'closePrice' in n else 0
    size = multiplier  # Use multiplier as size for assignments
```

**Descripción:** Options assignments tienen lógica especial de precio y cantidad.

**Impacto si falta:**
- Assignments con precio/cantidad incorrectos
- P&L de opciones asignadas mal calculado

**Nivel de Criticidad:** 🟡 **MEDIA-BAJA** - Afecta subset pequeño de trades

**Recomendación de implementación:**
```python
# En flexquery.py, añadir check de notes field
.with_columns([
    pl.when(pl.col("notes") == "A")
      .then(pl.col("closePrice").fill_null(pl.col("price")))
      .otherwise(pl.col("price"))
      .alias("price"),
    pl.when(pl.col("notes") == "A")
      .then(pl.col("multiplier"))
      .otherwise(pl.col("quantity"))
      .alias("quantity")
])
```

**Ubicación sugerida:** `flexquery.py:215-216` (después de price/quantity casting)

---

### CATEGORÍA 7: OPCIONES Y FUTUROS [CRITICIDAD: BAJA]

#### 💡 19. Option Strike Scaling (/1000)
**Ubicación Legacy:** `brokers_ib.py:1239-1248`
```python
if explicit_strike:
    strike = explicit_strike
else:
    strike = extracted_strike / 1000  # IB reports in cents
```

**Descripción:** Cuando el strike no viene explícito, IB lo reporta en centavos y debe dividirse por 1000.

**Impacto si falta:**
- Strikes incorrectos (e.g., 45000 en vez de 45.00)
- Opciones no matchean con market data
- Greeks calculation breaks

**Nivel de Criticidad:** 🟡 **MEDIA-BAJA** - Solo afecta visualización de opciones

**Recomendación de implementación:**
```python
# En flexquery.py, verificar si strike viene en formato correcto
.with_columns([
    pl.when((pl.col("asset") == "options") & (pl.col("strike") > 1000))
      .then(pl.col("strike") / 1000)
      .otherwise(pl.col("strike"))
      .alias("option_strike")
])
```

**Ubicación sugerida:** `flexquery.py:244` (después de option_strike casting)

**NOTA:** Verificar primero si IB FlexQuery ya lo reporta escalado correctamente.

---

#### 💡 20. Futures Expiry Date Parsing (Monthly Codes)
**Ubicación Legacy:** `brokers_ib.py:1423-1434`
```python
month_codes = {'JAN':'F', 'FEB':'G', 'MAR':'H', ..., 'DEC':'Z'}
# Extrae codigo de mes del symbol (e.g., ESH24 → H → MAR)
```

**Descripción:** Futures usan códigos de mes estándar (F=Jan, G=Feb, etc.) que deben parsearse.

**Impacto si falta:**
- Expiry dates incorrectos para futures
- No se pueden calcular días hasta vencimiento
- Rollover detection no funciona

**Nivel de Criticidad:** 🟢 **BAJA** - IB FlexQuery ya incluye expiry explícito en campo `expiry`

**Recomendación:** No necesario si FlexQuery incluye expiry date explícito. Verificar con datos de test.

---

### CATEGORÍA 8: VALIDACIONES DE ENCODING [CRITICIDAD: BAJA]

#### 💡 21. ASCII-Only Enforcement
**Ubicación Legacy:** `brokers_ib.py:88-89`
```python
content_b = ''.join([i if ord(i) < 128 else ' ' for i in content])
content_b = re.sub(r'[^\x00-\x7F]+', ' ', content_b)
```

**Descripción:** Elimina caracteres no-ASCII para prevenir errores de parsing.

**Impacto si falta:**
- Errores de encoding con símbolos internacionales
- XML parsing failures en casos raros

**Nivel de Criticidad:** 🟢 **BAJA** - Python 3 maneja UTF-8 correctamente

**Recomendación:** No necesario en nueva implementación con XML parsing moderno.

---

### CATEGORÍA 9: VALIDACIONES DE TIMEZONE [CRITICIDAD: BAJA]

#### 💡 22. Timezone Normalization a America/New_York
**Ubicación Legacy:** Implícito en date parsing
**Nueva Implementación:** No especifica timezone

**Descripción:** IB reporta en Eastern Time, todos timestamps deben normalizarse a America/New_York.

**Impacto si falta:**
- Timestamps en UTC o timezone del servidor
- Discrepancias de horas en trades cerca de market open/close
- Reportes diarios con cortes de tiempo incorrectos

**Nivel de Criticidad:** 🟡 **MEDIA-BAJA** - Afecta precisión temporal

**Recomendación de implementación:**
```python
# En flexquery.py, después de datetime parsing
.with_columns([
    pl.col("timestamp").dt.replace_time_zone("America/New_York")
])
```

**Ubicación sugerida:** `flexquery.py:165` (después de str.to_datetime)

**NOTA:** Verificar si IB FlexQuery ya reporta en ET o UTC.

---

### CATEGORÍA 10: VALIDACIONES ARQUITECTURALES [CRITICIDAD: INFORMATIVA]

#### ℹ️ 23. Multi-Account Handling
**Ubicación Legacy:** `brokers_ib.py:103-116`
```python
accounts_in_file = list(set([rec['account'] for rec in records]))
for account in accounts_in_file:
    account_records = [r for r in records if r['account'] == account]
    # Process each account separately
```

**Descripción:** El legacy procesa múltiples accounts en un solo archivo por separado.

**Impacto si falta:**
- Trades de múltiples accounts mezclados
- Portfolio assignment puede fallar

**Nivel de Criticidad:** 🟢 **BAJA** - Probablemente manejado en capa superior

**Recomendación:** Verificar que el orquestador separa accounts antes de llamar normalize() o añadir:
```python
# En orquestador
accounts = df["account_id"].unique()
for account in accounts:
    account_df = df.filter(pl.col("account_id") == account)
    normalized = normalizer.normalize(account_df)
```

---

## Resumen de Prioridades

### 🔴 CRÍTICO (Implementar INMEDIATAMENTE) - 5 validaciones
1. ✅ Quantity zero check
2. ✅ Symbol vacío check
3. ✅ Price zero validation con fallback
4. ✅ Action BUY/SELL validation
5. ✅ Timestamp null check

### 🟠 ALTA (Implementar en Sprint Actual) - 6 validaciones
6. ✅ Bitcoin symbol cleanup
7. ✅ Hong Kong stocks formatting
8. ✅ CFD symbol formatting
9. ✅ Forex symbol formatting
10. ✅ Futures symbol formatting
11. ✅ Swap detection (high commission)

### 🟡 MEDIA (Implementar en Próximo Sprint) - 6 validaciones
12. ✅ SPXW → SPX conversion
13. ✅ Asset type whitelist
14. ✅ TRADECANCEL reversal
15. ✅ Assignment trades handling
16. ✅ Option strike scaling
17. ✅ Timezone normalization

### 🟢 BAJA (Evaluar Necesidad) - 6 validaciones
18. Commission currency pip values (verificar si fxRateToBase es suficiente)
19. Duplicate detection (verificar si se hace en capa superior)
20. Futures expiry parsing (verificar si FlexQuery lo incluye)
21. ASCII-only enforcement (no necesario en Python 3)
22. Multi-account handling (verificar orquestador)

---

## Archivos Críticos a Modificar

### `brokers/interactive_brokers/flexquery.py`
**Líneas a modificar:**
- `165`: Añadir timezone + null check en timestamp
- `173`: Añadir validation de side BUY/SELL
- `178-179`: Añadir transformaciones de symbol (Bitcoin, HK, CFD, Forex, Futures, SPXW)
- `189`: Añadir asset type whitelist
- `215-216`: Añadir validations de price/quantity zero + assignment handling
- `223`: Modificar swap logic (high commission detection)
- `244`: Añadir option strike scaling check

### Tests a Añadir en `tests/brokers/test_interactive_brokers.py`
- Test de quantity=0 rejection
- Test de symbol="" rejection
- Test de price=0 handling
- Test de action inválido rejection
- Test de timestamp null rejection
- Test de symbol transformations (Bitcoin, HK, CFD, Forex, Futures)
- Test de swap detection
- Test de asset type whitelist
- Test de TRADECANCEL reversal
- Test de assignment trades
- Test de option strike scaling
- Test de timezone normalization

---

## Estrategia de Verificación

### 1. Validación con Datos Históricos
```bash
# Comparar outputs de legacy vs nuevo con mismo input
python compare_normalizers.py --user=49186 --trades=57430
```

### 2. Test de Regresión
- Cargar 10,000 trades históricos de IB
- Pasar por ambos normalizers
- Comparar field-by-field:
  - ✅ Symbol format
  - ✅ Side (BUY/SELL)
  - ✅ Quantity (absolute value)
  - ✅ Price (handling zeros)
  - ✅ Commission/Fees/Swap
  - ✅ Asset type classification
  - ✅ Timestamps

### 3. Edge Cases a Probar
- Trades con quantity=0
- Trades con symbol=""
- Trades con price=0 (con y sin closePrice)
- Bitcoin stocks (symbol="BITCOINUSD")
- Hong Kong stocks (symbol="0700")
- CFDs con mismo symbol que stock
- Forex pairs (EUR.USD → $EURUSD)
- Futures (/ESH24)
- SPXW options
- Comisiones >$1000 (swap detection)
- TRADECANCEL transactions
- Options assignments (notes='A')
- Multi-currency trades

---

## Implementación Incremental Sugerida

### Fase 1: Validaciones Críticas (Sprint 1)
- Quantity zero
- Symbol vacío
- Price zero
- Action validation
- Timestamp validation

### Fase 2: Symbol Transformations (Sprint 1)
- Bitcoin cleanup
- Hong Kong stocks
- CFD suffix
- Forex formatting
- Futures prefix

### Fase 3: Business Logic (Sprint 2)
- Swap detection
- SPXW conversion
- Asset whitelist
- TRADECANCEL handling
- Assignment trades

### Fase 4: Fine-Tuning (Sprint 2)
- Option strike scaling
- Timezone normalization
- Commission currency (si necesario)

---

## Riesgos Identificados

### ✋ Riesgo Alto
- **Sin validación de quantity=0**: Trades fantasma en producción
- **Sin validación de symbol vacío**: Datos no utilizables en DB
- **Price=0 sin fallback**: P&L calculations rotos

### ⚠️ Riesgo Medio
- **Symbol transformations faltantes**: UX degradada para ciertos asset types
- **Swap detection faltante**: Métricas de comisión infladas

### 💡 Riesgo Bajo
- **Timezone no normalizado**: Pequeñas discrepancias en reportes diarios
- **Option strike scaling**: Solo visualización afectada

---

## Conclusiones

La nueva implementación es **arquitecturalmente superior** (modular, testeable, mantenible) pero le faltan **23 validaciones** del legacy, de las cuales **11 son críticas o altas**.

**Acción Recomendada:** Implementar las 11 validaciones prioritarias (🔴🟠) en los próximos 2 sprints para alcanzar paridad funcional con legacy sin comprometer la nueva arquitectura.
