# Archivos de Código - Coinbase Broker

## Estructura de Archivos

```
brokers/coinbase/
├── coinbase.py.original      # Código actual (sin modificar) - 324 líneas
└── README.md                 # Este archivo
```

**Nota:** Coinbase no tiene detector.py separado (a diferencia de otros brokers).

---

## coinbase.py.original

**Descripción:** Código actual del normalizador Coinbase  
**Líneas:** 324  
**Estado:** REFERENCIA - No modificado

### Validaciones Actuales

**✅ YA IMPLEMENTADAS:**
- Detección de formato JSON (can_handle)
- Status == "FILLED" filtering
- Zero-fill rejection (filled_size == 0)
- Hash computation (MD5 de order_id) - 100% match rate
- Timezone conversion (UTC → America/New_York)
- Chronological sorting (CRÍTICO para position calc)
- Type-safe numeric conversions
- Symbol transformation (removes hyphens)
- Asset mapping (SPOT → crypto)

**❌ FALTANTES (a implementar):**
- Validación de product_id no vacío
- Validación de side válido (BUY/SELL)
- Validación de price > 0
- Fee absolute value
- Price rounding (6 decimales)

---

## Cambios Propuestos por Fase

### FASE 1: Validaciones Críticas

#### Cambio 1: Field Validation Gate
**Ubicación:** Línea ~117

**Antes:**
```python
# Solo valida status y filled_size
if status != "FILLED":
    continue
if float(filled_size or 0) == 0:
    continue
```

**Después:**
```python
# Además valida product_id, side, price
if not product_id:
    logger.warning(f"Skipping order {order_id}: missing product_id")
    continue

if not side or side.upper() not in ("BUY", "SELL"):
    logger.warning(f"Skipping order {order_id}: invalid side")
    continue

try:
    if float(average_filled_price or 0) <= 0:
        logger.warning(f"Skipping order {order_id}: invalid price")
        continue
except (ValueError, TypeError):
    logger.warning(f"Skipping order {order_id}: non-numeric price")
    continue
```

---

#### Cambio 2: Fee Absolute Value
**Ubicaciones:** Líneas 152, 262

**Antes:**
```python
# Línea 152
"total_fees": float(order.get("total_fees", 0) or 0),

# Línea 262
pl.col("total_fees").alias("fees"),
```

**Después:**
```python
# Línea 152
"total_fees": abs(float(order.get("total_fees", 0) or 0)),

# Línea 262
pl.col("total_fees").abs().alias("fees"),
```

---

### FASE 2: Calidad de Datos

#### Cambio 3: Price Rounding
**Ubicación:** Línea 248

**Antes:**
```python
pl.col("average_filled_price").alias("price"),
```

**Después:**
```python
pl.col("average_filled_price").round(6).alias("price"),
```

---

### FASE 3: Features Condicionales (SI NECESARIO)

#### Cambio 4: Futures Support (Condicional)
**Condición:** Si usuarios operan Coinbase futures

**Añadir:**
- Update `SUPPORTED_ASSETS` a {"SPOT", "FUTURE"}
- Update `PRODUCT_TYPE_MAP`
- Contract size scaling logic
- Symbol parsing para futures

---

#### Cambio 5: Currency Conversion (Condicional)
**Condición:** Si >5% de orders son non-USD currencies

**Añadir:**
- Forex rate lookup
- Price/fee conversion logic
- Currency field tracking

---

#### Cambio 6: Portfolio Routing (Condicional)
**Condición:** Si usuarios quieren portfolios separados

**Añadir:**
- Category-based portfolio assignment
- account_id routing logic

---

## Implementación

### Para implementar cambios:

1. **Fase 1 (Crítica):**
   ```bash
   # Copiar archivo original
   cp coinbase.py.original ../../brokers/coinbase/coinbase.py
   
   # Aplicar cambios 1, 2
   # Editar coinbase.py según ejemplos arriba
   ```

2. **Fase 2 (Calidad):**
   ```bash
   # Aplicar cambio 3
   ```

3. **Fase 3 (Condicional):**
   ```bash
   # Ejecutar queries de decisión primero
   # Solo implementar si datos lo justifican
   ```

4. **Testing:**
   ```bash
   # Tests unitarios
   pytest ../../tests/brokers/test_coinbase.py -v
   
   # Validación QA (si existe script)
   python qa/validate_broker.py --broker-id 163
   ```

---

## Métricas Actuales

### Performance
- **Hash Match Rate:** 100% (3,377/3,377 records)
- **Code Size:** 324 líneas (vs 1,725 legacy = 81% reducción)
- **Test Coverage:** 496 líneas de tests (ratio 1.53:1)

### Validaciones
- **Status Filtering:** ✅ Implementado
- **Zero-Fill Rejection:** ✅ Implementado
- **Hash Computation:** ✅ 100% compatible
- **Timezone Handling:** ✅ Implementado
- **Chronological Sorting:** ✅ CRÍTICO implementado
- **Field Validation:** ⚠️ Parcial (falta product_id/side/price)
- **Fee Absolute:** ❌ Faltante
- **Price Rounding:** ❌ Faltante

---

## Arquitectura

### BaseInterpreter Pattern
```python
class CoinbaseInterpreter(BaseInterpreter):
    BROKER_ID: ClassVar[str] = "coinbase"
    FORMAT_VERSION: ClassVar[str] = "1.0"
    SUPPORTED_ASSETS: ClassVar[Set[str]] = {"SPOT"}
    PRODUCT_TYPE_MAP: ClassVar[dict] = {"SPOT": "crypto"}
```

### Data Flow
```
JSON String
→ parse_json_content() [filtering, sorting, hashing]
→ DataFrame [raw parsed data]
→ normalize() [LazyFrame transformations]
→ Normalized DataFrame [matches schema]
```

### Hash Formula (Legacy Compatible)
```python
# MD5 de order_id solamente (fallback hash)
file_row_hash = hashlib.md5(
    json.dumps(order_id).encode('utf-8')
).hexdigest()
```

---

## Referencias

- **Plan completo:** `../PLAN_ANALISIS_VALIDACIONES_COINBASE.md`
- **Log de cambios:** `../CAMBIOS_IMPLEMENTADOS.md`
- **Ejemplos de código:** `../EJEMPLOS_CAMBIOS_CODIGO.md`
- **Código legacy:**
  - `../old_code_from_legacy/coinbase_export.py` (623 líneas)
  - `../old_code_from_legacy/brokers_coinbase.py` (212 líneas)
  - `../old_code_from_legacy/coinbasev2_export.py` (890 líneas)

---

**Estado Actual:** ✅ CÓDIGO REFERENCIA COMPLETO  
**Siguiente Paso:** Implementar Fase 1 (validaciones críticas)
