# Ejemplos de Cambios de Código - Coinbase

Este documento muestra ejemplos concretos con código antes/después para cada validación propuesta.

---

## FASE 1: Validaciones Críticas

### Cambio 1: Field Validation Gate

**Archivo:** `brokers/coinbase/coinbase.py`  
**Línea:** ~117  
**Criticidad:** 🔴 ALTA

#### Código Actual (ANTES)
```python
# Líneas 109-116
status = order.get("status", "")
if status != "FILLED":
    continue

try:
    if float(filled_size or 0) == 0:
        continue
except (ValueError, TypeError):
    continue

# Continúa sin validar product_id, side, price...
```

#### Código Propuesto (DESPUÉS)
```python
# Líneas 109-140 (expandido)
status = order.get("status", "")
if status != "FILLED":
    continue

try:
    if float(filled_size or 0) == 0:
        continue
except (ValueError, TypeError):
    continue

# NUEVO: Validación de campos críticos
product_id = order.get("product_id", "")
side = order.get("side", "")
average_filled_price = order.get("average_filled_price", "0")

# Validar product_id no vacío
if not product_id:
    logger.warning(f"Skipping order {order_id}: missing product_id")
    continue

# Validar side válido
if not side or side.upper() not in ("BUY", "SELL"):
    logger.warning(f"Skipping order {order_id}: invalid side '{side}'")
    continue

# Validar price válido
try:
    price_float = float(average_filled_price or 0)
    if price_float <= 0:
        logger.warning(f"Skipping order {order_id}: invalid price {average_filled_price}")
        continue
except (ValueError, TypeError):
    logger.warning(f"Skipping order {order_id}: non-numeric price {average_filled_price}")
    continue
```

#### Justificación
- **Legacy:** `brokers_coinbase.py:104-105`
- **Impacto:** Previene data corruption por campos vacíos
- **Casos rechazados:** product_id vacío, side inválido, price ≤ 0

#### Tests Requeridos
```python
def test_field_validation_missing_product_id():
    """Verifica que orders sin product_id son rechazadas"""
    content = {
        "orders": [{
            "order_id": "test-1",
            "product_id": "",  # Vacío
            "side": "BUY",
            "filled_size": "1.0",
            "average_filled_price": "50000.0",
            "created_time": "2025-01-01T00:00:00Z",
            "status": "FILLED"
        }]
    }
    
    result = CoinbaseInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_field_validation_invalid_side():
    """Verifica que sides inválidos son rechazados"""
    content = {
        "orders": [{
            "order_id": "test-1",
            "product_id": "BTC-USD",
            "side": "INVALID",  # Inválido
            "filled_size": "1.0",
            "average_filled_price": "50000.0",
            "created_time": "2025-01-01T00:00:00Z",
            "status": "FILLED"
        }]
    }
    
    result = CoinbaseInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado

def test_field_validation_zero_price():
    """Verifica que prices zero son rechazados"""
    content = {
        "orders": [{
            "order_id": "test-1",
            "product_id": "BTC-USD",
            "side": "BUY",
            "filled_size": "1.0",
            "average_filled_price": "0",  # Zero
            "created_time": "2025-01-01T00:00:00Z",
            "status": "FILLED"
        }]
    }
    
    result = CoinbaseInterpreter.parse_json_content(content)
    assert len(result) == 0  # Rechazado
```

---

### Cambio 2: Fee Absolute Value

**Archivo:** `brokers/coinbase/coinbase.py`  
**Líneas:** 152, 262  
**Criticidad:** 🟡 MEDIA-ALTA

#### Código Actual (ANTES)
```python
# Línea 152 (en parse_json_content)
"total_fees": float(order.get("total_fees", 0) or 0),

# Línea 262 (en normalize)
pl.col("total_fees").alias("fees"),
```

#### Código Propuesto (DESPUÉS)
```python
# Línea 152 (en parse_json_content)
"total_fees": abs(float(order.get("total_fees", 0) or 0)),

# Línea 262 (en normalize)
pl.col("total_fees").abs().alias("fees"),
```

#### Justificación
- **Legacy:** `brokers_coinbase.py:135-137`
- **Impacto:** Fees negativos confunden usuarios y afectan P&L
- **Casos corregidos:** API devuelve fees como débitos (-0.05 → 0.05)

#### Ejemplo de Datos
```json
// Caso 1: Fee negativo (debit notation)
{
  "order_id": "abc123",
  "total_fees": "-0.05",
  "fees_output": 0.05  // <- Convertido a positivo
}

// Caso 2: Fee positivo (sin cambio)
{
  "order_id": "abc456",
  "total_fees": "0.05",
  "fees_output": 0.05  // <- Sin cambio
}
```

#### Tests Requeridos
```python
def test_fee_absolute_value_negative():
    """Verifica conversión de fees negativos a positivos"""
    content = {
        "orders": [{
            "order_id": "test-1",
            "product_id": "BTC-USD",
            "side": "BUY",
            "filled_size": "1.0",
            "average_filled_price": "50000.0",
            "total_fees": "-0.05",  # Negativo
            "created_time": "2025-01-01T00:00:00Z",
            "status": "FILLED"
        }]
    }
    
    result = CoinbaseInterpreter.parse_json_content(content)
    assert result[0]["total_fees"] == 0.05  # Convertido a positivo

def test_fee_absolute_value_positive():
    """Verifica que fees positivos no cambian"""
    content = {
        "orders": [{
            "order_id": "test-1",
            "product_id": "BTC-USD",
            "side": "BUY",
            "filled_size": "1.0",
            "average_filled_price": "50000.0",
            "total_fees": "0.05",  # Ya positivo
            "created_time": "2025-01-01T00:00:00Z",
            "status": "FILLED"
        }]
    }
    
    result = CoinbaseInterpreter.parse_json_content(content)
    assert result[0]["total_fees"] == 0.05  # Sin cambio
```

---

## FASE 2: Calidad de Datos

### Cambio 3: Price Rounding

**Archivo:** `brokers/coinbase/coinbase.py`  
**Línea:** 248  
**Criticidad:** 🟡 MEDIA

#### Código Actual (ANTES)
```python
# Línea 248
pl.col("average_filled_price").alias("price"),
```

#### Código Propuesto (DESPUÉS)
```python
# Línea 248
pl.col("average_filled_price").round(6).alias("price"),
```

#### Justificación
- **Legacy:** `brokers_coinbase.py:118-121` (detección dinámica de decimales)
- **Impacto:** Consistencia en precisión de display
- **Simplificación:** Fijo 6 decimales en vez de detección dinámica

#### Ejemplo de Datos
```python
# Antes (sin redondeo)
price_before = 50000.123456789  # 9 decimales

# Después (redondeado)
price_after = 50000.123457  # 6 decimales
```

#### Tests Requeridos
```python
def test_price_rounding_six_decimals():
    """Verifica que precios se redondean a 6 decimales"""
    df = pl.DataFrame({
        "order_id": ["test-1"],
        "product_id": ["BTC-USD"],
        "side": ["BUY"],
        "filled_size": [1.0],
        "average_filled_price": [50000.123456789],  # 9 decimales
        "total_fees": [0.05],
        "created_time": ["2025-01-01T00:00:00Z"],
        "_file_row_hash": ["abc123"],
        "_original_file_row": ['{}'],
        "_row_index": [0]
    })
    
    result = CoinbaseInterpreter().normalize(df.lazy(), user_id=1, account_id=1).collect()
    
    # Precio redondeado a 6 decimales
    assert result["price"][0] == 50000.123457
```

---

## Resumen de Cambios por Archivo

### brokers/coinbase/coinbase.py
- **Línea ~117-140:** Field validation gate (Cambio 1)
- **Línea 152:** Fee abs() en parsing (Cambio 2)
- **Línea 248:** Price rounding (Cambio 3)
- **Línea 262:** Fee abs() en normalization (Cambio 2)

### tests/brokers/test_coinbase.py
- Añadir ~10 funciones de test nuevas
- Coverage de Fases 1 y 2

---

## Orden de Implementación

1. ✅ **Cambio 1:** Field validation + tests (Fase 1)
2. ✅ **Cambio 2:** Fee absolute value + tests (Fase 1)
3. ✅ **Validar Fase 1:** Tests passing, match rate ≥ baseline
4. ✅ **Cambio 3:** Price rounding + tests (Fase 2)
5. ✅ **Validar Fase 2:** Tests passing, consistencia verificada
6. ⚠️ **Decisiones:** Ejecutar queries para Fase 3

---

**Nota:** Todos los cambios mantienen compatibilidad con arquitectura existente y usan operaciones Polars para performance óptimo.
