"""
Charles Schwab format detector.

Detects Schwab API JSON format and returns appropriate interpreter.
"""

import polars as pl
from typing import Optional

from pipeline.p01_normalize.exceptions import UnknownFormatError
from .schwab import SchwabInterpreter


INTERPRETERS = [
    SchwabInterpreter,
]


def detect(df: pl.DataFrame, metadata: Optional[dict] = None) -> SchwabInterpreter:
    """
    Detect the Schwab format and return the appropriate interpreter.

    Args:
        df: DataFrame to analyze
        metadata: Optional metadata about the data source

    Returns:
        Appropriate interpreter instance

    Raises:
        UnknownFormatError: If no format matches
    """
    metadata = metadata or {}

    # Sort by priority (highest first)
    sorted_interpreters = sorted(
        INTERPRETERS,
        key=lambda x: -x.get_priority()
    )

    for interpreter_cls in sorted_interpreters:
        if interpreter_cls.can_handle(df, metadata):
            return interpreter_cls()

    raise UnknownFormatError(
        f"No Charles Schwab format matched the data. "
        f"Available columns: {df.columns}"
    )
