# Guía de Implementación - Charles Schwab Normalizer

Este documento proporciona una guía detallada para implementar las validaciones faltantes en el Charles Schwab normalizer, organizada por fases.

**Broker:** charles_schwab
**Archivo Principal:** `schwab.py`
**Fecha:** 2026-01-14

---

## Visión General del Proyecto

### Estado Actual

**Implementación:**
- Archivo principal: `schwab.py` (521 líneas)
- Tests: `test_charles_schwab.py` (504 líneas)
- Formato: JSON API (Equity, Options, ETF, Futures)
- Características: CUSIP resolution, multi-asset, transferItems structure

**Métrica Crítica:**
- **Hash match rate actual:** 42% (25,941/62,237 matches) ⚠️
- **Data integrity:** 100% ✓
- **Objetivo:** >= 95% hash match rate

### Issue Crítico Único de Charles Schwab

El campo `closingPrice` en Schwab API es el **precio de mercado EN VIVO** al momento de hacer la llamada API, NO el precio de ejecución histórico. Esto causa que 58% de las ejecuciones tengan hashes diferentes.

**Ejemplo Real:**
```
Trade: SAVA, 2021-09-15, activityId 31607727843
- Mirror (2026-01-11): closingPrice = $2.09 → hash A
- Legacy (2026-01-08): closingPrice = $2.18 → hash B
- TODOS los demás campos: IDÉNTICOS
```

### Arquitectura

```
Pipeline Flow:
p01_normalize → p02_deduplicate → p03_group → p04_calculate → p05_write

Charles Schwab Normalizer (p01_normalize):
├── detector.py - Format detection
├── schwab.py - Main interpreter
│   ├── can_handle() - Column detection
│   ├── parse_json_content() - JSON → DataFrame
│   ├── resolve_cusips() - CUSIP → ticker
│   └── normalize() - Transform to schema
└── __init__.py
```

---

## FASE 1: Critical Hash Fix (1-2 días) ⚠️ URGENTE

**Objetivo:** Corregir 42% hash match rate a 95-100%

**Issue:** closingPrice volatility + transferItems ordering

**Blocker:** Requiere coordinación con legacy team

### Paso 1.1: Implementar closingPrice Zeroing (0.5 días)

**Archivo:** `schwab.py` líneas 305-310

**Ubicación Actual:**
```python
# schwab.py línea ~305
def compute_file_row_hash(order):
    order_for_hash = {k: v for k, v in order.items() if k != 'account_hash'}
    file_row_hash = hashlib.md5(json.dumps(order_for_hash).encode('utf-8')).hexdigest()
    return file_row_hash
```

**Cambio Requerido:**
```python
import copy

def compute_file_row_hash(order):
    """
    Compute MD5 hash for deduplication.

    FIX: Zero out closingPrice (volatile field) antes de hash.
    """
    # Remove post-hash fields
    order_for_hash = {k: v for k, v in order.items() if k != 'account_hash'}

    # Deep copy para no modificar original
    order_copy = copy.deepcopy(order_for_hash)

    # FIX 1: Zero out closingPrice
    for item in order_copy.get('transferItems', []):
        if 'instrument' in item and 'closingPrice' in item['instrument']:
            item['instrument']['closingPrice'] = 0.0

    # (sorting viene en paso 1.2)

    # Compute hash
    file_row_hash = hashlib.md5(json.dumps(order_copy).encode('utf-8')).hexdigest()
    return file_row_hash
```

**Tests:**
```python
# tests/test_charles_schwab.py
def test_file_row_stable_across_closing_price_changes():
    """Verifica que closingPrice no afecta hash"""
    order1 = {"activityId": 123, "transferItems": [{"instrument": {"closingPrice": 150.0}}]}
    order2 = {"activityId": 123, "transferItems": [{"instrument": {"closingPrice": 155.0}}]}

    assert compute_hash(order1) == compute_hash(order2)

def test_file_row_preserves_original():
    """Verifica que original no se modifica"""
    order = {"transferItems": [{"instrument": {"closingPrice": 150.0}}]}
    original = order["transferItems"][0]["instrument"]["closingPrice"]

    compute_hash(order)

    assert order["transferItems"][0]["instrument"]["closingPrice"] == original
```

**Validación:**
- [ ] Tests pasan
- [ ] Original order no modificado
- [ ] Hash es determinístico (mismo input → mismo hash)

---

### Paso 1.2: Implementar transferItems Sorting (0.5 días)

**Archivo:** `schwab.py` líneas 305-310 (continuación)

**Cambio Requerido:**
```python
def compute_file_row_hash(order):
    # ... (deep copy del paso 1.1)

    # FIX 1: Zero out closingPrice (del paso 1.1)
    for item in order_copy.get('transferItems', []):
        if 'instrument' in item and 'closingPrice' in item['instrument']:
            item['instrument']['closingPrice'] = 0.0

    # FIX 2: Sort transferItems para orden determinístico
    def sort_key(item):
        """
        Sort by:
        1. assetType (CURRENCY < EQUITY < OPTION)
        2. feeType (COMMISSION < SEC_FEE < TAF_FEE)
        3. instrumentId (unique ID)
        """
        inst = item.get('instrument', {})
        return (
            inst.get('assetType', 'Z'),      # 'Z' pone al final si missing
            item.get('feeType', 'Z'),
            inst.get('instrumentId', 0)
        )

    if 'transferItems' in order_copy:
        order_copy['transferItems'] = sorted(
            order_copy['transferItems'],
            key=sort_key
        )

    # Compute hash
    file_row_hash = hashlib.md5(json.dumps(order_copy).encode('utf-8')).hexdigest()
    return file_row_hash
```

**Tests:**
```python
def test_file_row_stable_across_transfer_items_order():
    """Verifica que orden de array no afecta hash"""
    equity = {"instrument": {"assetType": "EQUITY", "instrumentId": 100}}
    fee1 = {"instrument": {"assetType": "CURRENCY"}, "feeType": "COMMISSION"}
    fee2 = {"instrument": {"assetType": "CURRENCY"}, "feeType": "SEC_FEE"}

    order1 = {"transferItems": [equity, fee1, fee2]}
    order2 = {"transferItems": [fee2, equity, fee1]}  # Orden diferente

    assert compute_hash(order1) == compute_hash(order2)
```

**Validación:**
- [ ] Tests pasan
- [ ] Hash es idéntico independientemente del orden
- [ ] Sort key cubre todos los casos

---

### Paso 1.3: Actualizar Legacy Code (0.5 días)

**Archivo:** `old_code_from_legacy/schwab_export.py` líneas 952, 968

**Coordinación:**
1. Compartir nueva hash formula con legacy team
2. Legacy debe aplicar mismo fix (closingPrice zeroing + sorting)
3. Code review de ambos lados

**Ubicación Legacy:**
```python
# schwab_export.py línea ~952
file_row = hashlib.md5(json.dumps(row).encode('utf-8')).hexdigest()

# Cambiar a:
file_row = compute_file_row_hash(row)  # Usar misma función
```

**Testing Legacy:**
- [ ] Legacy tests pasan
- [ ] Legacy y mirror producen mismo hash para mismo trade
- [ ] No hay regressions en legacy environment

---

### Paso 1.4: SQL Rebuild Script (0.5 días)

**Archivo:** `scripts/rebuild_schwab_file_row.py` (NUEVO)

**Propósito:** Rebuild hashes de todos los trades existentes de Charles Schwab

**Script:**
```python
#!/usr/bin/env python3
"""
Rebuild file_row hashes for Charles Schwab executions.

Steps:
1. Fetch all executions WHERE broker='CharlesSchwab'
2. Parse original_file_row JSON
3. Apply new hash formula (closingPrice zeroing + sorting)
4. UPDATE file_row column
5. Log progress

Usage:
  python scripts/rebuild_schwab_file_row.py --dry-run  # Test mode
  python scripts/rebuild_schwab_file_row.py --execute  # Production
"""

import json
import hashlib
import copy
import logging
from typing import Dict, Any
import psycopg2
from psycopg2.extras import execute_batch

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


def compute_file_row_hash(order: Dict[str, Any]) -> str:
    """
    Compute hash with new formula.

    IMPORTANT: Must match schwab.py implementation exactly.
    """
    # Remove post-hash fields
    order_for_hash = {k: v for k, v in order.items() if k != 'account_hash'}

    # Deep copy
    order_copy = copy.deepcopy(order_for_hash)

    # FIX 1: Zero out closingPrice
    for item in order_copy.get('transferItems', []):
        if 'instrument' in item and 'closingPrice' in item['instrument']:
            item['instrument']['closingPrice'] = 0.0

    # FIX 2: Sort transferItems
    def sort_key(item):
        inst = item.get('instrument', {})
        return (
            inst.get('assetType', 'Z'),
            item.get('feeType', 'Z'),
            inst.get('instrumentId', 0)
        )

    if 'transferItems' in order_copy:
        order_copy['transferItems'] = sorted(order_copy['transferItems'], key=sort_key)

    # Compute hash
    return hashlib.md5(json.dumps(order_copy).encode('utf-8')).hexdigest()


def rebuild_hashes(dry_run: bool = True):
    """
    Rebuild all Charles Schwab file_row hashes.

    Args:
        dry_run: If True, no database changes are made
    """
    conn = psycopg2.connect(
        host=os.getenv("DB_HOST"),
        database=os.getenv("DB_NAME"),
        user=os.getenv("DB_USER"),
        password=os.getenv("DB_PASSWORD")
    )

    try:
        cursor = conn.cursor()

        # Fetch all Charles Schwab executions
        logger.info("Fetching Charles Schwab executions...")
        cursor.execute("""
            SELECT id, original_file_row, file_row
            FROM executions
            WHERE broker = 'CharlesSchwab'
            ORDER BY id
        """)

        rows = cursor.fetchall()
        total_count = len(rows)
        logger.info(f"Found {total_count} executions")

        # Process each execution
        updates = []
        changed_count = 0
        error_count = 0

        for idx, (exec_id, original_file_row, old_hash) in enumerate(rows):
            try:
                # Parse JSON
                order = json.loads(original_file_row)

                # Compute new hash
                new_hash = compute_file_row_hash(order)

                # Check if changed
                if new_hash != old_hash:
                    changed_count += 1
                    updates.append((new_hash, exec_id))

                # Log progress every 10,000 rows
                if (idx + 1) % 10000 == 0:
                    logger.info(f"Processed {idx + 1}/{total_count} ({changed_count} changed)")

            except Exception as e:
                error_count += 1
                logger.error(f"Error processing execution {exec_id}: {e}")

        logger.info(f"Processing complete:")
        logger.info(f"  Total: {total_count}")
        logger.info(f"  Changed: {changed_count} ({100 * changed_count / total_count:.1f}%)")
        logger.info(f"  Unchanged: {total_count - changed_count}")
        logger.info(f"  Errors: {error_count}")

        if dry_run:
            logger.info("DRY RUN - No database changes made")
        else:
            # Execute batch update
            logger.info("Updating database...")
            execute_batch(
                cursor,
                "UPDATE executions SET file_row = %s WHERE id = %s",
                updates,
                page_size=1000
            )
            conn.commit()
            logger.info(f"Updated {changed_count} rows")

    finally:
        cursor.close()
        conn.close()


if __name__ == "__main__":
    import argparse
    import os

    parser = argparse.ArgumentParser(description="Rebuild Charles Schwab file_row hashes")
    parser.add_argument("--dry-run", action="store_true", help="Test mode (no DB changes)")
    parser.add_argument("--execute", action="store_true", help="Production mode (make DB changes)")
    args = parser.parse_args()

    if not args.dry_run and not args.execute:
        parser.error("Must specify either --dry-run or --execute")

    rebuild_hashes(dry_run=args.dry_run)
```

**Ejecución:**
```bash
# 1. Dry run (test mode)
python scripts/rebuild_schwab_file_row.py --dry-run

# Expected output:
# Found 62,237 executions
# Processing complete:
#   Total: 62,237
#   Changed: 36,296 (58.3%)
#   Unchanged: 25,941
#   Errors: 0
# DRY RUN - No database changes made

# 2. Production (si dry run OK)
python scripts/rebuild_schwab_file_row.py --execute
```

**Validación:**
- [ ] Dry run completa sin errores
- [ ] Changed percentage matches expected (~58%)
- [ ] Production execution exitosa
- [ ] Hash match rate >= 95% post-rebuild

---

### FASE 1: Deployment Checklist

**Pre-Deployment:**
- [ ] Código revisado (code review)
- [ ] Tests pasan (local + CI)
- [ ] Legacy changes coordinadas
- [ ] SQL script probado (dry-run)
- [ ] Backup de database tomado

**Deployment Sequence:**
1. [ ] Deploy legacy changes primero
2. [ ] Validar legacy funciona OK (24 horas)
3. [ ] Deploy mirror changes
4. [ ] Ejecutar SQL rebuild (dry-run)
5. [ ] Ejecutar SQL rebuild (production)
6. [ ] Validar hash match rate >= 95%

**Post-Deployment:**
- [ ] Monitor logs por 48 horas
- [ ] Verificar no hay regressions
- [ ] Documentar lessons learned

**Métricas de Éxito:**
- [ ] Hash match rate >= 95% (vs actual 42%)
- [ ] Data integrity mantenida (100%)
- [ ] Sin cambios user-facing
- [ ] Deduplicación funciona correctamente

---

## FASE 2: Data Validation (1.5-2 días)

**Objetivo:** Prevenir datos inválidos de entrar al pipeline

### Paso 2.1: Required Fields Validation (0.5 días)

**Archivo:** `schwab.py` línea 268 (en `parse_json_content()`)

**Ubicación:** Inicio del loop `for row_idx, order in enumerate(orders):`

**Código:**
```python
# Después de:
for row_idx, order in enumerate(orders):

# Agregar:
    # VALIDACIÓN: Campos requeridos
    activity_id = order.get("activityId")
    account_number = order.get("accountNumber")
    time = order.get("time", "")
    transfer_items = order.get("transferItems", [])

    if not activity_id:
        logger.warning(f"[SCHWAB] Skipping order at index {row_idx}: missing activityId")
        continue

    if not account_number:
        logger.warning(f"[SCHWAB] Skipping order {activity_id}: missing accountNumber")
        continue

    if not time:
        logger.warning(f"[SCHWAB] Skipping order {activity_id}: missing time")
        continue

    if not transfer_items:
        logger.warning(f"[SCHWAB] Skipping order {activity_id}: empty transferItems")
        continue
```

**Tests:** (ver `EJEMPLOS_CAMBIOS_CODIGO.md`)

**Validación:**
- [ ] Tests pasan
- [ ] Warning logged para cada skip
- [ ] Rejection rate < 0.1%

---

### Paso 2.2: Status Filter (0.25 días)

**Archivo:** `schwab.py` línea ~270 (después de required fields)

**Código:**
```python
# Después de required fields validation
# Agregar:
    status = order.get("status", "").upper()
    if status == "INVALID":
        logger.debug(f"[SCHWAB] Skipping order {activity_id}: status is INVALID")
        continue
```

**Tests:** (ver `EJEMPLOS_CAMBIOS_CODIGO.md`)

---

### Paso 2.3: Symbol Validation (0.25 días)

**Archivo:** `schwab.py` línea 351 (después de CUSIP resolution)

**Código:**
```python
# Después de:
resolved_symbol = cusip_to_ticker.get(raw_symbol, raw_symbol)

# Agregar:
if not resolved_symbol or not resolved_symbol.strip():
    logger.warning(
        f"[SCHWAB] Skipping order {activity_id}: empty symbol after CUSIP resolution (raw: {raw_symbol})"
    )
    continue
```

---

### Paso 2.4: Price/Quantity Zero Validation (0.5 días)

**Archivo:** `schwab.py` línea 302 (después de extracting values)

**Código:**
```python
# Después de:
total_amount = sum(float(item.get("amount", 0) or 0) for item in instrument_items)
price = float(first_item.get("price", 0) or 0)

# Agregar:
if price <= 0:
    logger.warning(f"[SCHWAB] Skipping order {activity_id}: zero or negative price {price}")
    continue

if abs(total_amount) <= 0:
    logger.warning(f"[SCHWAB] Skipping order {activity_id}: zero quantity")
    continue
```

---

### Paso 2.5: Disabled Instrument Filter (0.25 días)

**Archivo:** `schwab.py` líneas 277-282 (en transferItems loop)

**Código:**
```python
for item in transfer_items:
    inst = item.get("instrument", {})

    # Agregar:
    if inst.get("status", "").lower() == "disabled":
        logger.debug(f"[SCHWAB] Skipping disabled instrument in order {activity_id}")
        continue

    # Rest of logic...
```

---

### FASE 2: Testing & Validation

**Tests Requeridos:** 13 tests (ver `EJEMPLOS_CAMBIOS_CODIGO.md`)

**Validación:**
- [ ] Todos los tests pasan
- [ ] Zero symbols vacíos en output
- [ ] Zero trades con price zero
- [ ] Zero órdenes INVALID
- [ ] Warnings logged apropiadamente
- [ ] Rejection rate < 0.1% del total

---

## FASE 3: Quality Improvements (1-1.5 días)

### Paso 3.1: CUSIP Retry Logic (1 día)

**Archivo:** `schwab.py` líneas 172-199 (`resolve_cusips()`)

**Cambios:**
1. Agregar `import time` al inicio
2. Definir constantes:
   ```python
   MAX_RETRIES = 3
   RETRY_DELAY = 2  # seconds
   ```
3. Wrap API call en retry loop con exponential backoff
4. Handle HTTP 429 (rate limit)
5. Handle Timeout exceptions

**Código Completo:** (ver `EJEMPLOS_CAMBIOS_CODIGO.md`)

**Tests:** 4 tests para retry logic (ver ejemplos)

---

### Paso 3.2: API Key Security (0.25 días)

**Archivo:** `schwab.py` línea 119

**Cambio:**
```python
# ANTES:
CUSIP_API_KEY: ClassVar[str] = "nkqCXigD6ZeeX5QXcD3xQda3MOLK4zvo"

# DESPUÉS:
import os
CUSIP_API_KEY: ClassVar[str] = os.getenv("CUSIP_API_KEY", "")
```

**En `resolve_cusips()`:**
```python
@classmethod
def resolve_cusips(cls, cusips: List[str]) -> Dict[str, str]:
    # Validar API key configurado
    if not cls.CUSIP_API_KEY:
        logger.error(
            "[SCHWAB] CUSIP_API_KEY not configured. "
            "Set environment variable CUSIP_API_KEY."
        )
        return {cusip: None for cusip in cusips}

    # ... rest of logic
```

**Environment Setup:**
```bash
# .env
CUSIP_API_KEY=nkqCXigD6ZeeX5QXcD3xQda3MOLK4zvo

# Or export:
export CUSIP_API_KEY=nkqCXigD6ZeeX5QXcD3xQda3MOLK4zvo
```

**Documentation:**
Add to README.md:
```markdown
## Environment Variables

- `CUSIP_API_KEY`: Required. API key for financialmodelingprep.com CUSIP resolution.
  Get key at: https://financialmodelingprep.com/developer/docs/
```

---

### FASE 3: Testing & Validation

**Tests Requeridos:** 7 tests (ver `EJEMPLOS_CAMBIOS_CODIGO.md`)

**Métricas de Éxito:**
- [ ] CUSIP resolution success rate >= 98%
- [ ] API key no en source code
- [ ] Transient failures manejados
- [ ] Rate limiting funciona

---

## FASE 4: CSV Support (8-12 días - CONDICIONAL) 🎯

**BLOCKER:** Ejecutar SQL query primero

### Decisión Gate: SQL Query

**Query:**
```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*) OVER (), 2) as percentage,
    MAX(created_at) as last_used
FROM data_sources
WHERE broker_id = 'charles_schwab'
  AND created_at > NOW() - INTERVAL '12 months'
GROUP BY source_type
ORDER BY count DESC;
```

**Criterio de Decisión:**
- **SI CSV > 5%**: IMPLEMENTAR Fase 4 (8-12 días)
- **SI CSV < 5%**: OMITIR Fase 4

**Si OMITIR:**
- Documentar decisión en README
- Agregar comment en código: "CSV parsers omitted - usage < 5%"
- Monitor usage quarterly

**Si IMPLEMENTAR:**
- Crear `csv_parsers.py` con 5 parsers
- Crear tests para cada variant
- Update detector.py para CSV detection
- Estimado: 8-12 días

---

## Resumen de Archivos Modificados

### Archivos a Modificar

| Archivo | Líneas Actuales | Líneas Estimadas | Cambios |
|---------|----------------|------------------|---------|
| `schwab.py` | 521 | ~650 | Fases 1-3 |
| `test_charles_schwab.py` | 504 | ~750 | +23 tests |
| `scripts/rebuild_schwab_file_row.py` | 0 | ~200 | NEW |

### Archivos Nuevos

- `scripts/rebuild_schwab_file_row.py` - SQL rebuild script
- (Condicional) `csv_parsers.py` - CSV parsers si necesario

---

## Testing Strategy

### Unit Tests (23 nuevos)

**Fase 1:** 4 tests (hash stability)
**Fase 2:** 13 tests (validations)
**Fase 3:** 6 tests (retry + security)

### Integration Tests

```python
def test_end_to_end_hash_match_rate():
    """Verifica hash match rate >= 95% con datos reales"""
    # Load sample data (1000 trades)
    # Process with new code
    # Compare hashes con legacy
    # Assert match rate >= 95%

def test_rejection_rate():
    """Verifica rejection rate < 0.1%"""
    # Load sample data
    # Count rejects vs accepted
    # Assert rejection_rate < 0.001
```

### Coverage Target

**Antes:** ~75%
**Después:** >= 90%

---

## Troubleshooting

### Issue: Hash Match Rate Still Low Después de Fase 1

**Síntomas:** Hash match rate < 95% después de deployment

**Debug Steps:**
1. Verificar legacy code tiene mismo fix
2. Comparar hash formula byte-by-byte
3. Check JSON key ordering (no `sort_keys=True`)
4. Verificar deep copy funciona correctamente
5. Sample 10 trades y compare hashes manualmente

**Common Causes:**
- Legacy no aplicó closingPrice zeroing
- Sorting key diferente entre legacy/mirror
- JSON dump tiene flags diferentes

---

### Issue: CUSIP Resolution Failing

**Síntomas:** CUSIP success rate < 98%

**Debug Steps:**
1. Check API key configurado: `echo $CUSIP_API_KEY`
2. Test API manualmente: `curl "https://financialmodelingprep.com/api/v3/cusip/037833100?apikey=..."`
3. Check rate limiting logs
4. Verify retry logic ejecutando

**Common Causes:**
- API key inválido/expirado
- API rate limits alcanzados
- Network issues (timeouts)

---

### Issue: High Rejection Rate

**Síntomas:** Rejection rate > 0.1%

**Debug Steps:**
1. Grep logs para warnings: `grep "Skipping order" logs.txt | wc -l`
2. Categorizar rechazos por tipo
3. Sample rejected orders y verificar si son legítimos

**Common Causes:**
- Validation demasiado estricta
- Bug en validación (e.g., strip() issues)
- Datos de producción tienen formato inesperado

---

## Deployment Checklist Completo

### Pre-Deployment

- [ ] Code review completo (todas las fases)
- [ ] Todos los tests pasan (unit + integration)
- [ ] Coverage >= 90%
- [ ] Documentación actualizada
- [ ] CUSIP_API_KEY configurado en todos los environments

### Fase 1 Deployment (Hash Fix)

- [ ] Backup de database
- [ ] Deploy legacy changes primero
- [ ] Validar legacy funciona (24-48 horas)
- [ ] Deploy mirror changes
- [ ] SQL rebuild dry-run
- [ ] SQL rebuild production
- [ ] Validar hash match rate >= 95%
- [ ] Monitor logs 48 horas

### Fase 2-3 Deployment (Validations + Quality)

- [ ] Deploy en staging environment
- [ ] Smoke tests en staging
- [ ] Validar rejection rate < 0.1%
- [ ] Validar CUSIP success rate >= 98%
- [ ] Deploy en production
- [ ] Monitor logs 48 horas
- [ ] Verificar throughput no afectado

### Post-Deployment

- [ ] Confirmar métricas estables
- [ ] No errores inesperados en logs
- [ ] User reports OK
- [ ] Documentar lessons learned
- [ ] Update runbook si necesario

---

## Métricas de Éxito

### Fase 1 (Hash Fix)
- ✅ Hash match rate >= 95% (actual: 42%)
- ✅ Data integrity = 100%
- ✅ Legacy/mirror hashes idénticos
- ✅ Zero user-facing changes

### Fase 2 (Validations)
- ✅ Zero symbols vacíos en output
- ✅ Zero trades con price/quantity zero
- ✅ Zero órdenes INVALID procesadas
- ✅ Rejection rate < 0.1%

### Fase 3 (Quality)
- ✅ CUSIP success rate >= 98%
- ✅ API key no en source code
- ✅ Transient failures manejados
- ✅ Rate limiting funciona

---

## Referencias

**Documentación:**
- Plan completo: `../PLAN_ANALISIS_VALIDACIONES_CHARLES_SCHWAB.md`
- Tracking log: `../CAMBIOS_IMPLEMENTADOS.md`
- Ejemplos de código: `../EJEMPLOS_CAMBIOS_CODIGO.md`

**Código Legacy:**
- API sync: `../old_code_from_legacy/schwab_export.py`
- CSV parsers: `../old_code_from_legacy/brokers_charlesschwab.py`
- SchwabStreet: `../old_code_from_legacy/brokers_schwabstreet.py`

**External APIs:**
- CUSIP API: https://financialmodelingprep.com/developer/docs/

---

**Última Actualización:** 2026-01-14
**Próxima Revisión:** Post Fase 1 deployment
