# Plan: Análisis de Validaciones Críticas Faltantes - Charles Schwab Normalizer

## Resumen Ejecutivo

Tras analizar exhaustivamente las implementaciones legacy (3,196 líneas en 4 archivos) vs nueva (1,087 líneas), se identificaron **27 categorías de validaciones**, de las cuales **1 es CRÍTICA** (closingPrice hash volatility con 42% match rate) y **5-6 son HIGH priority** para integridad de datos.

**Hallazgo clave**: La nueva implementación es **arquitecturalmente superior** con **66% menos código** (3,196→1,087 líneas). Sin embargo, existe un **issue crítico único** de Charles Schwab:

### Issue Crítico: closingPrice Volatility
- **Hash match rate actual**: **42%** (25,941/62,237 matches)
- **Causa**: Schwab API incluye `closingPrice` (precio de mercado ACTUAL), no precio de ejecución
- **Impacto**: 58% de ejecuciones tienen hashes diferentes entre legacy/mirror
- **Integridad de datos**: 100% correcta (solo afecta deduplicación)

**Ejemplo real**:
```
Trade: SAVA 2021-09-15, activityId 31607727843
- Mirror (2026-01-11): closingPrice = $2.09 → hash A
- Legacy (2026-01-08): closingPrice = $2.18 → hash B
- TODOS los demás campos: IDÉNTICOS
```

**Scope Actual**: JSON API only (no CSV parsers) - Equity, Options, ETF, Futures support

---

## Validaciones Identificadas

### CATEGORÍA 1: CRÍTICO - Hash Deduplication Issues

#### ⚠️⚠️⚠️ 1. closingPrice Hash Volatility (CRÍTICO - DEBE CORREGIRSE)

**Criticidad:** ALTA (afecta 58% de ejecuciones)
**Estado:** PRESENTE EN AMBAS IMPLEMENTACIONES

**Ubicación Actual:**
- Nueva: `schwab.py:309` - Incluye todo el order en hash
- Legacy: `schwab_export.py:952,968` - Mismo comportamiento

**Problema:**
La API de Schwab incluye `transferItems[].instrument.closingPrice` que es el **precio de mercado EN VIVO** al momento de hacer la llamada API, NO el precio de ejecución histórico.

**Evidencia:**
```json
// Mirror sync (2026-01-11)
{
  "activityId": 31607727843,
  "transferItems": [{
    "instrument": {
      "symbol": "SAVA",
      "closingPrice": 2.09  // ← Precio de mercado hoy
    },
    "price": 43.93  // ← Precio de ejecución (2021-09-15)
  }]
}

// Legacy sync (2026-01-08)
{
  "activityId": 31607727843,
  "transferItems": [{
    "instrument": {
      "symbol": "SAVA",
      "closingPrice": 2.18  // ← Precio de mercado hace 3 días
    },
    "price": 43.93  // ← Mismo precio de ejecución
  }]
}

→ Diferentes hashes, pero MISMOS datos de trading
```

**Impacto:**
- Hash match rate: 42% (vs objetivo 95%+)
- No-matching: 36,296 ejecuciones (58%)
- Data integrity: 100% ✓ (datos reales son correctos)
- Solo afecta deduplicación

**Solución Recomendada: Opción A + B (Combinada)**

```python
# schwab.py líneas 305-310 (reemplazar)
import copy

def compute_file_row_hash(order):
    # Remover account_hash (post-hash field)
    order_for_hash = {k: v for k, v in order.items() if k != 'account_hash'}

    # Deep copy para no modificar original
    order_copy = copy.deepcopy(order_for_hash)

    # FIX 1: Zero out closingPrice (volatilidad)
    for item in order_copy.get('transferItems', []):
        if 'instrument' in item and 'closingPrice' in item['instrument']:
            item['instrument']['closingPrice'] = 0.0

    # FIX 2: Sort transferItems (orden no determinístico)
    def sort_key(item):
        inst = item.get('instrument', {})
        return (
            inst.get('assetType', 'Z'),      # CURRENCY < EQUITY < OPTION
            item.get('feeType', 'Z'),         # COMMISSION < SEC_FEE < TAF_FEE
            inst.get('instrumentId', 0)       # ID único
        )

    order_copy['transferItems'] = sorted(
        order_copy.get('transferItems', []),
        key=sort_key
    )

    # Compute hash
    return hashlib.md5(json.dumps(order_copy).encode('utf-8')).hexdigest()
```

**Pros:**
- ✅ Elimina volatilidad de closingPrice
- ✅ Elimina volatilidad de orden de array
- ✅ Hash 100% determinístico
- ✅ No afecta data integrity (closingPrice no se usa en cálculos)
- ✅ Expected match rate: 95-100%

**Cons:**
- ❌ Requiere actualizar legacy code también
- ❌ Requiere rebuild SQL de hashes existentes
- ❌ Deployment coordinado (legacy + mirror + SQL script)

**Alternativa (No Recomendada): Aceptar 42% match rate**
- Pros: No code changes
- Cons: Deduplicación no funciona correctamente

**Archivo:** `schwab.py` líneas 305-310
**Estimado:** 1-2 días (incluye legacy update, testing, SQL rebuild)

**Tests Requeridos:**
```python
def test_file_row_stable_across_closing_price_changes():
    """Verifica que cambios en closingPrice no afectan hash"""
    order1 = {
        "activityId": 123,
        "transferItems": [{
            "instrument": {"symbol": "AAPL", "closingPrice": 150.0}
        }]
    }
    order2 = {
        "activityId": 123,
        "transferItems": [{
            "instrument": {"symbol": "AAPL", "closingPrice": 155.0}  # Diferente
        }]
    }

    hash1 = compute_hash(order1)
    hash2 = compute_hash(order2)
    assert hash1 == hash2  # Mismo hash

def test_file_row_stable_across_transfer_items_order():
    """Verifica que orden de transferItems no afecta hash"""
    order1 = {"transferItems": [fee1, fee2, equity]}
    order2 = {"transferItems": [equity, fee1, fee2]}  # Orden diferente

    hash1 = compute_hash(order1)
    hash2 = compute_hash(order2)
    assert hash1 == hash2  # Mismo hash
```

**SQL Rebuild Script:**
```python
# scripts/rebuild_schwab_file_row.py
def rebuild_all_schwab_hashes():
    """
    Rebuild file_row hashes for all Charles Schwab executions.

    Steps:
    1. Fetch all WHERE broker='CharlesSchwab'
    2. Parse original_file_row JSON
    3. Apply new hash formula
    4. UPDATE file_row column
    5. Log progress
    """
    # Implementation required
```

---

### CATEGORÍA 2: HIGH - Data Validation Issues

#### ⭐⭐⭐ 2. Required Fields Validation (HIGH)

**Estado:** FALTANTE

**Ubicación Legacy:** `schwab_export.py:957-962`
```python
if not ('type' in row and
        row['type'].upper() == "TRADE" and
        'accountNumber' in row and
        'time' in row and row["time"] != '' and
        'transferItems' in row):
    continue
```

**Estado Nuevo:** Sin validación explícita

**Impacto si falta:**
- activityId vacío → hash collisions
- accountNumber vacío → no se puede asignar a usuario
- time vacío → timestamp parsing failures
- transferItems vacío → processing errors

**Implementación:**
```python
# schwab.py después de línea 268 en parse_json_content()
for row_idx, order in enumerate(orders):
    if order.get("type") != "TRADE":
        continue

    # VALIDACIÓN: Campos requeridos
    activity_id = order.get("activityId")
    account_number = order.get("accountNumber")
    time = order.get("time", "")
    transfer_items = order.get("transferItems", [])

    if not activity_id:
        logger.warning(f"Skipping order: missing activityId")
        continue

    if not account_number:
        logger.warning(f"Skipping order {activity_id}: missing accountNumber")
        continue

    if not time:
        logger.warning(f"Skipping order {activity_id}: missing time")
        continue

    if not transfer_items:
        logger.warning(f"Skipping order {activity_id}: empty transferItems")
        continue
```

**Archivo:** `schwab.py` líneas 268-295
**Estimado:** 0.5 días

---

#### ⭐⭐⭐ 3. Status Filter Validation (HIGH)

**Estado:** FALTANTE

**Ubicación Legacy:** `schwab_export.py:964-965`
```python
if 'status' in row and row["status"] == 'INVALID':
    continue
```

**Estado Nuevo:** Sin filtro de status

**Impacto si falta:**
- Órdenes INVALID pueden procesarse
- Órdenes cancelled/rejected contaminan datos

**Implementación:**
```python
# Después de validación de campos requeridos
status = order.get("status", "").upper()
if status == "INVALID":
    logger.debug(f"Skipping order {activity_id}: status is INVALID")
    continue
```

**Archivo:** `schwab.py` línea ~270
**Estimado:** 0.25 días

---

#### ⭐⭐ 4. Symbol Validation (MEDIUM-HIGH)

**Estado:** PARCIALMENTE IMPLEMENTADO

**Ubicación Legacy:**
```python
# schwab_export.py:1120-1121
if not symbol:
    continue
```

**Ubicación Nueva:** `schwab.py:350-351`
```python
raw_symbol = inst.get("symbol", "")
resolved_symbol = cusip_to_ticker.get(raw_symbol, raw_symbol)
# ❌ Sin validación si resolved_symbol está vacío
```

**Implementación:**
```python
# schwab.py después de línea 351
resolved_symbol = cusip_to_ticker.get(raw_symbol, raw_symbol)

# VALIDACIÓN: Symbol vacío
if not resolved_symbol or not resolved_symbol.strip():
    logger.warning(f"Skipping order {activity_id}: empty symbol")
    continue
```

**Archivo:** `schwab.py` línea 351
**Estimado:** 0.25 días

---

#### ⭐⭐ 5. Price/Quantity Zero Validation (MEDIUM-HIGH)

**Estado:** FALTANTE

**Ubicación Legacy:** Implícito en `schwab_export.py:1022-1025`

**Ubicación Nueva:** `schwab.py:300-302`
```python
total_amount = sum(float(item.get("amount", 0) or 0) for item in instrument_items)
price = float(first_item.get("price", 0) or 0)
# ❌ Sin validación de zero
```

**Implementación:**
```python
# schwab.py después de línea 302
price = float(first_item.get("price", 0) or 0)

# VALIDACIÓN: Price zero o negativo
if price <= 0:
    logger.warning(f"Skipping order {activity_id}: zero or negative price {price}")
    continue

# VALIDACIÓN: Quantity zero
if abs(total_amount) <= 0:
    logger.warning(f"Skipping order {activity_id}: zero quantity")
    continue
```

**Archivo:** `schwab.py` línea 302
**Estimado:** 0.5 días

---

#### ⭐⭐ 6. Disabled Instrument Filter (MEDIUM)

**Estado:** FALTANTE

**Ubicación Legacy:** `schwab_export.py:1015,1032,1051`
```python
if instrument.get('status', '').lower() == 'disabled':
    d = True  # Flag to skip
```

**Impacto si falta:**
- Instrumentos delisted pueden procesarse
- Artifacts de corporate actions contaminan datos

**Implementación:**
```python
# schwab.py líneas 277-282 en transferItems loop
inst = item.get("instrument", {})

# VALIDACIÓN: Skip disabled instruments
if inst.get("status", "").lower() == "disabled":
    logger.debug(f"Skipping disabled instrument in order {activity_id}")
    continue
```

**Archivo:** `schwab.py` líneas 277-282
**Estimado:** 0.25 días

---

### CATEGORÍA 3: MEDIUM - Data Quality Improvements

#### ⭐ 7. CUSIP Resolution Error Handling (MEDIUM)

**Estado:** PARCIAL

**Ubicación Actual:** `schwab.py:196-199`
```python
except Exception as e:
    logger.warning(f"[SCHWAB] Failed to resolve CUSIP {cusip}: {e}")
    _cusip_cache[cusip] = None
    # ❌ No retry logic
    # ❌ No rate limiting
```

**Mejoras:**
```python
# schwab.py:172-199
import time

MAX_RETRIES = 3
RETRY_DELAY = 2  # seconds

for cusip in uncached:
    retries = 0
    while retries < MAX_RETRIES:
        try:
            url = f"{cls.CUSIP_API_URL}{cusip}?apikey={cls.CUSIP_API_KEY}"
            response = requests.get(url, timeout=10)

            if response.status_code == 429:  # Rate limit
                wait_time = RETRY_DELAY * (2 ** retries)
                logger.warning(f"[SCHWAB] Rate limited, waiting {wait_time}s")
                time.sleep(wait_time)
                retries += 1
                continue

            # ... rest of logic
            break

        except requests.exceptions.Timeout:
            retries += 1
            if retries < MAX_RETRIES:
                logger.warning(f"[SCHWAB] Timeout resolving {cusip}, retry {retries}/{MAX_RETRIES}")
                time.sleep(RETRY_DELAY)
            else:
                logger.error(f"[SCHWAB] Failed to resolve {cusip} after {MAX_RETRIES} retries")
                _cusip_cache[cusip] = None
```

**Archivo:** `schwab.py` líneas 172-199
**Estimado:** 1 día

---

#### ⭐ 8. API Key Security (MEDIUM)

**Estado:** INSEGURO

**Ubicación Actual:** `schwab.py:119`
```python
CUSIP_API_KEY: ClassVar[str] = "nkqCXigD6ZeeX5QXcD3xQda3MOLK4zvo"  # ❌ Hardcoded
```

**Solución:**
```python
# schwab.py:119
import os

CUSIP_API_KEY: ClassVar[str] = os.getenv("CUSIP_API_KEY", "")

@classmethod
def resolve_cusips(cls, cusips: List[str]) -> Dict[str, str]:
    if not cls.CUSIP_API_KEY:
        logger.error("[SCHWAB] CUSIP_API_KEY not configured")
        return {}
    # ... rest
```

**Environment Variable:**
```bash
# .env
CUSIP_API_KEY=nkqCXigD6ZeeX5QXcD3xQda3MOLK4zvo
```

**Archivo:** `schwab.py` línea 119
**Estimado:** 0.25 días

---

### CATEGORÍA 4: CONDITIONAL - CSV Support

#### 🎯 9. CSV Support (5 variantes - CONDICIONAL)

**Estado:** NO IMPLEMENTADO

**Scope Legacy:**
- `brokers_charlesschwab.py` - 1,235 líneas con 5 parsers CSV:
  1. `getcsv()` - Standard format
  2. `getcsv2()` - Action Description format
  3. `getcsv3()` - Qty Filled At format
  4. `getcsv4()` - Quantity/Face Value format
  5. `getcsv5()` - Exec Time/Spread format

**Decisión Requerida:** Ejecutar SQL query

```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*)) OVER (), 2) as percentage,
    MAX(created_at) as last_used
FROM data_sources
WHERE broker_id = 'charles_schwab'
  AND created_at > NOW() - INTERVAL '12 months'
GROUP BY source_type
ORDER BY count DESC;
```

**Criterio:**
- Si CSV percentage > 5%: **IMPLEMENTAR** (8-12 días)
- Si CSV percentage < 5%: **OMITIR**

**Complejidad:** MUY ALTA (5 parsers, account validation, date variants)
**Estimado:** 8-12 días (solo si necesario)

---

#### 🎯 10. SchwabStreet Format Support (CONDICIONAL)

**Estado:** NO IMPLEMENTADO

**Legacy:** `brokers_schwabstreet.py` (306 líneas)

**Criterio:** Si > 1% del total: **IMPLEMENTAR** (2-3 días)

**Complejidad:** MEDIA
**Estimado:** 2-3 días (solo si necesario)

---

### CATEGORÍA 5: OUT OF SCOPE (Correctamente Excluido)

#### ✅ 11. Account Reference Code Validation
**Legacy:** `brokers_charlesschwab.py:62-103`
**Conclusión:** **OUT OF SCOPE** - Pertenece a **sync service** (portfolio management)

#### ✅ 12. Portfolio ID Lookup
**Legacy:** `schwab_export.py:973-984` - Database lookup for portfolio FK
**Conclusión:** **OUT OF SCOPE** - Pertenece a **p05_write** stage

#### ✅ 13. Duplicate Detection via Database Query
**Legacy:** `brokers_charlesschwab.py:1159-1243` - Queries original_file_row from DB
**Conclusión:** **OUT OF SCOPE** - Pertenece a **p05_write** stage

#### ✅ 14. Trade Regrouping / Spread Detection
**Legacy:** `schwab_export.py:991,1113` - Multi-leg spread detection
**Conclusión:** **OUT OF SCOPE** - Pertenece a **p03_group** stage

#### ✅ 15. Date Format Conversions
**Conclusión:** **ALREADY IMPLEMENTED** ✅ (`schwab.py:441-446`)

#### ✅ 16. Option Symbol Parsing (5 formats)
**Conclusión:** **ALREADY IMPLEMENTED** ✅ (`schwab.py:368-381`)

#### ✅ 17. Credential Validation / OAuth
**Legacy:** `schwab_export.py:62-86`
**Conclusión:** **OUT OF SCOPE** - Pertenece a **sync service**

---

### CATEGORÍA 6: ALREADY IMPLEMENTED ✅

#### 18. ✅ Column Detection
**Estado:** IMPLEMENTADO (`detector.py:42`)

#### 19. ✅ Order Type Filtering
**Estado:** IMPLEMENTADO (`schwab.py:250`)

#### 20. ✅ CUSIP Resolution
**Estado:** IMPLEMENTADO (`schwab.py:152-209`)

#### 21. ✅ Symbol Transformation
**Estado:** IMPLEMENTADO (`schwab.py:424`)

#### 22. ✅ Asset Type Mapping
**Estado:** IMPLEMENTADO (`schwab.py:91-101`)

#### 23. ✅ Timezone Conversion
**Estado:** IMPLEMENTADO (`schwab.py:441-446`)

#### 24. ✅ Fee Extraction (4 types)
**Estado:** IMPLEMENTADO (`schwab.py:284-290`)

#### 25. ✅ Fee Absolute Value
**Estado:** IMPLEMENTADO (`schwab.py:286`)

#### 26. ✅ Quantity Absolute Value
**Estado:** IMPLEMENTADO (`schwab.py:433`)

#### 27. ✅ Price Rounding
**Estado:** NO NECESARIO (API proporciona precisión correcta)

---

## Plan de Implementación por Fases

### FASE 1: Critical Hash Fix (1-2 días) ⚠️ URGENTE

**Objetivo:** Corregir 42% hash match rate a 95-100%

**Tareas:**
1. Implementar closingPrice zeroing (0.5 días)
2. Implementar transferItems sorting (0.5 días)
3. Actualizar legacy code (0.5 días)
4. Escribir SQL rebuild script (0.5 días)

**Archivos:**
- `schwab.py:305-310` (new)
- `schwab_export.py:952,968` (legacy)
- `scripts/rebuild_schwab_file_row.py` (new script)

**Métricas de Éxito:**
- Hash match rate >= 95% (vs actual 42%)
- Data integrity mantenida (100%)
- Legacy y mirror producen hashes idénticos
- SQL rebuild completa sin errores
- Sin cambios user-facing (solo hash values)

**Complejidad:** MEDIA
**Riesgo:** MEDIO (requiere legacy code change + data rebuild)
**Estimado:** 1-2 días

---

### FASE 2: Data Validation (1.5-2 días)

**Objetivo:** Prevenir datos inválidos de entrar al pipeline

**Tareas:**
1. Required fields validation (0.5 días)
2. Status filter validation (0.25 días)
3. Symbol validation (0.25 días)
4. Price/quantity zero validation (0.5 días)
5. Disabled instrument filter (0.25 días)

**Archivos:** `schwab.py:268-302`

**Métricas de Éxito:**
- Zero symbols vacíos en output
- Zero trades con price zero/negativo
- Zero trades con quantity zero
- Zero órdenes INVALID procesadas
- Warnings logged para todos rechazos
- Rejection rate < 0.1% del total

**Complejidad:** BAJA
**Riesgo:** BAJO
**Estimado:** 1.5-2 días

---

### FASE 3: Quality Improvements (1-1.5 días)

**Objetivo:** Robustez y seguridad

**Tareas:**
1. CUSIP retry logic + rate limiting (1 día)
2. API key security (env variable) (0.25 días)

**Archivos:** `schwab.py:119,172-199`

**Métricas de Éxito:**
- CUSIP resolution success rate >= 98%
- API key no en source code
- Transient failures manejados gracefully
- Rate limiting previene API throttling

**Complejidad:** MEDIA
**Riesgo:** BAJO
**Estimado:** 1-1.5 días

---

### FASE 4: CSV Support (8-12 días - SI NECESARIO) 🎯

**BLOQUEADOR:** Ejecutar SQL query primero

**Criterio:**
- Si CSV > 5%: **IMPLEMENTAR**
- Si CSV < 5%: **OMITIR**

**Complejidad:** MUY ALTA
**Riesgo:** ALTO
**Estimado:** 8-12 días (solo si necesario)

---

## Matriz de Priorización

| # | Validación | Criticidad | Estado | Complejidad | Fase | Días |
|---|------------|-----------|--------|-------------|------|------|
| 1 | closingPrice Hash Fix | ⚠️⚠️⚠️ CRÍTICO | MISSING | MEDIA | 1 | 1-2 |
| 2 | Required Fields | ⭐⭐⭐ ALTA | MISSING | BAJA | 2 | 0.5 |
| 3 | Status Filter | ⭐⭐⭐ ALTA | MISSING | BAJA | 2 | 0.25 |
| 4 | Symbol Validation | ⭐⭐ MEDIA-ALTA | PARCIAL | BAJA | 2 | 0.25 |
| 5 | Price/Qty Zero | ⭐⭐ MEDIA-ALTA | MISSING | BAJA | 2 | 0.5 |
| 6 | Disabled Instrument | ⭐⭐ MEDIA | MISSING | BAJA | 2 | 0.25 |
| 7 | CUSIP Retry Logic | ⭐ MEDIA | PARCIAL | MEDIA | 3 | 1 |
| 8 | API Key Security | ⭐ MEDIA | INSEGURO | BAJA | 3 | 0.25 |
| 9-10 | Fee/Qty Absolute | ✅ DONE | DONE | - | - | 0 |
| 11 | Price Rounding | - | N/A | - | - | 0 |
| 12 | CSV Support | 🎯 CONDICIONAL | NOT IMPL | MUY ALTA | 4 | 8-12 |
| 13 | SchwabStreet | 🎯 CONDICIONAL | NOT IMPL | MEDIA | 4 | 2-3 |
| 14-17 | Out of Scope | ✅ N/A | CORRECTO | - | - | 0 |
| 18-27 | Ya Implementadas | ✅ DONE | DONE | - | - | 0 |

**Total Fase 1 (Crítico):** 1-2 días
**Total Fase 2 (Data Validation):** 1.5-2 días
**Total Fase 3 (Quality):** 1-1.5 días
**Total Fase 1-3:** 3.5-5.5 días
**Total Fase 4 (Condicional):** 10-15 días adicionales si necesario

---

## Comparación con Otros Brokers

| Broker | Legacy Lines | New Lines | Reduction | Hash Match | Critical Validations | Phase 1-3 Days |
|--------|--------------|-----------|-----------|------------|---------------------|----------------|
| **Charles Schwab** | **3,196** | **1,087** | **66%** | **42%** ⚠️ | **6** | **3.5-5.5** |
| KuCoin | 2,015 | 404 | 80% | 100% | 4-5 | 2.5-3.5 |
| Kraken | 787 | 361 | 54% | 100% | 3-4 | 2 |
| Binance | ~1,500 | ~450 | 70% | ~98% | 3 | 2-3 |

**Observaciones Charles Schwab:**
- **Reducción moderada de código** (66%) - Legacy tenía 5 CSV parsers + API sync
- **ISSUE CRÍTICO único** (42% match) - closingPrice volatility única de Schwab
- **Más validaciones** (6 críticas) - Modelo de datos más complejo (options, CUSIPs)
- **Esfuerzo similar** (3.5-5.5 días Fase 1-3) - A pesar de complejidad
- **Excelente separación de scope** - Auth, regrouping, calculations correctamente excluidos

---

## Archivos Críticos

### 1. `brokers/charles_schwab/schwab.py` (PRINCIPAL)
**Líneas Actuales:** 521
**Estimadas Post-Cambios:** ~600-650 (Fases 1-3), ~900-1,100 (si Fase 4)

**Cambios:**
- Líneas 305-310: Hash computation (Fase 1 - CRÍTICO)
- Líneas 268-295: parse_json_content() validations (Fase 2)
- Líneas 172-199: CUSIP retry logic (Fase 3)
- Línea 119: API key security (Fase 3)

### 2. `OLD CODE FROM LEGACY/schwab_export.py` (LEGACY UPDATE)
**Líneas a Modificar:** 952, 968
**Cambio:** Match new hash formula

### 3. `tests/brokers/test_charles_schwab.py` (TESTS)
**Líneas Actuales:** 504
**Estimadas Post-Cambios:** 700-800

### 4. `scripts/rebuild_schwab_file_row.py` (NEW)
**Purpose:** Rebuild file_row hashes for all existing Schwab data
**Estimado:** 150-200 líneas

### 5. Legacy Files (REFERENCIA SOLO)
- `OLD CODE FROM LEGACY/brokers_charlesschwab.py` (1,235 líneas)
- `OLD CODE FROM LEGACY/brokers_schwabstreet.py` (306 líneas)

---

## Decisiones Pendientes

### Decisión 1: closingPrice Fix Strategy

**Pregunta:** ¿Qué approach para fixing hash volatility?

**Opciones:**
- A: Zero out closingPrice + sort transferItems (RECOMENDADO)
- B: Zero out closingPrice only
- C: Aceptar 42% match rate (NO RECOMENDADO)

**Recomendación:** **Opción A**
- Fixes both issues
- Expected: 95-100% match rate
- Requiere legacy update + SQL rebuild

### Decisión 2: CSV Support Necessity

**Pregunta:** ¿Implementar 5 CSV parsers (1,235 líneas)?

**Data Requerida:** Ejecutar SQL query (ver validación #12)

**Criterio:**
- Si CSV > 5%: **IMPLEMENTAR** (8-12 días)
- Si CSV < 5%: **OMITIR**

**Recomendación:** **Query data first, then decide**

### Decisión 3: SchwabStreet Support

**Pregunta:** ¿Soportar formato SchwabStreet CSV?

**Criterio:** Si > 1% del total: **IMPLEMENTAR** (2-3 días)

**Recomendación:** **Probablemente OMITIR** (formato legacy, likely unused)

---

## Conclusión

Análisis identifica **27 categorías de validaciones**:
- **1 CRÍTICA (Fase 1):** 1-2 días - closingPrice hash fix
- **5 HIGH (Fase 2):** 1.5-2 días - Data validations
- **2 MEDIA (Fase 3):** 1-1.5 días - Quality improvements
- **2 CONDICIONALES (Fase 4):** 10-15 días si necesario
- **7 OUT OF SCOPE:** Sin acción (correctamente excluidas)
- **10 YA IMPLEMENTADAS:** Sin acción

**Hallazgo Clave:** Nueva implementación es **arquitecturalmente superior**:
- 66% menos código (3,196→1,087 líneas)
- Correcta separación de concerns
- Solo 1 issue crítico (closingPrice - único de Schwab)

**Issue Principal:** closingPrice es **precio de mercado EN VIVO**, no precio de ejecución - DEBE corregirse para deduplicación.

**RECOMENDACIÓN FINAL:**
1. **Implementar Fase 1 inmediatamente** (1-2 días) - CRÍTICO
2. **Implementar Fase 2** (1.5-2 días) - Data integrity
3. **Implementar Fase 3** (1-1.5 días) - Robustez
4. **Ejecutar SQL queries antes de Fase 4** - Decisiones data-driven
5. **NO sobre-ingenierizar** - Preservar simplicidad arquitectural

Total estimado crítico: **3.5-5.5 días**
Total con condicionales: **13.5-20.5 días** (solo si datos muestran necesidad)
