# Ejemplos de Cambios de Código - Charles Schwab Normalizer

Este documento contiene ejemplos detallados de código **ANTES/DESPUÉS** para cada validación identificada, con tests correspondientes.

**Fecha:** 2026-01-14
**Broker:** charles_schwab
**Archivo Principal:** `brokers/charles_schwab/schwab.py`

---

## FASE 1: Critical Hash Fix ⚠️

### 1. closingPrice Hash Volatility (CRÍTICO)

**Problema:** El campo `closingPrice` en Schwab API es el precio de mercado EN VIVO, no el precio de ejecución histórico. Esto causa que 58% de las ejecuciones tengan hashes diferentes entre legacy y mirror.

**Evidencia Real:**
```python
# Mismo trade, diferentes sync dates
trade = {
    "activityId": 31607727843,
    "symbol": "SAVA",
    "executionDate": "2021-09-15",
    "transferItems": [{
        "instrument": {
            "closingPrice": 2.09,  # Mirror: 2026-01-11
            # vs
            "closingPrice": 2.18   # Legacy: 2026-01-08
        },
        "price": 43.93  # ← MISMO precio de ejecución
    }]
}

# Resultado: Diferentes hashes, pero MISMOS datos de trading
```

**Archivo:** `schwab.py` líneas 305-310

#### ANTES (Actual - PROBLEMA)

```python
def compute_file_row_hash(order):
    """
    Compute MD5 hash for deduplication.

    PROBLEMA: Incluye closingPrice que es volatile
    """
    # Remove post-hash fields
    order_for_hash = {k: v for k, v in order.items() if k != 'account_hash'}

    # Hash directly (includes volatile closingPrice)
    file_row_hash = hashlib.md5(
        json.dumps(order_for_hash).encode('utf-8')
    ).hexdigest()

    return file_row_hash

# Resultado: 42% hash match rate (58% false negatives)
```

#### DESPUÉS (Solución - FIX)

```python
import copy

def compute_file_row_hash(order):
    """
    Compute MD5 hash for deduplication.

    FIX: Zero out closingPrice + sort transferItems para hash determinístico
    """
    # Remove post-hash fields
    order_for_hash = {k: v for k, v in order.items() if k != 'account_hash'}

    # Deep copy para no modificar original
    order_copy = copy.deepcopy(order_for_hash)

    # FIX 1: Zero out closingPrice (volatile field)
    for item in order_copy.get('transferItems', []):
        if 'instrument' in item and 'closingPrice' in item['instrument']:
            item['instrument']['closingPrice'] = 0.0

    # FIX 2: Sort transferItems (orden no determinístico en API)
    def sort_key(item):
        """Sort by assetType, feeType, instrumentId para orden determinístico"""
        inst = item.get('instrument', {})
        return (
            inst.get('assetType', 'Z'),      # CURRENCY < EQUITY < OPTION
            item.get('feeType', 'Z'),         # COMMISSION < SEC_FEE < TAF_FEE
            inst.get('instrumentId', 0)       # ID único
        )

    if 'transferItems' in order_copy:
        order_copy['transferItems'] = sorted(
            order_copy['transferItems'],
            key=sort_key
        )

    # Compute hash
    file_row_hash = hashlib.md5(
        json.dumps(order_copy).encode('utf-8')
    ).hexdigest()

    return file_row_hash

# Expected resultado: 95-100% hash match rate
```

#### Tests

```python
# tests/brokers/test_charles_schwab.py

def test_file_row_stable_across_closing_price_changes():
    """
    Verifica que cambios en closingPrice no afectan hash.

    Simula el escenario real: mismo trade synced en diferentes fechas
    tiene diferentes closingPrice pero debe tener mismo hash.
    """
    # Mismo trade, diferentes closingPrice
    order1 = {
        "activityId": 31607727843,
        "accountNumber": "12345678",
        "time": "2021-09-15T14:30:00+0000",
        "type": "TRADE",
        "transferItems": [{
            "instrument": {
                "symbol": "SAVA",
                "assetType": "EQUITY",
                "instrumentId": 123,
                "closingPrice": 2.09  # Mirror sync (2026-01-11)
            },
            "price": 43.93,
            "amount": 100
        }]
    }

    order2 = {
        "activityId": 31607727843,
        "accountNumber": "12345678",
        "time": "2021-09-15T14:30:00+0000",
        "type": "TRADE",
        "transferItems": [{
            "instrument": {
                "symbol": "SAVA",
                "assetType": "EQUITY",
                "instrumentId": 123,
                "closingPrice": 2.18  # Legacy sync (2026-01-08) - DIFERENTE
            },
            "price": 43.93,
            "amount": 100
        }]
    }

    hash1 = compute_file_row_hash(order1)
    hash2 = compute_file_row_hash(order2)

    # Debe tener mismo hash a pesar de closingPrice diferente
    assert hash1 == hash2, f"Hash should be identical: {hash1} vs {hash2}"


def test_file_row_stable_across_transfer_items_order():
    """
    Verifica que orden de transferItems no afecta hash.

    Schwab API puede devolver fees y instrument items en cualquier orden.
    """
    equity = {
        "instrument": {
            "symbol": "AAPL",
            "assetType": "EQUITY",
            "instrumentId": 100,
            "closingPrice": 150.0
        },
        "price": 150.0,
        "amount": 100
    }

    fee1 = {
        "instrument": {"assetType": "CURRENCY"},
        "feeType": "COMMISSION",
        "amount": -5.0
    }

    fee2 = {
        "instrument": {"assetType": "CURRENCY"},
        "feeType": "SEC_FEE",
        "amount": -0.10
    }

    # Orden 1: [equity, fee1, fee2]
    order1 = {
        "activityId": 123,
        "transferItems": [equity, fee1, fee2]
    }

    # Orden 2: [fee2, equity, fee1] - DIFERENTE
    order2 = {
        "activityId": 123,
        "transferItems": [fee2, equity, fee1]
    }

    hash1 = compute_file_row_hash(order1)
    hash2 = compute_file_row_hash(order2)

    # Debe tener mismo hash después de sorting
    assert hash1 == hash2, f"Hash should be identical after sorting: {hash1} vs {hash2}"


def test_file_row_preserves_original():
    """
    Verifica que hash computation no modifica el order original.
    """
    order = {
        "activityId": 123,
        "transferItems": [{
            "instrument": {"closingPrice": 150.0}
        }]
    }

    original_closing_price = order["transferItems"][0]["instrument"]["closingPrice"]

    # Compute hash
    hash_value = compute_file_row_hash(order)

    # Verify original no modificado
    assert order["transferItems"][0]["instrument"]["closingPrice"] == original_closing_price
    assert order["transferItems"][0]["instrument"]["closingPrice"] == 150.0


def test_hash_integration_with_legacy():
    """
    Integration test: Verifica que nuevo hash match con legacy hash.

    Usa datos reales de producción.
    """
    # Datos reales de trade SAVA
    real_order = {
        "activityId": 31607727843,
        "accountNumber": "12345678",
        "time": "2021-09-15T14:30:00+0000",
        "type": "TRADE",
        "transferItems": [{
            "instrument": {
                "symbol": "SAVA",
                "assetType": "EQUITY",
                "instrumentId": 99999,
                "closingPrice": 2.09  # Este valor cambia
            },
            "price": 43.93,
            "amount": 100
        }]
    }

    # Compute new hash
    new_hash = compute_file_row_hash(real_order)

    # Expected legacy hash (computed con closingPrice = 0.0)
    # Este hash fue calculado en legacy con el mismo fix
    expected_legacy_hash = "abc123..."  # Replace con hash real de legacy

    assert new_hash == expected_legacy_hash
```

**Impacto:**
- ✅ Hash match rate: 42% → 95-100%
- ✅ False negatives: 36,296 → ~3,000
- ✅ Data integrity: Mantenida (100%)
- ✅ No cambios user-facing

---

## FASE 2: Data Validation

### 2. Required Fields Validation

**Problema:** Órdenes sin campos requeridos pueden causar hash collisions (activityId vacío), errores de processing (transferItems vacío), o no se pueden asignar a usuario (accountNumber vacío).

**Archivo:** `schwab.py` línea 268

#### ANTES

```python
def parse_json_content(cls, json_content: str, account_id: str = "") -> pl.DataFrame:
    data = json.loads(json_content)
    orders = data.get("orders", [])

    records = []
    for row_idx, order in enumerate(orders):
        # ❌ Sin validación de campos requeridos

        # Process order directly
        activity_id = order.get("activityId")  # Puede ser None o ""
        # ... rest of processing
```

#### DESPUÉS

```python
def parse_json_content(cls, json_content: str, account_id: str = "") -> pl.DataFrame:
    data = json.loads(json_content)
    orders = data.get("orders", [])

    records = []
    for row_idx, order in enumerate(orders):
        # ✅ VALIDACIÓN: Campos requeridos
        activity_id = order.get("activityId")
        account_number = order.get("accountNumber")
        time = order.get("time", "")
        transfer_items = order.get("transferItems", [])

        # Check activityId (required para hash uniqueness)
        if not activity_id:
            logger.warning(f"[SCHWAB] Skipping order at index {row_idx}: missing activityId")
            continue

        # Check accountNumber (required para user assignment)
        if not account_number:
            logger.warning(f"[SCHWAB] Skipping order {activity_id}: missing accountNumber")
            continue

        # Check time (required para timestamp)
        if not time:
            logger.warning(f"[SCHWAB] Skipping order {activity_id}: missing time")
            continue

        # Check transferItems (required para data extraction)
        if not transfer_items:
            logger.warning(f"[SCHWAB] Skipping order {activity_id}: empty transferItems")
            continue

        # Process order
        # ... rest of processing
```

#### Tests

```python
def test_required_fields_validation_activity_id():
    """Verifica que órdenes sin activityId son rechazadas"""
    json_data = {
        "orders": [{
            # "activityId": MISSING
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": [{"instrument": {"symbol": "AAPL"}}]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))

    # Debe ser vacío (orden rechazada)
    assert len(df) == 0


def test_required_fields_validation_account_number():
    """Verifica que órdenes sin accountNumber son rechazadas"""
    json_data = {
        "orders": [{
            "activityId": 123,
            # "accountNumber": MISSING
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": [{"instrument": {"symbol": "AAPL"}}]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 0


def test_required_fields_validation_time():
    """Verifica que órdenes sin time son rechazadas"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "",  # Vacío
            "type": "TRADE",
            "transferItems": [{"instrument": {"symbol": "AAPL"}}]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 0


def test_required_fields_validation_transfer_items():
    """Verifica que órdenes sin transferItems son rechazadas"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": []  # Vacío
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 0


def test_required_fields_validation_valid_order():
    """Verifica que órdenes válidas pasan la validación"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": [{
                "instrument": {
                    "symbol": "AAPL",
                    "assetType": "EQUITY"
                },
                "price": 150.0,
                "amount": 100
            }]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))

    # Debe tener 1 row
    assert len(df) == 1
```

---

### 3. Status Filter Validation

**Problema:** Órdenes con status "INVALID" pueden contaminar datos.

**Archivo:** `schwab.py` línea 270

#### ANTES

```python
for row_idx, order in enumerate(orders):
    # Required fields validation
    # ...

    # ❌ Sin filtro de status

    # Process order
```

#### DESPUÉS

```python
for row_idx, order in enumerate(orders):
    # Required fields validation
    # ...

    # ✅ VALIDACIÓN: Status filter
    status = order.get("status", "").upper()
    if status == "INVALID":
        logger.debug(f"[SCHWAB] Skipping order {activity_id}: status is INVALID")
        continue

    # Process order
```

#### Tests

```python
def test_status_filter_invalid():
    """Verifica que órdenes INVALID son rechazadas"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "status": "INVALID",  # ← INVALID
            "transferItems": [{"instrument": {"symbol": "AAPL"}}]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 0


def test_status_filter_valid():
    """Verifica que órdenes con status válido pasan"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "status": "EXECUTED",  # ← Válido
            "transferItems": [{"instrument": {"symbol": "AAPL", "assetType": "EQUITY"}, "price": 150, "amount": 100}]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 1


def test_status_filter_missing_status():
    """Verifica que órdenes sin campo status pasan (default OK)"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            # No "status" field
            "transferItems": [{"instrument": {"symbol": "AAPL", "assetType": "EQUITY"}, "price": 150, "amount": 100}]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 1
```

---

### 4. Symbol Validation

**Problema:** Symbols vacíos después de CUSIP resolution pueden causar errores.

**Archivo:** `schwab.py` línea 351

#### ANTES

```python
# Resolve CUSIPs
raw_symbol = inst.get("symbol", "")
resolved_symbol = cusip_to_ticker.get(raw_symbol, raw_symbol)

# ❌ Sin validación si resolved_symbol está vacío

# Use resolved symbol
symbol = resolved_symbol
```

#### DESPUÉS

```python
# Resolve CUSIPs
raw_symbol = inst.get("symbol", "")
resolved_symbol = cusip_to_ticker.get(raw_symbol, raw_symbol)

# ✅ VALIDACIÓN: Symbol vacío
if not resolved_symbol or not resolved_symbol.strip():
    logger.warning(f"[SCHWAB] Skipping order {activity_id}: empty symbol after CUSIP resolution (raw: {raw_symbol})")
    continue

# Use resolved symbol
symbol = resolved_symbol
```

#### Tests

```python
def test_empty_symbol_validation():
    """Verifica que órdenes con symbol vacío son rechazadas"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": [{
                "instrument": {
                    "symbol": "",  # Vacío
                    "assetType": "EQUITY"
                },
                "price": 150.0,
                "amount": 100
            }]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 0


def test_whitespace_symbol_validation():
    """Verifica que symbols con solo whitespace son rechazados"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": [{
                "instrument": {
                    "symbol": "   ",  # Solo espacios
                    "assetType": "EQUITY"
                },
                "price": 150.0,
                "amount": 100
            }]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 0
```

---

### 5. Price/Quantity Zero Validation

**Problema:** Price zero o quantity zero pueden causar cálculos inválidos (multiplier divide by zero, etc).

**Archivo:** `schwab.py` línea 302

#### ANTES

```python
# Extract values
total_amount = sum(float(item.get("amount", 0) or 0) for item in instrument_items)
price = float(first_item.get("price", 0) or 0)

# ❌ Sin validación de zero

# Use values
quantity = abs(total_amount)
```

#### DESPUÉS

```python
# Extract values
total_amount = sum(float(item.get("amount", 0) or 0) for item in instrument_items)
price = float(first_item.get("price", 0) or 0)

# ✅ VALIDACIÓN: Price zero o negativo
if price <= 0:
    logger.warning(f"[SCHWAB] Skipping order {activity_id}: zero or negative price {price}")
    continue

# ✅ VALIDACIÓN: Quantity zero
if abs(total_amount) <= 0:
    logger.warning(f"[SCHWAB] Skipping order {activity_id}: zero quantity")
    continue

# Use values
quantity = abs(total_amount)
```

#### Tests

```python
def test_zero_price_validation():
    """Verifica que órdenes con price zero son rechazadas"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": [{
                "instrument": {"symbol": "AAPL", "assetType": "EQUITY"},
                "price": 0.0,  # Zero
                "amount": 100
            }]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 0


def test_negative_price_validation():
    """Verifica que órdenes con price negativo son rechazadas"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": [{
                "instrument": {"symbol": "AAPL", "assetType": "EQUITY"},
                "price": -150.0,  # Negativo
                "amount": 100
            }]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 0


def test_zero_quantity_validation():
    """Verifica que órdenes con quantity zero son rechazadas"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": [{
                "instrument": {"symbol": "AAPL", "assetType": "EQUITY"},
                "price": 150.0,
                "amount": 0  # Zero
            }]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 0
```

---

### 6. Disabled Instrument Filter

**Problema:** Instrumentos disabled (delisted stocks, corporate actions) pueden contaminar datos.

**Archivo:** `schwab.py` líneas 277-282

#### ANTES

```python
# Extract instrument items and fee items
instrument_items = []
fee_items = []

for item in transfer_items:
    inst = item.get("instrument", {})

    # ❌ Sin filtro de disabled instruments

    if inst.get("assetType") == "CURRENCY":
        fee_items.append(item)
    else:
        instrument_items.append(item)
```

#### DESPUÉS

```python
# Extract instrument items and fee items
instrument_items = []
fee_items = []

for item in transfer_items:
    inst = item.get("instrument", {})

    # ✅ VALIDACIÓN: Skip disabled instruments
    if inst.get("status", "").lower() == "disabled":
        logger.debug(f"[SCHWAB] Skipping disabled instrument in order {activity_id}")
        continue

    if inst.get("assetType") == "CURRENCY":
        fee_items.append(item)
    else:
        instrument_items.append(item)
```

#### Tests

```python
def test_disabled_instrument_filter():
    """Verifica que instrumentos disabled son filtrados"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": [{
                "instrument": {
                    "symbol": "DELISTED",
                    "assetType": "EQUITY",
                    "status": "disabled"  # ← Disabled
                },
                "price": 150.0,
                "amount": 100
            }]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))

    # Debe ser vacío (no instrument items left después de filter)
    assert len(df) == 0


def test_active_instrument_passes():
    """Verifica que instrumentos activos pasan el filter"""
    json_data = {
        "orders": [{
            "activityId": 123,
            "accountNumber": "12345678",
            "time": "2024-01-15T10:00:00+0000",
            "type": "TRADE",
            "transferItems": [{
                "instrument": {
                    "symbol": "AAPL",
                    "assetType": "EQUITY",
                    "status": "active"  # ← Active
                },
                "price": 150.0,
                "amount": 100
            }]
        }]
    }

    df = SchwabInterpreter.parse_json_content(json.dumps(json_data))
    assert len(df) == 1
```

---

## FASE 3: Quality Improvements

### 7. CUSIP Retry Logic

**Problema:** CUSIP resolution falla en transient errors (timeouts, rate limits) sin retry.

**Archivo:** `schwab.py` líneas 172-199

#### ANTES

```python
@classmethod
def resolve_cusips(cls, cusips: List[str]) -> Dict[str, str]:
    """Resolve CUSIPs to ticker symbols"""
    cusip_to_ticker = {}

    for cusip in cusips:
        try:
            url = f"{cls.CUSIP_API_URL}{cusip}?apikey={cls.CUSIP_API_KEY}"
            response = requests.get(url, timeout=10)

            # ❌ No retry logic
            # ❌ No rate limit handling

            if response.status_code == 200:
                data = response.json()
                ticker = data[0].get("symbol")
                cusip_to_ticker[cusip] = ticker

        except Exception as e:
            logger.warning(f"[SCHWAB] Failed to resolve CUSIP {cusip}: {e}")
            cusip_to_ticker[cusip] = None

    return cusip_to_ticker
```

#### DESPUÉS

```python
import time

@classmethod
def resolve_cusips(cls, cusips: List[str]) -> Dict[str, str]:
    """
    Resolve CUSIPs to ticker symbols with retry logic.

    Implementa:
    - Exponential backoff retry (max 3 attempts)
    - Rate limit handling (HTTP 429)
    - Timeout handling
    """
    MAX_RETRIES = 3
    RETRY_DELAY = 2  # seconds

    cusip_to_ticker = {}

    for cusip in cusips:
        retries = 0

        while retries < MAX_RETRIES:
            try:
                url = f"{cls.CUSIP_API_URL}{cusip}?apikey={cls.CUSIP_API_KEY}"
                response = requests.get(url, timeout=10)

                # ✅ Handle rate limiting
                if response.status_code == 429:
                    wait_time = RETRY_DELAY * (2 ** retries)  # Exponential backoff
                    logger.warning(
                        f"[SCHWAB] Rate limited on CUSIP {cusip}, "
                        f"waiting {wait_time}s (retry {retries + 1}/{MAX_RETRIES})"
                    )
                    time.sleep(wait_time)
                    retries += 1
                    continue

                # Success
                if response.status_code == 200:
                    data = response.json()
                    if data and len(data) > 0:
                        ticker = data[0].get("symbol")
                        cusip_to_ticker[cusip] = ticker
                        logger.debug(f"[SCHWAB] Resolved CUSIP {cusip} → {ticker}")
                        break
                else:
                    logger.warning(
                        f"[SCHWAB] CUSIP API returned {response.status_code} for {cusip}"
                    )
                    cusip_to_ticker[cusip] = None
                    break

            except requests.exceptions.Timeout:
                # ✅ Retry on timeout
                retries += 1
                if retries < MAX_RETRIES:
                    logger.warning(
                        f"[SCHWAB] Timeout resolving CUSIP {cusip}, "
                        f"retry {retries}/{MAX_RETRIES}"
                    )
                    time.sleep(RETRY_DELAY)
                else:
                    logger.error(
                        f"[SCHWAB] Failed to resolve CUSIP {cusip} "
                        f"after {MAX_RETRIES} retries"
                    )
                    cusip_to_ticker[cusip] = None

            except Exception as e:
                logger.error(f"[SCHWAB] Error resolving CUSIP {cusip}: {e}")
                cusip_to_ticker[cusip] = None
                break

    return cusip_to_ticker
```

#### Tests

```python
import pytest
from unittest.mock import patch, Mock

def test_cusip_retry_on_timeout():
    """Verifica retry en timeout"""
    with patch('requests.get') as mock_get:
        # Primera llamada: timeout
        # Segunda llamada: success
        mock_get.side_effect = [
            requests.exceptions.Timeout(),
            Mock(status_code=200, json=lambda: [{"symbol": "AAPL"}])
        ]

        result = SchwabInterpreter.resolve_cusips(["037833100"])

        # Debe haber 2 llamadas (1 timeout + 1 retry)
        assert mock_get.call_count == 2
        assert result["037833100"] == "AAPL"


def test_cusip_rate_limit_handling():
    """Verifica handling de HTTP 429 con exponential backoff"""
    with patch('requests.get') as mock_get, \
         patch('time.sleep') as mock_sleep:

        # Primera llamada: 429
        # Segunda llamada: 429
        # Tercera llamada: success
        mock_get.side_effect = [
            Mock(status_code=429),
            Mock(status_code=429),
            Mock(status_code=200, json=lambda: [{"symbol": "TSLA"}])
        ]

        result = SchwabInterpreter.resolve_cusips(["88160R101"])

        # Debe haber 3 llamadas
        assert mock_get.call_count == 3

        # Debe haber 2 sleeps con exponential backoff
        assert mock_sleep.call_count == 2
        mock_sleep.assert_any_call(2)  # 2 * (2^0) = 2
        mock_sleep.assert_any_call(4)  # 2 * (2^1) = 4

        assert result["88160R101"] == "TSLA"


def test_cusip_max_retries_exceeded():
    """Verifica que después de MAX_RETRIES se abandona"""
    with patch('requests.get') as mock_get:
        # Todas las llamadas: timeout
        mock_get.side_effect = requests.exceptions.Timeout()

        result = SchwabInterpreter.resolve_cusips(["037833100"])

        # Debe haber MAX_RETRIES llamadas
        assert mock_get.call_count == 3
        assert result["037833100"] is None
```

---

### 8. API Key Security

**Problema:** API key hardcoded en source code (security risk).

**Archivo:** `schwab.py` línea 119

#### ANTES

```python
class SchwabInterpreter(BaseInterpreter):
    BROKER_ID: ClassVar[str] = "charles_schwab"

    # ❌ INSEGURO: API key hardcoded
    CUSIP_API_KEY: ClassVar[str] = "nkqCXigD6ZeeX5QXcD3xQda3MOLK4zvo"
    CUSIP_API_URL: ClassVar[str] = "https://financialmodelingprep.com/api/v3/cusip/"
```

#### DESPUÉS

```python
import os

class SchwabInterpreter(BaseInterpreter):
    BROKER_ID: ClassVar[str] = "charles_schwab"

    # ✅ SEGURO: API key desde environment variable
    CUSIP_API_KEY: ClassVar[str] = os.getenv("CUSIP_API_KEY", "")
    CUSIP_API_URL: ClassVar[str] = "https://financialmodelingprep.com/api/v3/cusip/"

    @classmethod
    def resolve_cusips(cls, cusips: List[str]) -> Dict[str, str]:
        """Resolve CUSIPs with API key validation"""
        # ✅ Validar API key configurado
        if not cls.CUSIP_API_KEY:
            logger.error(
                "[SCHWAB] CUSIP_API_KEY not configured. "
                "Set environment variable CUSIP_API_KEY."
            )
            return {cusip: None for cusip in cusips}

        # ... rest of resolution logic
```

**Environment Variable:**
```bash
# .env
CUSIP_API_KEY=nkqCXigD6ZeeX5QXcD3xQda3MOLK4zvo
```

#### Tests

```python
def test_cusip_api_key_from_env():
    """Verifica que API key se lee desde environment"""
    with patch.dict(os.environ, {"CUSIP_API_KEY": "test_key_123"}):
        # Reload class para leer nueva env var
        from importlib import reload
        import pipeline.p01_normalize.brokers.charles_schwab.schwab as schwab_module
        reload(schwab_module)

        assert schwab_module.SchwabInterpreter.CUSIP_API_KEY == "test_key_123"


def test_cusip_api_key_missing():
    """Verifica que error cuando API key no configurado"""
    with patch.dict(os.environ, {}, clear=True):  # Sin CUSIP_API_KEY
        result = SchwabInterpreter.resolve_cusips(["037833100"])

        # Debe retornar None para todos los CUSIPs
        assert result["037833100"] is None


def test_cusip_resolution_with_valid_key():
    """Verifica resolución con API key válido"""
    with patch.dict(os.environ, {"CUSIP_API_KEY": "valid_key"}), \
         patch('requests.get') as mock_get:

        mock_get.return_value = Mock(
            status_code=200,
            json=lambda: [{"symbol": "AAPL"}]
        )

        result = SchwabInterpreter.resolve_cusips(["037833100"])

        # Debe haber llamado API con key correcto
        mock_get.assert_called_once()
        call_url = mock_get.call_args[0][0]
        assert "apikey=valid_key" in call_url

        assert result["037833100"] == "AAPL"
```

---

## Resumen de Coverage

### Tests Nuevos Requeridos

**Fase 1 (Hash Fix):** 4 tests
- test_file_row_stable_across_closing_price_changes()
- test_file_row_stable_across_transfer_items_order()
- test_file_row_preserves_original()
- test_hash_integration_with_legacy()

**Fase 2 (Validations):** 13 tests
- test_required_fields_validation_activity_id()
- test_required_fields_validation_account_number()
- test_required_fields_validation_time()
- test_required_fields_validation_transfer_items()
- test_required_fields_validation_valid_order()
- test_status_filter_invalid()
- test_status_filter_valid()
- test_status_filter_missing_status()
- test_empty_symbol_validation()
- test_whitespace_symbol_validation()
- test_zero_price_validation()
- test_negative_price_validation()
- test_zero_quantity_validation()
- test_disabled_instrument_filter()
- test_active_instrument_passes()

**Fase 3 (Quality):** 6 tests
- test_cusip_retry_on_timeout()
- test_cusip_rate_limit_handling()
- test_cusip_exponential_backoff()
- test_cusip_max_retries_exceeded()
- test_cusip_api_key_from_env()
- test_cusip_api_key_missing()
- test_cusip_resolution_with_valid_key()

**Total:** 23 tests nuevos

### Coverage Estimado

**Antes:** ~75% (504 líneas tests / ~700 líneas código)
**Después:** ~92% (750+ líneas tests / ~800 líneas código)

---

## Notas de Implementación

1. **Hash Formula:** Preservar orden de keys en JSON dump (no `sort_keys=True`)
2. **Deep Copy:** Usar `copy.deepcopy()` para no modificar original
3. **Logger Levels:**
   - `logger.warning()`: Para rechazos de datos (validation failures)
   - `logger.debug()`: Para skips esperados (disabled instruments)
   - `logger.error()`: Para errores de configuración (missing API key)
4. **Retry Strategy:** Exponential backoff con max 3 retries
5. **API Key:** Leer desde `CUSIP_API_KEY` env variable
6. **Tests:** Usar `unittest.mock.patch` para external API calls

---

**Fecha:** 2026-01-14
**Total Líneas de Código Nuevas:** ~250 líneas
**Total Líneas de Tests Nuevas:** ~450 líneas
