"""
Format detector for Deribit data.

Examines DataFrame columns to identify Deribit format.
"""

import polars as pl
from typing import List, Type

from pipeline.p01_normalize.base import BaseInterpreter
from pipeline.p01_normalize.exceptions import UnknownFormatError
from .deribit import DeribitInterpreter


# Register all interpreters for Deribit
INTERPRETERS: List[Type[BaseInterpreter]] = [
    DeribitInterpreter,
]


def detect(df: pl.DataFrame, metadata: dict = None) -> BaseInterpreter:
    """
    Detect the correct interpreter for the given data.

    Args:
        df: DataFrame to examine
        metadata: Optional metadata about the data source

    Returns:
        Interpreter instance that can handle the data

    Raises:
        UnknownFormatError: If no interpreter can handle the data
    """
    metadata = metadata or {}

    # Sort by priority
    sorted_interpreters = sorted(
        INTERPRETERS,
        key=lambda x: x.get_priority(),
        reverse=True
    )

    for interpreter_cls in sorted_interpreters:
        if interpreter_cls.can_handle(df, metadata):
            return interpreter_cls()

    raise UnknownFormatError(
        f"No interpreter found for Deribit data. Columns: {df.columns}"
    )
