# Plan: Análisis de Validaciones Críticas Faltantes - Deribit Normalizer

## Resumen Ejecutivo

Tras analizar exhaustivamente las implementaciones legacy (875 líneas en 2 archivos) vs nueva (1,188 líneas), se identificaron **22 categorías de validaciones**, de las cuales **1 es CRÍTICA** (expired options hash mismatch con 74.27% match rate) y **5-6 son HIGH/MEDIUM priority** para integridad de datos.

**Hallazgo clave**: La nueva implementación es **arquitecturalmente superior** con **strong test coverage** (616 líneas de tests). Sin embargo, existe un **issue crítico con expired options**:

### Issue Crítico: Expired Options Hash Mismatch
- **Hash match rate actual**: **74.27%** (1,146/1,543 matches) ⚠️
- **Causa**: 397 expired options records (25.73%) usan diferente hash formula con "EXP_A" suffix
- **Impacto**: Expired options pueden duplicarse en re-imports
- **Integridad de datos**: 100% correcta para datos de trading

### Estado Actual: BUENO (con issue de hash)
- **Hash match rate**: 74.27% (necesita investigación de expired options)
- **Data integrity**: 100% correcta
- **Test coverage**: 616 líneas (comprehensive tests)
- **Scope**: JSON API only (crypto perpetuals + options)

**Diferencias con otros brokers:**
- ⚠️ **Issue de hash con expired options** (74.27% vs objetivo 95%+)
- ✅ **Strong test coverage** - 616 líneas de tests comprehensivos
- ✅ **Soporta options** - European-style crypto options (BTC, ETH)
- ⚠️ **No CSV support** - Legacy tiene 2 CSV parsers (326 líneas)

---

## Comparación de Implementaciones

### Nueva Implementación (1,188 líneas totales)

| Archivo | Líneas | Propósito |
|---------|--------|-----------|
| deribit.py | 516 | Main interpreter (JSON API) |
| detector.py | 50 | Format detection |
| __init__.py | 6 | Module exports |
| test_deribit.py | 616 | Comprehensive tests |
| **TOTAL** | **1,188** | **Production-ready con tests** |

### Legacy Implementación (875 líneas totales)

| Archivo | Líneas | Scope |
|---------|--------|-------|
| brokers_deribit.py | 326 | CSV normalization - 2 formats (IN SCOPE) |
| deribit_export.py | 549 | API sync service (OUT OF SCOPE) |
| **TOTAL** | **875** | **Mixed concerns** |

**Reducción efectiva:** 326 líneas (legacy CSV) → 516 líneas (new JSON API) = +58% expansion
**Pero:** Nueva implementación incluye mejor estructura + comprehensive tests

---

## Validaciones Identificadas

### CATEGORÍA 1: CRÍTICO - Hash Deduplication Issues

#### ⚠️⚠️⚠️ 1. Expired Options Hash Mismatch (CRÍTICO)

**Criticidad:** ALTA (afecta 25.73% de opciones expiradas)
**Estado:** PRESENTE EN NUEVA IMPLEMENTACIÓN

**Ubicación Actual:**
- Nueva: `deribit.py:238` - Usa `MD5(json.dumps(id))`
- Legacy: No encontrado "EXP_A" suffix en código legacy analizado

**Problema:**
La nueva implementación tiene 74.27% hash match rate. Los 397 records no coincidentes (25.73%) son expired options que probablemente usan diferente formula con "EXP_A" suffix.

**Evidencia:**
```python
# Current implementation (deribit.py:238)
file_row_hash = hashlib.md5(json.dumps(order_id).encode('utf-8')).hexdigest()

# Expected for expired options (from test comments):
# May need: file_row_hash = hashlib.md5(json.dumps(order_id + "_EXP_A").encode('utf-8')).hexdigest()
```

**Impacto:**
- Hash match rate: **74.27%** (vs objetivo 95%+)
- No-matching: 397 expired options (25.73%)
- Data integrity: 100% ✓ (datos reales son correctos)
- Riesgo: Expired options pueden duplicarse en re-import

**Solución Recomendada:**

```python
# deribit.py línea ~238
def compute_file_row_hash(order_id: str, instrument_name: str) -> str:
    """
    Compute hash with expired options handling.

    For expired options, append "_EXP_A" suffix to match legacy.
    """
    # Check if instrument is expired option
    # Pattern: BTC-DDMMMYY-STRIKE-C/P
    option_pattern = r'^[A-Z]+-\d{1,2}[A-Z]{3}\d{2}-\d+-[CP]$'

    if re.match(option_pattern, instrument_name):
        # Parse expiry date
        parts = instrument_name.split('-')
        if len(parts) == 4:
            expiry_str = parts[1]  # e.g., "26JUL24"
            try:
                expiry_date = datetime.strptime(expiry_str, '%d%b%y')
                # Check if expired
                if expiry_date < datetime.now():
                    order_id = order_id + "_EXP_A"
            except ValueError:
                pass  # Not a valid date, skip suffix

    return hashlib.md5(json.dumps(order_id).encode('utf-8')).hexdigest()
```

**Archivo:** `deribit.py` líneas 230-240
**Estimado:** 1-2 días (incluye investigación + testing)

**Tests Requeridos:**
```python
def test_expired_option_hash_suffix():
    """Verifica que expired options usan _EXP_A suffix"""
    # Expired option
    order_id = "12345"
    instrument = "BTC-26JUL24-56000-P"  # Expired

    hash_result = compute_file_row_hash(order_id, instrument)

    # Should match legacy formula with suffix
    expected = hashlib.md5(json.dumps("12345_EXP_A").encode('utf-8')).hexdigest()
    assert hash_result == expected

def test_active_option_hash_no_suffix():
    """Verifica que active options NO usan suffix"""
    order_id = "12345"
    instrument = "BTC-26DEC26-80000-C"  # Future expiry

    hash_result = compute_file_row_hash(order_id, instrument)

    # Should NOT have suffix
    expected = hashlib.md5(json.dumps("12345").encode('utf-8')).hexdigest()
    assert hash_result == expected
```

---

### CATEGORÍA 2: HIGH - Data Validation Issues

#### ⭐⭐⭐ 2. Symbol Validation (HIGH)

**Estado:** FALTANTE

**Ubicación Legacy:** `brokers_deribit.py:91, 233`
```python
if not n['symbol'] or not n['date'] or not n['price'] or not action:
    continue
```

**Estado Nuevo:** Sin validación de symbol vacío

**Impacto si falta:**
- Symbols vacíos pueden entrar al sistema
- Errores en stages posteriores

**Implementación:**
```python
# deribit.py después de línea 180
instrument_name = order.get("instrument_name", "")
if not instrument_name or not instrument_name.strip():
    logger.warning(f"[DERIBIT] Skipping order {order_id}: empty instrument_name")
    continue
```

**Archivo:** `deribit.py` línea ~180
**Estimado:** 0.25 días

---

#### ⭐⭐⭐ 3. Price Validation (HIGH)

**Estado:** FALTANTE

**Ubicación Legacy:** `brokers_deribit.py:91, 233`
```python
if not n['price']:
    continue
```

**Estado Nuevo:** Sin validación de price zero/negativo

**Implementación:**
```python
# deribit.py después de línea 185
price = float(order.get("price", 0) or 0)
if price <= 0:
    logger.warning(f"[DERIBIT] Skipping order {order_id}: zero or negative price {price}")
    continue
```

**Archivo:** `deribit.py` línea ~185
**Estimado:** 0.25 días

---

#### ⭐⭐ 4. Quantity Validation (MEDIUM-HIGH)

**Estado:** FALTANTE

**Ubicación Legacy:** Implícito en amount/quantity handling

**Implementación:**
```python
# deribit.py después de línea 190
amount = float(order.get("amount", 0) or 0)
contracts = float(order.get("contracts", 0) or 0)

if amount <= 0 and contracts <= 0:
    logger.warning(f"[DERIBIT] Skipping order {order_id}: zero quantity")
    continue
```

**Archivo:** `deribit.py` línea ~190
**Estimado:** 0.25 días

---

#### ⭐⭐ 5. Timestamp Validation (MEDIUM)

**Estado:** FALTANTE

**Ubicación Legacy:** `brokers_deribit.py:97`
```python
if ImportParams.isfloat(n['date'].strip()):
```

**Implementación:**
```python
# deribit.py después de línea 195
timestamp = order.get("timestamp")
if not timestamp:
    logger.warning(f"[DERIBIT] Skipping order {order_id}: missing timestamp")
    continue
```

**Archivo:** `deribit.py` línea ~195
**Estimado:** 0.25 días

---

#### ⭐⭐ 6. Trade Type Validation (MEDIUM-HIGH)

**Estado:** IMPLEMENTADO pero puede mejorar

**Ubicación Actual:** `deribit.py:170-173`
```python
trade_type = order.get("type", "")
if trade_type != "trade":
    continue
```

**Estado:** ✅ IMPLEMENTADO (no changes needed)

---

### CATEGORÍA 3: MEDIUM - Data Quality Improvements

#### ⭐ 7. Side Mapping Completeness (MEDIUM)

**Estado:** PARCIALMENTE IMPLEMENTADO

**Ubicación Legacy:** `deribit_export.py:330-336` (6 side combinations)
```python
# open buy, close short, liquidation buy → BUY
# open sell, close long, liquidation sell → SELL
```

**Ubicación Nueva:** `deribit.py:201-206` (similar pero sin documentación de liquidation)

**Mejora:**
```python
# deribit.py líneas 201-206 (documentar mejor)
# Compound side values from Deribit API:
# - "open buy", "close short", "liquidation buy" → BUY
# - "open sell", "close long", "liquidation sell" → SELL
side_raw = order.get("side", "").lower()
side = "BUY" if any(x in side_raw for x in ["buy", "short"]) else \
       "SELL" if any(x in side_raw for x in ["sell", "long"]) else ""

if not side:
    logger.warning(f"[DERIBIT] Skipping order {order_id}: unknown side '{side_raw}'")
    continue
```

**Archivo:** `deribit.py` líneas 201-206
**Estimado:** 0.25 días

---

### CATEGORÍA 4: CONDITIONAL - CSV Support

#### 🎯 8. CSV Parser Support (CONDICIONAL)

**Estado:** NO IMPLEMENTADO

**Legacy Scope:**
- `brokers_deribit.py` - 326 líneas con 2 CSV formats:
  1. `getcsv()` - CSV v1 (lines 20-180)
  2. `getcsv2()` - CSV v2 with "TRADE ID" header (lines 182-326)

**Decisión Requerida:** Ejecutar SQL query

```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*)) OVER (), 2) as percentage,
    MAX(created_at) as last_used
FROM data_sources
WHERE broker_id = 'deribit'
  AND created_at > NOW() - INTERVAL '12 months'
GROUP BY source_type
ORDER BY count DESC;
```

**Criterio:**
- Si CSV percentage > 5%: **IMPLEMENTAR** (5-8 días)
- Si CSV percentage < 5%: **OMITIR**

**Complejidad:** ALTA (2 formats, options parsing, ~326 líneas)
**Estimado:** 5-8 días (solo si necesario)

**CSV Features requeridas:**
- Format detection (v1 vs v2)
- Side mapping (compound values)
- Symbol normalization
- Options parsing (strike, expiry, type)
- Fee extraction
- Date parsing (Unix timestamp vs string)
- Quantity source priority (quantity → amount)
- Price decimal calculation

---

### CATEGORÍA 5: OUT OF SCOPE (Correctamente Excluido)

#### ✅ 9. Credential Validation
**Legacy:** `deribit_export.py:149`
**Conclusión:** **OUT OF SCOPE** - Pertenece a **sync service**

#### ✅ 10. Test Account Toggle
**Legacy:** `deribit_export.py:162`
**Conclusión:** **OUT OF SCOPE** - Pertenece a **sync service**

#### ✅ 11. API Rate Limiting
**Legacy:** `deribit_export.py:265-271`
**Conclusión:** **OUT OF SCOPE** - Pertenece a **sync service**

#### ✅ 12. Pagination Logic
**Legacy:** `deribit_export.py:261, 276-277`
**Conclusión:** **OUT OF SCOPE** - Pertenece a **sync service**

#### ✅ 13. Deduplication Check via DB
**Legacy:** `deribit_export.py:417-428`
**Conclusión:** **OUT OF SCOPE** - Pertenece to **p05_write** stage

---

### CATEGORÍA 6: ALREADY IMPLEMENTED ✅

#### 14. ✅ Trade Type Filtering
**Estado:** IMPLEMENTADO (`deribit.py:170-173`)

#### 15. ✅ JSON Validation
**Estado:** IMPLEMENTADO (parse_json_content validates JSON)

#### 16. ✅ Timestamp Conversion
**Estado:** IMPLEMENTADO (`deribit.py:217-227`)
- Converts ms → UTC → America/New_York

#### 17. ✅ Symbol Normalization
**Estado:** IMPLEMENTADO (`deribit.py:282-285`)
- Uppercase, remove underscores

#### 18. ✅ Option Parsing
**Estado:** IMPLEMENTADO (`deribit.py:291-308`)
- Regex pattern: `^([A-Z]+)-(\d{1,2}[A-Z]{3}\d{2})-(\d+)-([CP])$`
- Extracts: underlying, expiry, strike, type

#### 19. ✅ Asset Type Classification
**Estado:** IMPLEMENTADO (`deribit.py:382-387`)
- Perpetuals → crypto
- Options → options

#### 20. ✅ Commission Calculation
**Estado:** IMPLEMENTADO (`deribit.py:345-351`)
- Options: commission × price
- Perpetuals: commission × index_price

#### 21. ✅ Pip Value Calculation
**Estado:** IMPLEMENTADO (`deribit.py:359-367`)
- balance / quantity (with absolute value)

#### 22. ✅ Column Detection
**Estado:** IMPLEMENTADO (`detector.py:21`)
- Required: trade_id, instrument_name, side, amount, price, timestamp

---

## Plan de Implementación por Fases

### FASE 1: Critical Hash Fix (1-2 días) ⚠️ URGENTE

**Objetivo:** Corregir 74.27% hash match rate a 95%+

**Tareas:**
1. Investigar expired options hash formula (0.5 días)
2. Implementar "_EXP_A" suffix detection (0.5 días)
3. Testing con expired + active options (0.5 días)
4. Validación con datos reales (0.5 días)

**Archivos:**
- `deribit.py:230-240` (hash computation)

**Métricas de Éxito:**
- Hash match rate >= 95% (vs actual 74.27%)
- Expired options correctamente deduplicados
- Active options sin cambios
- Data integrity mantenida (100%)

**Complejidad:** MEDIA
**Riesgo:** MEDIO (requires understanding legacy "EXP_A" logic)
**Estimado:** 1-2 días

---

### FASE 2: Data Validation (1-1.5 días)

**Objetivo:** Prevenir datos inválidos

**Tareas:**
1. Symbol validation (0.25 días)
2. Price validation (0.25 días)
3. Quantity validation (0.25 días)
4. Timestamp validation (0.25 días)
5. Side mapping improvement (0.25 días)

**Archivos:** `deribit.py:180-210`

**Métricas de Éxito:**
- Zero symbols vacíos
- Zero prices zero/negativo
- Zero quantities zero
- Zero timestamps missing
- Rejection rate < 0.1%

**Complejidad:** BAJA
**Riesgo:** BAJO
**Estimado:** 1-1.5 días

---

### FASE 3: CSV Support (5-8 días - SI NECESARIO) 🎯

**BLOQUEADOR:** Ejecutar SQL query primero

**Criterio:**
- Si CSV > 5%: **IMPLEMENTAR**
- Si CSV < 5%: **OMITIR**

**Complejidad:** ALTA (2 formats, options parsing)
**Riesgo:** MEDIO
**Estimado:** 5-8 días (solo si necesario)

---

## Matriz de Priorización

| # | Validación | Criticidad | Estado | Complejidad | Fase | Días |
|---|------------|-----------|--------|-------------|------|------|
| 1 | Expired Options Hash | ⚠️⚠️⚠️ CRÍTICO | MISSING | MEDIA | 1 | 1-2 |
| 2 | Symbol Validation | ⭐⭐⭐ ALTA | MISSING | BAJA | 2 | 0.25 |
| 3 | Price Validation | ⭐⭐⭐ ALTA | MISSING | BAJA | 2 | 0.25 |
| 4 | Quantity Validation | ⭐⭐ MEDIA-ALTA | MISSING | BAJA | 2 | 0.25 |
| 5 | Timestamp Validation | ⭐⭐ MEDIA | MISSING | BAJA | 2 | 0.25 |
| 6 | Trade Type | ✅ DONE | DONE | - | - | 0 |
| 7 | Side Mapping | ⭐ MEDIA | PARCIAL | BAJA | 2 | 0.25 |
| 8 | CSV Support | 🎯 CONDICIONAL | NOT IMPL | ALTA | 3 | 5-8 |
| 9-13 | Out of Scope | ✅ N/A | CORRECTO | - | - | 0 |
| 14-22 | Ya Implementadas | ✅ DONE | DONE | - | - | 0 |

**Total Fase 1 (Hash Fix):** 1-2 días
**Total Fase 2 (Validations):** 1-1.5 días
**Total Fase 1-2:** 2-3.5 días
**Total Fase 3 (Condicional):** 5-8 días adicionales si necesario

---

## Comparación con Otros Brokers

| Broker | Legacy Lines | New Lines | Reduction | Hash Match | Critical Issues | Phase 1-2 Days |
|--------|--------------|-----------|-----------|------------|----------------|----------------|
| **Deribit** | **326** | **516** | **-58%*** | **74.27%** ⚠️ | **5** | **2-3.5** |
| Charles Schwab | 3,196 | 1,087 | 66% | 42% ⚠️ | 6 | 3.5-5.5 |
| OKX | 170 | 362 | -113% | 95-100% ✅ | 5 | 1.25-1.75 |
| KuCoin | 2,015 | 404 | 80% | 100% | 4-5 | 2.5-3.5 |
| Kraken | 787 | 361 | 54% | 100% | 3-4 | 2 |

*Nota: Líneas nuevas incluyen 616 líneas de tests comprehensivos (legacy: 0 tests)

**Observaciones Deribit:**
- **Issue crítico moderado** - 74.27% hash match (expired options)
- **Más complejo que OKX** - Soporta crypto options (strike, expiry, type)
- **Strong test coverage** - 616 líneas de tests
- **Esfuerzo medio** (2-3.5 días) - Necesita hash fix + validaciones básicas
- **Expansion de código es positiva** - Incluye tests + mejor estructura

---

## Archivos Críticos

### 1. `brokers/deribit/deribit.py` (PRINCIPAL)
**Líneas Actuales:** 516
**Estimadas Post-Cambios:** ~570-600 (Fases 1-2), ~850-950 (si Fase 3)

**Cambios:**
- Líneas 230-240: Hash computation con expired options (Fase 1 - CRÍTICO)
- Líneas 180-210: parse_json_content() validations (Fase 2)

### 2. `tests/brokers/test_deribit.py` (TESTS)
**Líneas Actuales:** 616
**Estimadas Post-Cambios:** 700-750

**Tests Nuevos Requeridos:**
- test_expired_option_hash_suffix()
- test_active_option_hash_no_suffix()
- test_symbol_empty_validation()
- test_price_zero_validation()
- test_quantity_zero_validation()
- test_timestamp_missing_validation()
- test_side_mapping_liquidation()

### 3. Legacy Files (REFERENCIA SOLO)
- `old_code_from_legacy/brokers_deribit.py` (326 líneas - CSV parsers)
- `old_code_from_legacy/deribit_export.py` (549 líneas - API sync service)

---

## Decisiones Pendientes

### Decisión 1: Expired Options Hash Strategy

**Pregunta:** ¿Cómo handling expired options hash?

**Opciones:**
- **A: Investigar + implementar "_EXP_A" suffix (RECOMENDADO)**
  - Expected: 95%+ match rate
  - Requiere: expiry date parsing + comparison
  - Esfuerzo: 1-2 días

- **B: Aceptar 74.27% match rate (NO RECOMENDADO)**
  - No code changes
  - Expired options pueden duplicarse

**Recomendación:** **Opción A**

### Decisión 2: CSV Support Necessity

**Pregunta:** ¿Implementar 2 CSV parsers (326 líneas)?

**Data Requerida:** Ejecutar SQL query

**Criterio:**
- Si CSV > 5%: **IMPLEMENTAR** (5-8 días)
- Si CSV < 5%: **OMITIR**

**Recomendación:** **Query data first, then decide**

---

## Conclusión

Análisis identifica **22 categorías de validaciones**:
- **1 CRÍTICA (Fase 1):** 1-2 días - Expired options hash fix
- **5 HIGH/MEDIUM (Fase 2):** 1-1.5 días - Data validations
- **1 CONDICIONAL (Fase 3):** 5-8 días si necesario
- **5 OUT OF SCOPE:** Sin acción (correctamente excluidas)
- **9 YA IMPLEMENTADAS:** Sin acción

**Hallazgo Clave:** Nueva implementación es **buena** con strong test coverage:
- Issue crítico con expired options (74.27% match)
- Arquitectura limpia con options support
- Comprehensive tests (616 líneas)
- Solo necesita hash fix + validaciones básicas

**Diferencia vs Charles Schwab:**
- Deribit: 74.27% hash match (expired options issue)
- Schwab: 42% hash match (closingPrice volatility)
- Ambos tienen issues de hash pero diferentes causas

**RECOMENDACIÓN FINAL:**
1. **Implementar Fase 1 inmediatamente** (1-2 días) - Hash fix URGENTE
2. **Implementar Fase 2** (1-1.5 días) - Data validations
3. **Ejecutar SQL query antes de Fase 3** - Decisión sobre CSV
4. **NO sobre-ingenierizar** - La implementación actual es sólida

Total estimado crítico: **2-3.5 días**
Total con CSV condicional: **7-11.5 días** (solo si datos muestran necesidad)

---

## Verificación

### Tests a Ejecutar Post-Implementación

**Unit Tests:**
```bash
pytest tests/brokers/test_deribit.py -v
```

**Integration Tests:**
```python
def test_hash_match_rate_expired_options():
    # Load sample with expired options
    # Process with new code
    # Compare hashes
    # Assert match_rate >= 0.95

def test_rejection_rate():
    # Load 1000 sample orders
    # Assert rejection_rate < 0.001
```

**Manual Verification:**
```bash
# 1. Check expired options hash
grep "EXP_A" logs.txt

# 2. Verify hash match rate
# Compare file_row hashes with legacy data
# Target: >= 95%

# 3. Check validation warnings
grep "Skipping order" logs.txt
```

---

**Fecha de Análisis:** 2026-01-14
**Broker ID:** deribit
**Formato:** JSON API (crypto perpetuals + European-style options)
**Assets:** crypto (BTC_USDT, ETH_USDT) + options (BTC-DDMMMYY-STRIKE-C/P)
