# Tracking de Implementación - Deribit Normalizer

Este documento rastrea el progreso de implementación de validaciones críticas identificadas en el análisis del Deribit normalizer.

**Fecha de Inicio:** 2026-01-14
**Broker ID:** deribit
**Hash Match Rate Actual:** 74.27% (1,146/1,543 matches) ⚠️
**Hash Match Rate Objetivo:** 95%+ ✅

---

## FASE 1: Critical Hash Fix (1-2 días) ⚠️ URGENTE

**Objetivo:** Corregir 74.27% hash match rate a 95%+

### ⚠️⚠️⚠️ 1. Expired Options Hash Mismatch (CRÍTICO)

**Criticidad:** ALTA (afecta 25.73% de opciones expiradas)
**Estado:** ❌ PENDIENTE
**Archivo:** `deribit.py` líneas 230-240
**Estimado:** 1-2 días

**Problema:**
- 397 expired options records (25.73%) usan diferente hash formula con "_EXP_A" suffix
- Actual: `MD5(json.dumps(id))`
- Expected: `MD5(json.dumps(id + "_EXP_A"))` para expired options

**Subtareas:**
- [ ] Investigar código legacy para confirmar "_EXP_A" logic (0.5 días)
- [ ] Implementar función `compute_file_row_hash()` con expiry detection (0.5 días)
- [ ] Agregar tests para expired options (test_expired_option_hash_suffix) (0.25 días)
- [ ] Agregar tests para active options (test_active_option_hash_no_suffix) (0.25 días)
- [ ] Validar con datos reales (0.5 días)

**Tests a Agregar:**
```python
def test_expired_option_hash_suffix():
    """Verifica que expired options usan _EXP_A suffix"""
    order_id = "12345"
    instrument = "BTC-26JUL24-56000-P"  # Expired

    hash_result = compute_file_row_hash(order_id, instrument)

    # Should match legacy formula with suffix
    expected = hashlib.md5(json.dumps("12345_EXP_A").encode('utf-8')).hexdigest()
    assert hash_result == expected

def test_active_option_hash_no_suffix():
    """Verifica que active options NO usan suffix"""
    order_id = "12345"
    instrument = "BTC-26DEC26-80000-C"  # Future expiry

    hash_result = compute_file_row_hash(order_id, instrument)

    # Should NOT have suffix
    expected = hashlib.md5(json.dumps("12345").encode('utf-8')).hexdigest()
    assert hash_result == expected
```

**Métricas de Éxito:**
- [ ] Hash match rate >= 95% (vs actual 74.27%)
- [ ] Expired options correctamente deduplicados
- [ ] Active options sin cambios
- [ ] Data integrity mantenida (100%)
- [ ] Tests passing: test_expired_option_hash_suffix
- [ ] Tests passing: test_active_option_hash_no_suffix

**Complejidad:** MEDIA
**Riesgo:** MEDIO (requires understanding legacy "_EXP_A" logic)

---

## FASE 2: Data Validation (1-1.5 días)

**Objetivo:** Prevenir datos inválidos

### ⭐⭐⭐ 2. Symbol Validation (HIGH)

**Estado:** ❌ PENDIENTE
**Archivo:** `deribit.py` línea ~180
**Estimado:** 0.25 días

**Implementación:**
```python
# deribit.py después de línea 180
instrument_name = order.get("instrument_name", "")
if not instrument_name or not instrument_name.strip():
    logger.warning(f"[DERIBIT] Skipping order {order_id}: empty instrument_name")
    continue
```

**Subtareas:**
- [ ] Agregar validación de symbol vacío
- [ ] Agregar test: test_symbol_empty_validation
- [ ] Verificar rejection rate < 0.1%

**Test a Agregar:**
```python
def test_symbol_empty_validation():
    """Verifica que symbols vacíos son rechazados"""
    trade = {
        "id": "12345",
        "type": "trade",
        "instrument_name": "",  # Empty symbol
        "price": 2500.0,
        "amount": 10.0,
        "timestamp": 1751852373399,
    }
    df = DeribitInterpreter.parse_json_content(json.dumps([trade]))
    assert df.height == 0  # Should be rejected
```

**Métricas de Éxito:**
- [ ] Zero symbols vacíos en output
- [ ] Warning log generado para cada rechazo
- [ ] Rejection rate < 0.1%

---

### ⭐⭐⭐ 3. Price Validation (HIGH)

**Estado:** ❌ PENDIENTE
**Archivo:** `deribit.py` línea ~185
**Estimado:** 0.25 días

**Implementación:**
```python
# deribit.py después de línea 185
price = float(order.get("price", 0) or 0)
if price <= 0:
    logger.warning(f"[DERIBIT] Skipping order {order_id}: zero or negative price {price}")
    continue
```

**Subtareas:**
- [ ] Agregar validación de price <= 0
- [ ] Agregar test: test_price_zero_validation
- [ ] Agregar test: test_price_negative_validation
- [ ] Verificar rejection rate < 0.1%

**Tests a Agregar:**
```python
def test_price_zero_validation():
    """Verifica que prices zero son rechazados"""
    trade = {
        "id": "12345",
        "type": "trade",
        "instrument_name": "BTC_USDT",
        "price": 0.0,  # Zero price
        "amount": 10.0,
        "timestamp": 1751852373399,
    }
    df = DeribitInterpreter.parse_json_content(json.dumps([trade]))
    assert df.height == 0

def test_price_negative_validation():
    """Verifica que prices negativos son rechazados"""
    trade = {
        "id": "12345",
        "type": "trade",
        "instrument_name": "BTC_USDT",
        "price": -2500.0,  # Negative price
        "amount": 10.0,
        "timestamp": 1751852373399,
    }
    df = DeribitInterpreter.parse_json_content(json.dumps([trade]))
    assert df.height == 0
```

**Métricas de Éxito:**
- [ ] Zero prices zero/negativo en output
- [ ] Warning log generado para cada rechazo
- [ ] Rejection rate < 0.1%

---

### ⭐⭐ 4. Quantity Validation (MEDIUM-HIGH)

**Estado:** ❌ PENDIENTE
**Archivo:** `deribit.py` línea ~190
**Estimado:** 0.25 días

**Implementación:**
```python
# deribit.py después de línea 190
amount = float(order.get("amount", 0) or 0)
contracts = float(order.get("contracts", 0) or 0)

if amount <= 0 and contracts <= 0:
    logger.warning(f"[DERIBIT] Skipping order {order_id}: zero quantity")
    continue
```

**Subtareas:**
- [ ] Agregar validación de quantity zero (both amount and contracts)
- [ ] Agregar test: test_quantity_zero_validation
- [ ] Verificar rejection rate < 0.1%

**Test a Agregar:**
```python
def test_quantity_zero_validation():
    """Verifica que quantities zero son rechazados"""
    trade = {
        "id": "12345",
        "type": "trade",
        "instrument_name": "BTC_USDT",
        "price": 2500.0,
        "amount": 0.0,      # Zero amount
        "contracts": None,  # No contracts
        "timestamp": 1751852373399,
    }
    df = DeribitInterpreter.parse_json_content(json.dumps([trade]))
    assert df.height == 0
```

**Métricas de Éxito:**
- [ ] Zero quantities zero en output
- [ ] Warning log generado para cada rechazo
- [ ] Rejection rate < 0.1%

---

### ⭐⭐ 5. Timestamp Validation (MEDIUM)

**Estado:** ❌ PENDIENTE
**Archivo:** `deribit.py` línea ~195
**Estimado:** 0.25 días

**Implementación:**
```python
# deribit.py después de línea 195
timestamp = order.get("timestamp")
if not timestamp:
    logger.warning(f"[DERIBIT] Skipping order {order_id}: missing timestamp")
    continue
```

**Subtareas:**
- [ ] Agregar validación de timestamp missing
- [ ] Agregar test: test_timestamp_missing_validation
- [ ] Verificar rejection rate < 0.1%

**Test a Agregar:**
```python
def test_timestamp_missing_validation():
    """Verifica que timestamps missing son rechazados"""
    trade = {
        "id": "12345",
        "type": "trade",
        "instrument_name": "BTC_USDT",
        "price": 2500.0,
        "amount": 10.0,
        "timestamp": None,  # Missing timestamp
    }
    df = DeribitInterpreter.parse_json_content(json.dumps([trade]))
    assert df.height == 0
```

**Métricas de Éxito:**
- [ ] Zero timestamps missing en output
- [ ] Warning log generado para cada rechazo
- [ ] Rejection rate < 0.1%

---

### ✅ 6. Trade Type Validation (MEDIUM-HIGH)

**Estado:** ✅ COMPLETADO
**Archivo:** `deribit.py` líneas 160-162
**Estimado:** 0 días (ya implementado)

**Implementación Actual:**
```python
trade_type = order.get("type", "")
if trade_type != "trade":
    continue
```

**Validación:**
- ✅ Filtra non-trade types (deposit, withdrawal, transfer, settlement)
- ✅ Solo procesa type="trade"

**Sin cambios requeridos** - Funciona correctamente

---

### ⭐ 7. Side Mapping Completeness (MEDIUM)

**Estado:** ⚠️ PARCIALMENTE IMPLEMENTADO
**Archivo:** `deribit.py` líneas 238-240
**Estimado:** 0.25 días

**Mejora de Documentación:**
```python
# deribit.py líneas 238-240 (mejorar documentación)
# Compound side values from Deribit API:
# - "open buy", "close short", "liquidation buy" → BUY
# - "open sell", "close long", "liquidation sell" → SELL
side_raw = order.get("side", "").lower()
side = "BUY" if any(x in side_raw for x in ["buy", "short"]) else \
       "SELL" if any(x in side_raw for x in ["sell", "long"]) else ""

if not side:
    logger.warning(f"[DERIBIT] Skipping order {order_id}: unknown side '{side_raw}'")
    continue
```

**Subtareas:**
- [ ] Agregar documentación completa de side mapping
- [ ] Agregar validación de unknown side
- [ ] Agregar test: test_side_mapping_liquidation
- [ ] Agregar test: test_side_mapping_unknown

**Tests a Agregar:**
```python
def test_side_mapping_liquidation():
    """Verifica que liquidation sides mapean correctamente"""
    # Test liquidation buy
    trade_buy = {
        "id": "12345",
        "type": "trade",
        "instrument_name": "BTC_USDT",
        "side": "liquidation buy",
        "price": 2500.0,
        "amount": 10.0,
        "timestamp": 1751852373399,
    }
    df = DeribitInterpreter.parse_json_content(json.dumps([trade_buy]))
    assert df.height == 1
    result = df.collect()
    assert result["side"][0] == "BUY"

    # Test liquidation sell
    trade_sell = {
        "id": "12346",
        "type": "trade",
        "instrument_name": "BTC_USDT",
        "side": "liquidation sell",
        "price": 2500.0,
        "amount": 10.0,
        "timestamp": 1751852373399,
    }
    df = DeribitInterpreter.parse_json_content(json.dumps([trade_sell]))
    assert df.height == 1
    result = df.collect()
    assert result["side"][0] == "SELL"

def test_side_mapping_unknown():
    """Verifica que unknown sides son rechazados"""
    trade = {
        "id": "12345",
        "type": "trade",
        "instrument_name": "BTC_USDT",
        "side": "unknown_side",  # Unknown side
        "price": 2500.0,
        "amount": 10.0,
        "timestamp": 1751852373399,
    }
    df = DeribitInterpreter.parse_json_content(json.dumps([trade]))
    assert df.height == 0  # Should be rejected
```

**Métricas de Éxito:**
- [ ] Documentación completa de 6 side combinations
- [ ] Unknown sides rechazados con warning
- [ ] Rejection rate < 0.1%

---

## FASE 3: CSV Support (5-8 días - CONDICIONAL) 🎯

**Objetivo:** Implementar soporte para 2 formatos CSV legacy

### 🎯 8. CSV Parser Support (CONDICIONAL)

**Estado:** ⏸️ BLOQUEADO (requiere SQL query)
**Complejidad:** ALTA (2 formats, options parsing, ~326 líneas)
**Estimado:** 5-8 días (solo si necesario)

**Decisión Requerida:** Ejecutar SQL query para determinar uso

```sql
SELECT
    source_type,
    COUNT(*) as count,
    COUNT(DISTINCT user_id) as user_count,
    ROUND(100.0 * COUNT(*) / SUM(COUNT(*)) OVER (), 2) as percentage,
    MAX(created_at) as last_used
FROM data_sources
WHERE broker_id = 'deribit'
  AND created_at > NOW() - INTERVAL '12 months'
GROUP BY source_type
ORDER BY count DESC;
```

**Criterio:**
- Si CSV percentage > 5%: **IMPLEMENTAR** (5-8 días)
- Si CSV percentage < 5%: **OMITIR** (no implementar)

**Legacy CSV Features (Scope si se implementa):**

**CSV Format v1 (getcsv):**
- [ ] Format detection (detect "DATE" header)
- [ ] Side mapping (buy/sell, open buy, close buy, open sell, close sell)
- [ ] Symbol normalization (BTC-PERPETUAL → BTC, remove dashes)
- [ ] Options parsing (extract strike, expiry, type)
- [ ] Fee extraction
- [ ] Unix timestamp vs string date parsing
- [ ] Quantity source (quantity → amount priority)
- [ ] Price decimal calculation

**CSV Format v2 (getcsv2):**
- [ ] Format detection (detect "TRADE ID" header)
- [ ] Different column mappings
- [ ] Future/Option type detection
- [ ] Similar processing as v1

**Subtareas (solo si se implementa):**
- [ ] Crear CSVv1Interpreter class (2 días)
- [ ] Crear CSVv2Interpreter class (2 días)
- [ ] Agregar format detection en detector.py (0.5 días)
- [ ] Tests comprehensivos para ambos formats (1.5 días)
- [ ] Documentación completa (0.5 días)

**Métricas de Éxito (si se implementa):**
- [ ] CSV v1 format correctamente parseado
- [ ] CSV v2 format correctamente parseado
- [ ] Hash match rate >= 95% para CSVs
- [ ] Tests passing para ambos formats
- [ ] Documentación actualizada

**Estado:** ⏸️ ESPERANDO DECISIÓN (ejecutar SQL query primero)

---

## Validaciones OUT OF SCOPE ✅

Estas validaciones pertenecen a otros pipeline stages y están correctamente excluidas:

### ✅ 9. Credential Validation
**Estado:** OUT OF SCOPE (sync service)
**Ubicación Legacy:** `deribit_export.py:149`
**Sin acción requerida**

### ✅ 10. Test Account Toggle
**Estado:** OUT OF SCOPE (sync service)
**Ubicación Legacy:** `deribit_export.py:162`
**Sin acción requerida**

### ✅ 11. API Rate Limiting
**Estado:** OUT OF SCOPE (sync service)
**Ubicación Legacy:** `deribit_export.py:265-271`
**Sin acción requerida**

### ✅ 12. Pagination Logic
**Estado:** OUT OF SCOPE (sync service)
**Ubicación Legacy:** `deribit_export.py:261, 276-277`
**Sin acción requerida**

### ✅ 13. Deduplication Check via DB
**Estado:** OUT OF SCOPE (p05_write stage)
**Ubicación Legacy:** `deribit_export.py:417-428`
**Sin acción requerida**

---

## Validaciones Ya Implementadas ✅

Estas validaciones ya están implementadas correctamente:

### ✅ 14. Trade Type Filtering
**Estado:** COMPLETO (`deribit.py:160-162`)
**Sin cambios requeridos**

### ✅ 15. JSON Validation
**Estado:** COMPLETO (parse_json_content validates JSON)
**Sin cambios requeridos**

### ✅ 16. Timestamp Conversion
**Estado:** COMPLETO (`deribit.py:210-227`)
**Sin cambios requeridos**

### ✅ 17. Symbol Normalization
**Estado:** COMPLETO (`deribit.py:262-268`)
**Sin cambios requeridos**

### ✅ 18. Option Parsing
**Estado:** COMPLETO (`deribit.py:110-127`)
**Sin cambios requeridos**

### ✅ 19. Asset Type Classification
**Estado:** COMPLETO (`deribit.py:447-450`)
**Sin cambios requeridos**

### ✅ 20. Commission Calculation
**Estado:** COMPLETO (`deribit.py:430-435`)
**Sin cambios requeridos**

### ✅ 21. Pip Value Calculation
**Estado:** COMPLETO (`deribit.py:185-197`)
**Sin cambios requeridos**

### ✅ 22. Column Detection
**Estado:** COMPLETO (`detector.py:21`)
**Sin cambios requeridos**

---

## Resumen de Progreso

### Fase 1: Critical Hash Fix
- **Total:** 1 validación
- **Completadas:** 0
- **Pendientes:** 1
- **Progreso:** 0%

### Fase 2: Data Validation
- **Total:** 6 validaciones
- **Completadas:** 1 (Trade Type)
- **Parcialmente Completadas:** 1 (Side Mapping)
- **Pendientes:** 4
- **Progreso:** 16.7%

### Fase 3: CSV Support
- **Total:** 1 validación condicional
- **Bloqueadas:** 1 (requiere SQL query)
- **Progreso:** 0%

### Out of Scope + Ya Implementadas
- **Total:** 14 validaciones
- **Completadas:** 14
- **Progreso:** 100%

---

## Métricas de Calidad

### Objetivos
- [ ] Hash match rate >= 95% (actual: 74.27%)
- [ ] Data rejection rate < 0.1%
- [ ] Zero symbols vacíos en output
- [ ] Zero prices zero/negativos en output
- [ ] Zero quantities zero en output
- [ ] Zero timestamps missing en output
- [ ] Test coverage >= 95%
- [ ] All tests passing

### Status Actual
- ⚠️ Hash match rate: 74.27% (objetivo: 95%)
- ⚠️ Data rejection rate: unknown (objetivo: < 0.1%)
- ✅ Test coverage: 616 líneas (comprehensive)
- ✅ Tests passing: 29/29 (100%)

---

## Próximos Pasos

### Inmediato (Esta Semana)
1. **Fase 1:** Implementar expired options hash fix (1-2 días)
2. **Fase 2:** Implementar data validations (1-1.5 días)

### Corto Plazo (Próximas 2 Semanas)
3. **Decisión CSV:** Ejecutar SQL query para determinar necessity
4. **Fase 3 (Condicional):** Implementar CSV support si es necesario (5-8 días)

### Validación Final
5. Ejecutar integration tests con datos reales
6. Validar hash match rate >= 95%
7. Validar rejection rate < 0.1%
8. Code review completo
9. Deploy a producción

---

**Última Actualización:** 2026-01-14
**Próxima Revisión:** TBD
**Responsable:** Development Team
