# Archivos de Código - Bybit Broker

## Estructura de Archivos

```
brokers/bybit/
├── bybit.py.original         # Código actual (sin modificar) - 333 líneas
└── detector.py               # Detector de formato (sin cambios) - 51 líneas
```

## Archivos Incluidos

### 1. bybit.py.original
**Descripción:** Código actual del normalizador Bybit sin modificaciones
**Líneas:** 333
**Estado:** REFERENCIA - No modificado

**Validaciones actuales:**
- ✅ Detección de formato JSON
- ✅ Parsing de campos básicos (execId, symbol, side, qty, price)
- ✅ Transformación a schema normalizado
- ✅ File row hash (MD5)
- ✅ Timezone conversion (UTC → America/New_York)
- ✅ Filtrado por execType="Trade" (excluye Funding, Liquidation, etc.)
- ✅ Category mapping (spot/linear/inverse → crypto)
- ✅ Hash computation (100% compatibilidad legacy)

**Validaciones faltantes (a implementar):**
- ❌ Validación de precio cero/negativo
- ❌ Validación de campos requeridos
- ❌ Conversión de fees por moneda (USD vs USDT)
- ❌ Cálculo de pip value para contratos inverse (CONDICIONAL)
- ❌ Detección de crypto options (CONDICIONAL)

### 2. detector.py
**Descripción:** Detector de formato Bybit
**Estado:** SIN CAMBIOS - No requiere modificación

**Funcionalidad:**
- Verifica columnas requeridas: execId, symbol, side, execQty, execPrice, execTime
- Valida formato JSON de Bybit API

---

## Cambios Propuestos por Fase

### FASE 1: Validaciones Críticas

#### Cambio 1: Validación de Precio
**Ubicación:** Línea ~250 en bybit.py

**Antes:**
```python
.with_columns([
    pl.col("execPrice").cast(pl.Float64).alias("price")
])
```

**Después:**
```python
.filter(pl.col("execPrice") > 0)  # Rechazar precio cero o negativo
.with_columns([
    pl.col("execPrice").cast(pl.Float64).round(6).alias("price")  # Redondear a 6 decimales
])
```

---

#### Cambio 2: Validación de Campos Requeridos
**Ubicación:** Línea ~120-180 en bybit.py (parse_json_content loop)

**Añadir después de línea 123:**
```python
# Validación de campos requeridos
if not exec_id or not symbol or not side or not exec_time:
    logger.warning(f"Skipping execution with missing fields: execId={exec_id}")
    continue

if exec_price <= 0 or exec_qty <= 0:
    logger.warning(f"Skipping execution with invalid price/qty: execId={exec_id}")
    continue

if side.upper() not in ("BUY", "SELL"):
    logger.warning(f"Skipping execution with invalid side: execId={exec_id}")
    continue
```

---

### FASE 2: Validaciones de Calidad

#### Cambio 3: Conversión de Fees por Moneda
**Ubicación:** Línea ~267 en bybit.py

**Antes:**
```python
.with_columns([
    pl.col("execFee").abs().alias("fees")
])
```

**Después:**
```python
.with_columns([
    pl.when(pl.col("symbol").str.contains("USDT|PERP|USDC"))
      .then(pl.col("execFee").abs())  # Ya en USDT
    .when(pl.col("symbol").str.ends_with("USD"))
      .then(pl.col("execFee").abs() * pl.col("execPrice"))  # Convertir USD
    .otherwise(pl.col("execFee").abs())
    .alias("fees")
])
```

---

### FASE 3: Asset-Específico (CONDICIONAL)

#### Cambio 4: Cálculo de Pip Value (CONDICIONAL)
**Ubicación:** Línea ~293 en bybit.py
**⚠️ Solo si usuarios operan contratos inverse (BTCUSD, ETHUSD)**

**Modificar:**
```python
.with_columns([
    pl.when(pl.col("symbol").str.contains("USDT|PERP|USDC"))
      .then(pl.lit(1.0))  # USDT-margined
    .when(pl.col("symbol").str.ends_with("USD") & (pl.col("execValue") > 0))
      .then((pl.col("execValue") / (pl.col("execQty") * pl.col("execPrice"))) * pl.col("execPrice"))
    .otherwise(pl.lit(1.0))
    .alias("pip_value")
])
```

---

#### Cambio 5: Detección de Crypto Options (CONDICIONAL)
**Ubicación:** Nuevo método + integración en parse_json_content
**⚠️ Solo si usuarios operan Bybit crypto options**

**Añadir nuevo método:**
```python
def parse_bybit_option_symbol(symbol: str) -> dict:
    """
    Parse Bybit option format: BTC-31MAY25-80000-C
    Returns: {is_option, base_symbol, expire_date, strike, option_type}
    """
    parts = symbol.split('-')
    if len(parts) != 4:
        return {"is_option": False}

    base, expiry_str, strike_str, type_code = parts
    # ... parsing logic ...
```

---

## Implementación

### Para implementar los cambios:

1. **Fase 1 (Crítica):**
   ```bash
   # Copiar archivo original
   cp bybit.py.original ../../brokers/bybit/bybit.py

   # Aplicar cambios 1, 2
   # Editar bybit.py según cambios propuestos arriba
   ```

2. **Fase 2 (Calidad):**
   ```bash
   # Aplicar cambio 3
   ```

3. **Fase 3 (Condicional):**
   ```bash
   # Investigar primero si aplicable
   # Si SÍ → Aplicar cambios 4 y/o 5
   ```

4. **Testing:**
   ```bash
   # Ejecutar tests
   pytest ../../tests/brokers/test_bybit.py -v

   # Validación QA
   python qa/validate_broker.py --broker-id 236
   ```

---

## Referencias

- **Plan completo:** `../PLAN_ANALISIS_VALIDACIONES_BYBIT.md`
- **Código legacy:** `../old_code_from_legacy/bybit_export.py`, `../old_code_from_legacy/brokers_bybit.py`
- **Log de cambios:** `../CAMBIOS_IMPLEMENTADOS.md`
- **Ejemplos de código:** `../EJEMPLOS_CAMBIOS_CODIGO.md`

---

## Métricas Actuales

- **Hash Compatibility:** 100% ✅
- **Supported Features:**
  - JSON API parsing ✅
  - execId deduplication (via p02) ✅
  - Timezone handling ✅
  - Basic fee calculation ✅
  - Category normalization ✅

- **Missing Features:**
  - Price validation ❌
  - Field validation ❌
  - Fee currency conversion ❌
  - Inverse contract support ❌ (condicional)
  - Crypto options support ❌ (condicional)

---

**Estado Actual:** ✅ CÓDIGO REFERENCIA COMPLETO
**Siguiente Paso:** Implementar Fase 1 (validaciones críticas)
