# Plan: Análisis de Validaciones Críticas Faltantes - Bybit Normalizer

## Resumen Ejecutivo

Tras analizar exhaustivamente ambas implementaciones (legacy: 1,584 líneas en `bybit_export.py` + `brokers_bybit.py` vs nueva: 384 líneas en `bybit.py` + `detector.py`), se identificaron **10 categorías de validaciones**, de las cuales **solo 2-3 son críticas** para integridad de datos.

**Hallazgo clave**: La nueva implementación es **intencionalmente más simple** y **arquitecturalmente superior**. Muchas "validaciones faltantes" son en realidad:
- Artefactos de CSV (no necesarios para JSON API)
- Manejadas downstream (p02_deduplicate, p04_calculate)
- Deuda técnica legacy (workarounds de microservicios, usuarios especiales)

## Contexto de la Migración

### Arquitectura Legacy
- Dos archivos principales: `bybit_export.py` (941 líneas) + `brokers_bybit.py` (643 líneas)
- 3 parsers CSV diferentes (getcsv, getcsv2, getcsv3)
- Validación de credenciales con API key/secret
- 3 métodos de detección de duplicados (execId, hash JSON, clave compuesta)
- Cálculo complejo de pip value (4 rutas diferentes)
- Conversión de fees con múltiples monedas (BTC, USD, USDT)
- Detección de crypto options desde formato de 4 partes
- Manejo dual local/Lambda para usuarios especiales
- Rate limiting con backoff basado en headers
- Ajuste de receive window para APIs

### Arquitectura Nueva
- Archivo único: `bybit.py` (333 líneas) + `detector.py` (51 líneas)
- Solo JSON API (sin soporte CSV)
- Enfoque en BaseInterpreter pattern
- Separación de responsabilidades: normalización en p01, deduplicación en p02
- Hash computation con 100% compatibilidad legacy
- Timezone handling (UTC → America/New_York)
- Filtrado execType="Trade" solamente
- Mapeo de categorías (spot/linear/inverse → crypto)

### Arquitectura del Pipeline (Confirmada)
```
p01_normalize → p02_deduplicate → p03_group → p04_calculate → p05_write
     ↓                 ↓                ↓            ↓             ↓
  Parse JSON      Hash + execId    Group trades   Pip value      DB write
  Normalize       vs DB            Spreads        Multiplier
  Validate        Skip dups        Orders         Fee calc
```

---

## Validaciones Críticas Identificadas

### CATEGORÍA 1: VALIDACIONES DE INTEGRIDAD DE DATOS [CRITICIDAD: ALTA]

#### ⭐ 1. Validación de Precio (Crítico)
**Ubicación Legacy:** `bybit_export.py:792-796, 808-811` + `brokers_bybit.py:321-325`
```python
# bybit_export.py
if float(order['execPrice']) != 0:
    order['quantity'] = float(order['execQty']) / float(order['execPrice'])
else:
    continue  # Skip if price is 0

# brokers_bybit.py (CSV)
fp = n['price'].replace(',', '').replace('$','')
decimal = fp[::-1].find('.')
decimal = decimal if decimal > 1 else 2
price = round(float(fp), decimal)
```

**Descripción:** El legacy valida que el precio no sea cero (división por cero) y redondea a lugares decimales apropiados.

**Impacto si falta:**
- Errores de división por cero en cálculos
- Pérdida de precisión en precios
- Inconsistencia en display de precios
- Cálculos de quantity incorrectos

**Nivel de Criticidad:** ⭐⭐⭐ **ALTA** - Afecta integridad de datos y cálculos

**Recomendación de implementación:**
```python
# En normalize() método, después de parsear price
.filter(pl.col("execPrice") > 0)  # Rechazar precio cero o negativo
.with_columns([
    pl.col("execPrice").round(6).alias("price")  # Redondear a 6 decimales
])
```

**Ubicación sugerida:** `bybit.py` línea ~250, en sección de price

---

#### 🔍 2. Validación de Campos Requeridos
**Ubicación Legacy:** `bybit_export.py:797-798` + `brokers_bybit.py:297-303`
```python
# bybit_export.py (API)
if not order['symbol'] or not order['date'] or not order['price'] or not action or order['type'] != "TRADE":
    continue

# brokers_bybit.py (CSV)
if not n['symbol'] \
    or not n['date'] \
    or not n['price'] \
    or action not in ["BUY","SELL"] \
    or n['filled type'].upper() not in ['TRADE','LIQUIDATION','FILLED'] \
    or not n['quantity']:
    continue
```

**Descripción:** Valida que campos críticos no estén vacíos y que el action sea válido.

**Impacto si falta:**
- Registros con datos faltantes en DB
- Errores en procesamiento downstream
- Cálculos incorrectos por datos incompletos

**Nivel de Criticidad:** 🟡 **MEDIA** - Defensive programming, API debería enviar datos válidos

**Recomendación de implementación:**
```python
# En parse_json_content(), añadir validación
if not exec_id or not symbol or not side or not exec_time:
    logger.warning(f"Skipping execution with missing fields: {exec_id}")
    continue

if exec_price <= 0 or exec_qty <= 0:
    logger.warning(f"Skipping execution with invalid price/qty: {exec_id}")
    continue

if side.upper() not in ("BUY", "SELL"):
    logger.warning(f"Skipping execution with invalid side: {exec_id}")
    continue
```

**Ubicación sugerida:** `bybit.py` línea ~120-180, en parse_json_content loop

---

### CATEGORÍA 2: VALIDACIONES DE CALIDAD DE DATOS [CRITICIDAD: MEDIA]

#### 💰 3. Conversión de Fees por Moneda
**Ubicación Legacy:** `bybit_export.py:903-925, 885-892`
```python
# Fee conversion logic
if 'execFee' in order:
    order['fees'] = (float(order['execFee']) * float(pip_value_btc))
elif 'cumExecFee' in order:
    order['fees'] = order['cumExecFee']
else:
    order['fees'] = (float(order['feeAmount']) * float(pip_value_btc))

# USD symbol conversion
if ImportParams.isfloat(order['fees']) and 'USD' in order['symbol'] \
      and not 'USDT' in order['symbol'] and not 'BTCUSD' in order['symbol']:
    order['fees'] = float(order['fees']) * float(price)
    convertFees = False

# Fee currency conversion
if 'feeCurrency' in order and order['feeCurrency'] not in ['USDT'] \
      and 'USDT' in order['symbol'] and order['feeCurrency'] in order['symbol']:
    order['fees'] = float(order['fees']) * float(price)
```

**Descripción:** Convierte fees a USDT basándose en la moneda del fee (BTC, USD, etc.) para reporte consistente.

**Impacto si falta:**
- Fees reportados en moneda incorrecta
- Inconsistencia en totales de fees
- Errores en P&L si fees no en moneda base

**Nivel de Criticidad:** 🟡 **MEDIA** - Afecta precisión de fees para contratos inverse/USD

**Recomendación de implementación (Fase 1 - Basado en símbolo):**
```python
# En normalize(), sección de fees
.with_columns([
    pl.when(pl.col("symbol").str.contains("USDT|PERP|USDC"))
      .then(pl.col("execFee").abs())  # Ya en USDT
    .when(pl.col("symbol").str.ends_with("USD"))
      .then(pl.col("execFee").abs() * pl.col("price"))  # Convertir USD
    .otherwise(pl.col("execFee").abs())
    .alias("fees")
])
```

**Ubicación sugerida:** `bybit.py` línea ~267, donde se calcula fees

---

#### 🔄 4. Cálculo de Pip Value
**Ubicación Legacy:** `bybit_export.py:862-892` (4 rutas diferentes)
```python
# Path 1: No execValue
if not 'execValue' in order:
    pip_value = self.convert_usdt(order, pip_value_order, date)

# Path 2: USD symbols
elif sym in ['USD']:
    value_order = float(order['execQty']) * float(order['execPrice'])
    exec_value = float(order['execValue'])
    pip_value_exec = (exec_value / value_order)
    pip_value = pip_value_exec * price

# Path 3: USDT/PERP/USDC → pip_value = 1
elif 'USDT' in order['symbol'] or 'PERP' in order['symbol']:
    pip_value = 1

# Path 4: Others
else:
    # Complex calculation using execValue
```

**Descripción:** Calcula factor de conversión para contratos inverse (BTCUSD) vs USDT-margined.

**Impacto si falta:**
- Pip value incorrecto para contratos inverse
- Multiplier incorrecto en p04_calculate
- Sizing de posiciones incorrecto

**Nivel de Criticidad:** 🔵 **BAJA** - Solo afecta contratos inverse, mayoría es USDT-margined

**Recomendación de implementación:**
```python
# Solo si usuarios tienen contratos inverse
.with_columns([
    pl.when(pl.col("symbol").str.contains("USDT|PERP|USDC"))
      .then(pl.lit(1.0))
    .when(pl.col("symbol").str.ends_with("USD") & (pl.col("execValue") > 0))
      .then((pl.col("execValue") / (pl.col("execQty") * pl.col("execPrice"))) * pl.col("price"))
    .otherwise(pl.lit(1.0))
    .alias("pip_value")
])
```

**Ubicación sugerida:** `bybit.py` línea ~293, donde se define pip_value

**DECISIÓN NECESARIA:** ¿Usuarios tienen contratos inverse (BTCUSD, ETHUSD)?

---

### CATEGORÍA 3: VALIDACIONES ASSET-ESPECÍFICAS [CRITICIDAD: CONDICIONAL]

#### 🎯 5. Detección de Crypto Options
**Ubicación Legacy:** `bybit_export.py:854-860`
```python
if len(order['symbol']) > 8 and len(order['symbol'].split('-')) == 4:
    split_symbol = order['symbol'].split('-')
    type = 'crypto option'
    option = 'CALL' if split_symbol[3].upper() == 'C' else 'PUT'
    strike = split_symbol[2]
    expire = str(parse(split_symbol[1]).strftime("%d %b %y")).upper()
    order['symbol'] = '#{}{}'.format(split_symbol[0],'USDT')
```

**Formato de símbolo:** `BTC-31MAY25-80000-C` → Base-Expiry-Strike-Type

**Descripción:** Detecta y parsea symbols de crypto options (formato de 4 partes con guiones).

**Impacto si falta:**
- Options tratadas como crypto regular
- Strike/expire no capturados
- Grouping incorrecto en p03_group
- Asset type incorrecto

**Nivel de Criticidad:** 🎯 **CONDICIONAL** - Solo si usuarios operan Bybit options

**Recomendación de implementación:**
```python
def parse_bybit_option_symbol(symbol: str) -> dict:
    parts = symbol.split('-')
    if len(parts) != 4:
        return {"is_option": False}

    base, expiry_str, strike_str, type_code = parts
    expiry_dt = datetime.strptime(expiry_str, "%d%b%y")
    expiry_iso = expiry_dt.strftime("%Y-%m-%d")
    strike = float(strike_str)
    option_type = "CALL" if type_code.upper() == "C" else "PUT"

    return {
        "is_option": True,
        "base_symbol": f"#{base}USDT",
        "expire_date": expiry_iso,
        "strike": strike,
        "option_type": option_type,
    }

# En parse_json_content()
if category == "option":
    option_info = parse_bybit_option_symbol(symbol)
```

**Ubicación sugerida:** Nuevo método helper + integración en parse_json_content

**DECISIÓN NECESARIA:** ¿Usuarios operan Bybit crypto options?

---

### CATEGORÍA 4: VALIDACIONES YA MANEJADAS [NO REQUIEREN ACCIÓN]

#### ✅ 6. Filtrado de Funding Executions
**Ubicación Legacy:** `bybit_export.py:765-767`
**Estado Nuevo:** ✅ **YA IMPLEMENTADO** en línea 121-123
```python
# Legacy
if order['execType'].lower() == 'funding' and not self.funding:
    continue

# Nuevo (MEJOR)
if exec_type != "Trade":
    continue
```

**Conclusión:** Nueva implementación es superior (filtra más temprano, más simple)

---

#### ✅ 7. Detección de Duplicados (execId)
**Ubicación Legacy:** `bybit_export.py:773-778`
**Estado Nuevo:** ✅ **MANEJADO EN p02_deduplicate**
```python
# Legacy (in-memory)
if not order['execId'] in self.orders_execid:
    self.orders_execid.append(order['execId'])
else:
    continue

# Nuevo (database-backed en p02_deduplicate)
pl.col("execId").alias("execution_id")
# p02 queries DB for existing execution_id (ticket field)
```

**Conclusión:** Separación de responsabilidades correcta, p02 maneja deduplicación

---

### CATEGORÍA 5: ARTEFACTOS LEGACY [NO IMPLEMENTAR]

#### ❌ 8. Soporte de Formatos CSV
**Ubicación Legacy:** `brokers_bybit.py` - 3 parsers (571 líneas total)
- getcsv() - Formato básico (142 líneas)
- getcsv2() - Con "Filled Type" (211 líneas)
- getcsv3() - Entry/Exit price (218 líneas)

**Estado Nuevo:** Solo JSON API

**Conclusión:** NO IMPLEMENTAR - Arquitectura nueva es API-only por diseño

---

#### ❌ 9. Cálculo de Quantity desde Price
**Ubicación Legacy:** `bybit_export.py:792-796`
```python
if float(order['execPrice']) != 0:
    order['quantity'] = float(order['execQty']) / float(order['execPrice'])
else:
    continue
```

**Análisis:** Esta fórmula es matemáticamente incorrecta (quantity/price no tiene sentido)
- Probablemente bug legacy o workaround CSV-específico
- API proporciona execQty directamente (es autoritativo)

**Conclusión:** NO IMPLEMENTAR - Nueva implementación es correcta

---

#### ❌ 10. Validación de Status (CSV-específico)
**Ubicación Legacy:** `brokers_bybit.py:301` (CSV only)
```python
or n['filled type'].upper() not in ['TRADE','LIQUIDATION','FILLED']
```

**Estado Nuevo:** API usa execType (ya filtrado)

**Conclusión:** NO APLICABLE - Campo CSV-específico

---

## Diferencias No Críticas (No Requieren Implementación)

| Funcionalidad | Legacy | Nuevo | Razón |
|---------------|--------|-------|-------|
| Soporte Multi-CSV | 3 parsers | Solo JSON API | Arquitectura cambió a API-only |
| Deduplicación in-memory | Lista execId | DB-backed p02 | Mejor separación de responsabilidades |
| Microservice dual path | Local/Lambda | Unificado | Simplificación arquitectural |
| Rate limit manual | Header-based backoff | Manejado upstream | Separación de responsabilidades |
| Receive window adjust | Retry con increment | No necesario | API client maneja |
| Usuarios especiales | Hardcoded IDs | No necesario | Workaround legacy |

---

## Plan de Implementación por Fases

### **FASE 1: Validaciones Críticas de Integridad** (Prioridad ALTA - 1-2 días)

**Objetivo:** Prevenir datos inválidos que causen errores downstream

**Tareas:**
1. ✅ Validación de precio cero/negativo
2. ✅ Validación de campos requeridos (defensive)

**Validaciones:**
- ⭐ 1. Validación de Precio
- 🔍 2. Validación de Campos Requeridos

**Archivos a Modificar:**
- `brokers/bybit/bybit.py` (parse_json_content y normalize)

**Métricas de Éxito:**
- Sin precios cero/negativos en output
- Campos requeridos siempre presentes
- Warnings logged para data inválida
- Match rate ≥ baseline (sin regresión)

**Complejidad:** BAJA - Validaciones simples
**Riesgo:** BAJO - Solo filtra datos inválidos
**Estimado:** 1-2 días

---

### **FASE 2: Validaciones de Calidad de Datos** (Prioridad MEDIA - 2-3 días)

**Objetivo:** Mejorar precisión de fees

**Tareas:**
1. ✅ Conversión de fees basada en símbolo (Fase 1 - sin API externa)

**Validaciones:**
- 💰 3. Conversión de Fees por Moneda (Fase 1)

**Archivos a Modificar:**
- `brokers/bybit/bybit.py` (normalize, sección fees)

**Métricas de Éxito:**
- Fees convertidos para símbolos USD
- Fees en USDT para símbolos USDT/PERP
- Match rate mejorado (+1-3%)

**Complejidad:** BAJA-MEDIA - Lógica basada en strings
**Riesgo:** BAJO-MEDIO - Cambio en valores de fees
**Estimado:** 2-3 días

---

### **FASE 3: Asset-Específico y Avanzado** (Prioridad CONDICIONAL - 3-5 días)

**Objetivo:** Soporte para asset classes específicos

**Tareas:**
1. ⚠️ Investigar: ¿Usuarios operan Bybit options?
2. ⚠️ Investigar: ¿Usuarios operan contratos inverse (BTCUSD)?
3. ⚠️ Si SÍ options: Implementar detección crypto options
4. ⚠️ Si SÍ inverse: Implementar pip value calculation
5. ⚠️ Si SÍ inverse: Implementar fee conversion Fase 2 (con API externa)

**Validaciones:**
- 🎯 5. Detección de Crypto Options (condicional)
- 🔄 4. Cálculo de Pip Value (condicional)
- 💰 3. Conversión de Fees Fase 2 (condicional)

**Archivos a Modificar:**
- `brokers/bybit/bybit.py` (parse + normalize)
- Potencialmente: nuevo módulo de pricing API

**Queries de Decisión:**
```sql
-- Verificar options trading
SELECT COUNT(*)
FROM trades_mirror
WHERE broker = 'bybit'
  AND symbol LIKE '%-%-%-%'
  AND created_at > NOW() - INTERVAL '3 months';

-- Verificar inverse contracts
SELECT COUNT(*)
FROM trades_mirror
WHERE broker = 'bybit'
  AND symbol LIKE '%USD'
  AND symbol NOT LIKE '%USDT%'
  AND created_at > NOW() - INTERVAL '3 months';
```

**Métricas de Éxito (si implementado):**
- Options parseadas correctamente (strike, expire, type)
- Pip value correcto para contratos inverse
- Fees convertidos para todas las monedas
- Match rate ≥95%

**Complejidad:** MEDIA-ALTA
**Riesgo:** MEDIO - Dependencias externas si se necesita API
**Estimado:** 3-5 días (incluye investigación)

---

### **FASE 4: Testing y Verificación** (Continuo)

**Objetivo:** Validar que cambios no introducen regresiones

**Tareas:**
1. ✅ Tests unitarios por validación
2. ✅ Tests de integración end-to-end
3. ✅ Validación contra datos legacy
4. ✅ Verificación de match rate

**Tests Requeridos:**

```python
# Fase 1
def test_price_validation_rejects_zero():
    """Verifica rechazo de precio cero"""
def test_price_validation_rejects_negative():
    """Verifica rechazo de precio negativo"""
def test_field_validation_missing_symbol():
    """Verifica rechazo de symbol vacío"""
def test_field_validation_invalid_side():
    """Verifica rechazo de side inválido"""

# Fase 2
def test_fee_conversion_usdt_symbols():
    """Verifica fees sin conversión para USDT"""
def test_fee_conversion_usd_symbols():
    """Verifica conversión fees para USD"""

# Fase 3 (condicional)
def test_option_symbol_parsing():
    """Verifica parsing de símbolos option"""
def test_pip_value_inverse_contracts():
    """Verifica pip value para BTCUSD"""
```

---

## Matriz de Priorización

| # | Validación | Criticidad | Complejidad | Fase | Dependencias | Días |
|---|------------|-----------|-------------|------|--------------|------|
| 1 | Validación Precio | ⭐⭐⭐ ALTA | BAJA | 1 | Ninguna | 0.5 |
| 2 | Campos Requeridos | 🟡 MEDIA | BAJA | 1 | Ninguna | 0.5 |
| 3 | Fee Conversion Fase 1 | 🟡 MEDIA | BAJA-MEDIA | 2 | Ninguna | 2 |
| 4 | Pip Value Calc | 🔵 BAJA | MEDIA | 3 | Decisión: ¿inverse contracts? | 2 |
| 5 | Crypto Options | 🎯 CONDICIONAL | MEDIA | 3 | Decisión: ¿options trading? | 3 |
| 3b | Fee Conversion Fase 2 | 🔵 BAJA | ALTA | 3 | API externa, Fase 3 | 5 |
| 6 | Funding Filter | ✅ DONE | - | - | - | 0 |
| 7 | Duplicate Detection | ✅ DONE | - | - | - | 0 |
| 8 | CSV Support | ❌ NO | - | - | - | 0 |
| 9 | Quantity Calc | ❌ NO (bug) | - | - | - | 0 |
| 10 | Status Validation | ❌ N/A | - | - | - | 0 |

**Total Estimado Fase 1-2:** 3-5 días (crítico)
**Total Estimado Fase 3:** 5-10 días adicionales (condicional)

---

## Archivos Críticos a Modificar

### 1. **`brokers/bybit/bybit.py`** (PRINCIPAL)
**Líneas Actuales:** 333
**Líneas Estimadas Post-Cambios:** ~420-480
**Cambios:**
- Línea 120-180 (parse_json_content): Validación campos requeridos
- Línea 250 (normalize): Validación precio
- Línea 267 (normalize): Conversión fees
- Línea 293 (normalize): Pip value (condicional)
- Nuevo método: parse_bybit_option_symbol() (condicional)

**Responsabilidad:** Todas las validaciones Fase 1, 2, y 3

---

### 2. **`tests/brokers/test_bybit.py`** (MODIFICAR)
**Líneas Actuales:** 559
**Líneas Estimadas Post-Cambios:** ~700-800
**Cambios:**
- Añadir ~10-15 tests nuevos
- Coverage de Fase 1, 2, y 3
- Tests de regresión

**Responsabilidad:** Cobertura completa de tests

---

### 3. **Archivos Legacy (REFERENCIA SOLO)**
- `old_code_from_legacy/bybit_export.py` (941 líneas)
- `old_code_from_legacy/brokers_bybit.py` (643 líneas)

**Uso:** Referencia para entender intención original de validaciones

---

### 4. **`p02_deduplicate/deduplication.py`** (VERIFICAR SOLO)
**Acción:** Confirmar que maneja execution_id correctamente
**Sin cambios esperados**

---

### 5. **`p04_calculate/calculation.py`** (VERIFICAR SOLO)
**Acción:** Entender uso de pip_value y multiplier
**Sin cambios esperados**

---

## Decisiones Pendientes y Preguntas para el Usuario

### 🟡 DECISIÓN MEDIA 1: Crypto Options Trading
**Pregunta:** ¿Los usuarios de Bybit operan crypto options?

**Impacto:**
- Si SÍ: Implementar validación #5 (detección de options)
- Si NO: OMITIR validación #5

**Acción Sugerida:** Query para verificar
```sql
SELECT
    symbol,
    COUNT(*) as executions
FROM trades_mirror
WHERE broker = 'bybit'
  AND symbol LIKE '%-%-%-%'  -- 4-part format
  AND created_at > NOW() - INTERVAL '6 months'
GROUP BY symbol
ORDER BY executions DESC
LIMIT 20;
```

---

### 🟡 DECISIÓN MEDIA 2: Contratos Inverse
**Pregunta:** ¿Los usuarios operan contratos inverse (BTCUSD, ETHUSD)?

**Impacto:**
- Si SÍ: Implementar validación #4 (pip value) y #3 Fase 2 (fee conversion API)
- Si NO: OMITIR, usar pip_value=1 para todos

**Acción Sugerida:** Query para verificar
```sql
SELECT
    symbol,
    COUNT(*) as executions
FROM trades_mirror
WHERE broker = 'bybit'
  AND (symbol LIKE '%USD' AND symbol NOT LIKE '%USDT%')
  AND created_at > NOW() - INTERVAL '6 months'
GROUP BY symbol
ORDER BY executions DESC
LIMIT 20;
```

---

### 🟢 DECISIÓN BAJA 3: Fee Currency Distribution
**Pregunta:** ¿Qué monedas de fees son comunes?

**Impacto:**
- Determina si Fase 2 de fee conversion es necesaria
- Si >95% son USDT: Fase 1 es suficiente
- Si hay BTC/ETH significativo: Considerar Fase 2

**Acción Sugerida:** Query para verificar
```sql
-- Requiere campo feeCurrency en tabla
-- O análisis de raw data
```

---

## Riesgos y Mitigación

### Riesgo 1: Breaking Hash Compatibility
**Severidad:** ALTA
**Probabilidad:** BAJA

**Descripción:** Validaciones cambian datos antes de hash computation

**Mitigación:**
- ✅ Filtrar ANTES de hash computation
- ✅ No modificar campos usados en hash
- ✅ Tests de regresión con hashes legacy conocidos
- ✅ Hash computation actual tiene 100% match rate

---

### Riesgo 2: Over-Filtering Valid Data
**Severidad:** MEDIA
**Probabilidad:** BAJA

**Descripción:** Validaciones muy estrictas rechazan datos válidos

**Mitigación:**
- Log warnings en vez de silent filter
- Monitorear tasa de rechazo en producción
- Feature flag para deshabilitar si necesario
- Threshold conservador (ej: price > 0, no price > 0.001)

---

### Riesgo 3: Performance Degradation
**Severidad:** BAJA
**Probabilidad:** BAJA

**Descripción:** Validaciones adicionales afectan performance

**Mitigación:**
- Usar operaciones vectorizadas Polars
- Evitar loops Python
- Profiling antes/después
- Target: <5ms overhead per execution

---

## Estrategia de Verificación End-to-End

### Paso 1: Test Unitario
```bash
pytest tests/brokers/test_bybit.py::test_price_validation_rejects_zero -v
pytest tests/brokers/test_bybit.py::test_field_validation_missing_symbol -v
```

### Paso 2: Test de Integración
```bash
pytest tests/brokers/test_bybit.py::test_end_to_end_with_validations -v
```

### Paso 3: Validación contra Legacy (si existe script QA)
```bash
python qa/validate_broker.py \
    --broker-id 236 \
    --user-id 99999 \
    --legacy-user-id 49186
```

**Output Esperado:**
```
Broker: Bybit (236)
Match Rate Summary:
  - execution_id: 100.0%
  - symbol: 100.0%
  - side: 100.0%
  - price: 100.0%  (post-validation)
  - quantity: 100.0%
  - fees: 98.5%  (+3.2% vs baseline con fee conversion)
  - file_row: 100.0%  (hash compatibility maintained)

Overall Match: 99.5% ✅
Critical Fields: 100.0% ✅
```

### Paso 4: Smoke Test en Staging
```bash
# Procesar batch real de Bybit
# Verificar output en DB
```

---

## Métricas de Éxito del Proyecto

### Criterios de Aceptación

#### Fase 1 (1-2 días)
- ✅ Validación precio implementada
- ✅ Validación campos implementada
- ✅ Tests unitarios passing
- ✅ Match rate ≥ baseline
- ✅ Sin precios cero/negativos en output
- ✅ Warnings logged apropiadamente

#### Fase 2 (2-3 días)
- ✅ Fee conversion Fase 1 implementada
- ✅ Fees convertidos para USD symbols
- ✅ Tests de fee conversion passing
- ✅ Match rate + 1-3%

#### Fase 3 (Condicional, 3-5 días)
- ✅ Decisiones tomadas (options, inverse)
- ✅ Validaciones condicionales implementadas (si aplica)
- ✅ Tests passing
- ✅ Match rate ≥95% (si aplica)

### Overall Success
- ✅ Sin pérdida de datos vs legacy
- ✅ Sin regresión de performance
- ✅ Hash compatibility mantenida (100%)
- ✅ Arquitectura limpia preservada
- ✅ Documentación completa

---

## Resumen para Guardar en new_changes_bybit/

Al finalizar, se deberá crear el directorio `new_changes_bybit/` con:

### 1. **README.md**
- Resumen ejecutivo de cambios
- Validaciones agregadas (lista priorizada)
- Impacto de cada validación
- Métricas antes/después

### 2. **PLAN_ANALISIS_VALIDACIONES_BYBIT.md**
- Análisis detallado legacy vs nuevo
- Cada validación con ubicación en código
- Ejemplos de código legacy y nuevo
- Decisiones de diseño

### 3. **CAMBIOS_IMPLEMENTADOS.md**
- Log de cambios por archivo
- Diff summary
- Tests agregados
- Match rates

### 4. **EJEMPLOS_CAMBIOS_CODIGO.md**
- Ejemplos antes/después por validación
- Código propuesto concreto
- Justificaciones

### 5. **brokers/bybit/** (código)
- `bybit.py.original` (referencia)
- `detector.py` (sin cambios)
- Tests: `test_bybit.py.original`

### 6. **Legacy code** (referencia)
- `bybit_export.py`
- `brokers_bybit.py`

---

## Conclusión

Este plan identifica **10 categorías de validaciones**, de las cuales:
- **2 son CRÍTICAS (Fase 1)** - Implementación inmediata (1-2 días)
- **1 es MEDIA (Fase 2)** - Mejora de calidad (2-3 días)
- **3 son CONDICIONALES (Fase 3)** - Requieren decisiones (3-5 días si aplica)
- **4 son YA MANEJADAS o NO NECESARIAS** - Sin acción

**Hallazgo Clave:** La nueva implementación es **intencionalmente superior**:
- 76% menos código (1,584→384 líneas)
- Arquitectura más limpia (separación de responsabilidades)
- 100% hash compatibility
- Solo necesita 2-3 validaciones críticas adicionales

**RECOMENDACIÓN FINAL:**
1. **Proceder con Fase 1 inmediatamente** (1-2 días)
2. **Implementar Fase 2** (2-3 días)
3. **Responder decisiones antes de Fase 3**
4. **NO sobre-ingenierizar** - Preservar simplicidad arquitectural
