# Ejemplos de Cambios de Código - Bybit

Este documento muestra ejemplos concretos de los cambios propuestos con código real antes/después.

---

## FASE 1: Validaciones Críticas

### Cambio 1: Validación de Precio

**Archivo:** `brokers/bybit/bybit.py`
**Línea:** ~250
**Criticidad:** 🔴 ALTA

#### Código Actual (ANTES)
```python
# Línea 246-252
.with_columns([
    pl.col("execId").alias("execution_id"),
    pl.col("symbol").str.to_uppercase().alias("symbol"),
    pl.col("side").str.to_uppercase().alias("side"),
    pl.col("execQty").cast(pl.Float64).alias("quantity"),
    pl.col("execPrice").cast(pl.Float64).alias("price"),
    pl.col("execFee").abs().alias("fees"),
])
```

#### Código Propuesto (DESPUÉS)
```python
# Filtrar precios inválidos ANTES de procesamiento
.filter(
    pl.col("execPrice") > 0  # Rechazar precio cero o negativo
)
# Procesamiento normal con redondeo
.with_columns([
    pl.col("execId").alias("execution_id"),
    pl.col("symbol").str.to_uppercase().alias("symbol"),
    pl.col("side").str.to_uppercase().alias("side"),
    pl.col("execQty").cast(pl.Float64).alias("quantity"),
    pl.col("execPrice").cast(pl.Float64).round(6).alias("price"),  # Redondear a 6 decimales
    pl.col("execFee").abs().alias("fees"),
])
```

#### Justificación
- **Legacy:** `bybit_export.py:792-796, 808-811`
- **Impacto:** Previene división por cero en cálculos downstream
- **Precisión:** Consistencia en 6 decimales para todos los precios

#### Tests Requeridos
```python
def test_price_validation_rejects_zero():
    """Verifica que precios cero son rechazados"""
    df = pl.DataFrame({
        "execId": ["1", "2", "3"],
        "symbol": ["BTCUSDT", "ETHUSDT", "ADAUSDT"],
        "side": ["BUY", "BUY", "BUY"],
        "execQty": [1.0, 1.0, 1.0],
        "execPrice": [50000.0, 0.0, 1.5],  # Segundo tiene precio cero
        "execTime": ["1234567890000", "1234567891000", "1234567892000"],
        "execFee": [-0.05, -0.01, -0.001],
        "execType": ["Trade", "Trade", "Trade"],
        "category": ["linear", "linear", "linear"],
    })

    result = BybitInterpreter().normalize(df.lazy(), account_id=1).collect()

    # Solo 2 registros deben pasar (precio cero filtrado)
    assert len(result) == 2
    assert "2" not in result["execution_id"].to_list()  # execId "2" filtrado

    # Precios válidos redondeados a 6 decimales
    assert result.filter(pl.col("execution_id") == "1")["price"][0] == 50000.0
    assert result.filter(pl.col("execution_id") == "3")["price"][0] == 1.5


def test_price_validation_rejects_negative():
    """Verifica que precios negativos son rechazados"""
    df = pl.DataFrame({
        "execId": ["1"],
        "symbol": ["BTCUSDT"],
        "side": ["BUY"],
        "execQty": [1.0],
        "execPrice": [-50000.0],  # Precio negativo
        "execTime": ["1234567890000"],
        "execFee": [-0.05],
        "execType": ["Trade"],
        "category": ["linear"],
    })

    result = BybitInterpreter().normalize(df.lazy(), account_id=1).collect()

    # Ningún registro debe pasar
    assert len(result) == 0


def test_price_rounding_precision():
    """Verifica redondeo a 6 decimales"""
    df = pl.DataFrame({
        "execId": ["1"],
        "symbol": ["BTCUSDT"],
        "side": ["BUY"],
        "execQty": [1.0],
        "execPrice": [50000.123456789],  # 9 decimales
        "execTime": ["1234567890000"],
        "execFee": [-0.05],
        "execType": ["Trade"],
        "category": ["linear"],
    })

    result = BybitInterpreter().normalize(df.lazy(), account_id=1).collect()

    # Precio redondeado a 6 decimales
    assert result["price"][0] == 50000.123457  # Redondeado desde 50000.123456789
```

---

### Cambio 2: Validación de Campos Requeridos

**Archivo:** `brokers/bybit/bybit.py`
**Línea:** ~120-180 (dentro de parse_json_content loop)
**Criticidad:** 🟡 MEDIA

#### Código Actual (ANTES)
```python
# Línea 120-140 (parse_json_content)
for execution in executions:
    exec_id = execution.get("execId", "")
    symbol = execution.get("symbol", "")
    side = execution.get("side", "")
    exec_qty = execution.get("execQty", 0)
    exec_price = execution.get("execPrice", 0)
    exec_time = execution.get("execTime", "")
    exec_type = execution.get("execType", "")

    # Solo filtrado de execType
    if exec_type != "Trade":
        continue

    # Continúa procesamiento sin validación adicional...
```

#### Código Propuesto (DESPUÉS)
```python
# Línea 120-140 (parse_json_content)
for execution in executions:
    exec_id = execution.get("execId", "")
    symbol = execution.get("symbol", "")
    side = execution.get("side", "")
    exec_qty = execution.get("execQty", 0)
    exec_price = execution.get("execPrice", 0)
    exec_time = execution.get("execTime", "")
    exec_type = execution.get("execType", "")

    # Validación: Filtrado de execType
    if exec_type != "Trade":
        continue

    # NUEVO: Validación de campos requeridos
    if not exec_id or not symbol or not side or not exec_time:
        logger.warning(
            f"Skipping execution with missing required fields: "
            f"execId={exec_id}, symbol={symbol}, side={side}, execTime={exec_time}"
        )
        continue

    # NUEVO: Validación de valores numéricos
    if exec_price <= 0 or exec_qty <= 0:
        logger.warning(
            f"Skipping execution with invalid price/qty: "
            f"execId={exec_id}, price={exec_price}, qty={exec_qty}"
        )
        continue

    # NUEVO: Validación de side
    if side.upper() not in ("BUY", "SELL"):
        logger.warning(
            f"Skipping execution with invalid side: "
            f"execId={exec_id}, side={side}"
        )
        continue

    # Continúa procesamiento normal...
```

#### Justificación
- **Legacy:** `bybit_export.py:797-798` + `brokers_bybit.py:297-303`
- **Impacto:** Previene datos incompletos en downstream processing
- **Defensive programming:** API debería enviar datos válidos, pero validamos por seguridad

#### Tests Requeridos
```python
def test_field_validation_missing_symbol():
    """Verifica que executions sin symbol son rechazadas"""
    content = {
        "result": {
            "list": [
                {
                    "execId": "test-1",
                    "symbol": "",  # Symbol vacío
                    "side": "Buy",
                    "execQty": "1.0",
                    "execPrice": "50000.0",
                    "execTime": "1234567890000",
                    "execType": "Trade",
                }
            ]
        }
    }

    result = BybitInterpreter().parse_json_content(content)

    # Ningún registro debe pasar
    assert len(result) == 0


def test_field_validation_invalid_side():
    """Verifica que sides inválidos son rechazados"""
    content = {
        "result": {
            "list": [
                {
                    "execId": "test-1",
                    "symbol": "BTCUSDT",
                    "side": "INVALID",  # Side inválido
                    "execQty": "1.0",
                    "execPrice": "50000.0",
                    "execTime": "1234567890000",
                    "execType": "Trade",
                }
            ]
        }
    }

    result = BybitInterpreter().parse_json_content(content)

    # Ningún registro debe pasar
    assert len(result) == 0


def test_field_validation_zero_quantity():
    """Verifica que quantity cero es rechazada"""
    content = {
        "result": {
            "list": [
                {
                    "execId": "test-1",
                    "symbol": "BTCUSDT",
                    "side": "Buy",
                    "execQty": "0",  # Quantity cero
                    "execPrice": "50000.0",
                    "execTime": "1234567890000",
                    "execType": "Trade",
                }
            ]
        }
    }

    result = BybitInterpreter().parse_json_content(content)

    # Ningún registro debe pasar
    assert len(result) == 0


def test_field_validation_allows_valid_data():
    """Verifica que datos válidos pasan la validación"""
    content = {
        "result": {
            "list": [
                {
                    "execId": "test-1",
                    "symbol": "BTCUSDT",
                    "side": "Buy",
                    "execQty": "1.5",
                    "execPrice": "50000.0",
                    "execTime": "1234567890000",
                    "execType": "Trade",
                },
                {
                    "execId": "test-2",
                    "symbol": "ETHUSDT",
                    "side": "Sell",
                    "execQty": "10.0",
                    "execPrice": "3000.0",
                    "execTime": "1234567891000",
                    "execType": "Trade",
                }
            ]
        }
    }

    result = BybitInterpreter().parse_json_content(content)

    # Ambos registros deben pasar
    assert len(result) == 2
```

---

## FASE 2: Validaciones de Calidad

### Cambio 3: Conversión de Fees por Moneda

**Archivo:** `brokers/bybit/bybit.py`
**Línea:** ~267 (donde se calcula fees)
**Criticidad:** 🟡 MEDIA

#### Código Actual (ANTES)
```python
# Línea 265-272
.with_columns([
    pl.col("execId").alias("execution_id"),
    pl.col("symbol").str.to_uppercase().alias("symbol"),
    pl.col("side").str.to_uppercase().alias("side"),
    pl.col("execQty").cast(pl.Float64).alias("quantity"),
    pl.col("execPrice").cast(pl.Float64).alias("price"),
    pl.col("execFee").abs().alias("fees"),  # Sin conversión
])
```

#### Código Propuesto (DESPUÉS)
```python
# Línea 265-272
.with_columns([
    pl.col("execId").alias("execution_id"),
    pl.col("symbol").str.to_uppercase().alias("symbol"),
    pl.col("side").str.to_uppercase().alias("side"),
    pl.col("execQty").cast(pl.Float64).alias("quantity"),
    pl.col("execPrice").cast(pl.Float64).alias("price"),

    # NUEVO: Conversión de fees basada en símbolo
    pl.when(pl.col("symbol").str.contains("USDT|PERP|USDC"))
      .then(pl.col("execFee").abs())  # Ya en USDT, no conversión
    .when(pl.col("symbol").str.ends_with("USD"))
      .then(pl.col("execFee").abs() * pl.col("execPrice"))  # USD: convertir multiplicando por precio
    .otherwise(pl.col("execFee").abs())  # Default: sin conversión
    .alias("fees"),
])
```

#### Justificación
- **Legacy:** `bybit_export.py:903-925, 885-892`
- **Impacto:** Fees correctos para contratos USD vs USDT
- **Regla:** USDT-margined (fee ya en USDT), USD-margined (convertir fee × precio)

#### Ejemplo de Datos
```json
// Caso 1: USDT-margined (sin conversión)
{
  "symbol": "BTCUSDT",
  "execPrice": 50000.0,
  "execFee": -0.05,
  "fees": 0.05  // <- Sin conversión (ya en USDT)
}

// Caso 2: USD-margined (con conversión)
{
  "symbol": "BTCUSD",
  "execPrice": 50000.0,
  "execFee": -0.00000100,  // Fee en BTC
  "fees": 0.05  // <- Convertido: 0.00000100 × 50000 = 0.05 USDT
}

// Caso 3: PERP (sin conversión)
{
  "symbol": "ETHPERP",
  "execPrice": 3000.0,
  "execFee": -0.003,
  "fees": 0.003  // <- Sin conversión (ya en USDT)
}
```

#### Tests Requeridos
```python
def test_fee_conversion_usdt_symbols():
    """Verifica que fees para USDT symbols no se convierten"""
    df = pl.DataFrame({
        "execId": ["1"],
        "symbol": ["BTCUSDT"],
        "side": ["BUY"],
        "execQty": [1.0],
        "execPrice": [50000.0],
        "execTime": ["1234567890000"],
        "execFee": [-0.05],  # Fee ya en USDT
        "execType": ["Trade"],
        "category": ["linear"],
    })

    result = BybitInterpreter().normalize(df.lazy(), account_id=1).collect()

    # Fee sin conversión
    assert result["fees"][0] == 0.05


def test_fee_conversion_usd_symbols():
    """Verifica conversión de fees para USD symbols"""
    df = pl.DataFrame({
        "execId": ["1"],
        "symbol": ["BTCUSD"],
        "side": ["BUY"],
        "execQty": [1.0],
        "execPrice": [50000.0],
        "execTime": ["1234567890000"],
        "execFee": [-0.00000100],  # Fee en BTC
        "execType": ["Trade"],
        "category": ["inverse"],
    })

    result = BybitInterpreter().normalize(df.lazy(), account_id=1).collect()

    # Fee convertido: 0.00000100 × 50000 = 0.05
    assert result["fees"][0] == 0.05


def test_fee_conversion_perp_symbols():
    """Verifica que fees para PERP symbols no se convierten"""
    df = pl.DataFrame({
        "execId": ["1"],
        "symbol": ["ETHPERP"],
        "side": ["SELL"],
        "execQty": [10.0],
        "execPrice": [3000.0],
        "execTime": ["1234567890000"],
        "execFee": [-0.003],  # Fee ya en USDT
        "execType": ["Trade"],
        "category": ["linear"],
    })

    result = BybitInterpreter().normalize(df.lazy(), account_id=1).collect()

    # Fee sin conversión
    assert result["fees"][0] == 0.003
```

---

## FASE 3: Asset-Específico (Condicional)

### Cambio 4: Cálculo de Pip Value para Contratos Inverse

**Archivo:** `brokers/bybit/bybit.py`
**Línea:** ~293 (donde se define pip_value)
**Criticidad:** 🔵 BAJA (Solo si usuarios operan inverse)

**⚠️ NOTA:** Este cambio es CONDICIONAL. Solo implementar si la investigación confirma que usuarios operan contratos inverse (BTCUSD, ETHUSD).

#### Código Actual (ANTES)
```python
# Línea 293-295
.with_columns([
    pl.lit(1.0).alias("pip_value")  # Hardcoded a 1.0
])
```

#### Código Propuesto (DESPUÉS) - Si se implementa
```python
# Línea 293-300
.with_columns([
    # Cálculo condicional de pip_value
    pl.when(pl.col("symbol").str.contains("USDT|PERP|USDC"))
      .then(pl.lit(1.0))  # USDT-margined: pip_value = 1
    .when(pl.col("symbol").str.ends_with("USD") & (pl.col("execValue") > 0))
      .then(
          # Inverse contracts: calcular desde execValue
          (pl.col("execValue") / (pl.col("execQty") * pl.col("execPrice"))) * pl.col("execPrice")
      )
    .otherwise(pl.lit(1.0))  # Default
    .alias("pip_value")
])
```

#### Justificación
- **Legacy:** `bybit_export.py:862-892`
- **Impacto:** Solo afecta contratos inverse (BTCUSD, ETHUSD)
- **Mayoría:** USDT-margined contracts (pip_value=1) funcionan correctamente

#### Ejemplo de Cálculo
```python
# Contrato USDT-margined (mayoría)
symbol = "BTCUSDT"
pip_value = 1.0  # ✅ No requiere cálculo

# Contrato Inverse (raro)
symbol = "BTCUSD"
execValue = 1000.0   # USD value
execQty = 1.0        # Contracts
execPrice = 50000.0  # USD per BTC
pip_value = (1000.0 / (1.0 * 50000.0)) * 50000.0 = 1.0  # Simplifica a 1 también
```

**NOTA:** En la práctica, incluso para inverse contracts, el valor suele ser ~1.0. La complejidad del cálculo puede no justificar la implementación.

---

### Cambio 5: Detección de Crypto Options

**Archivo:** `brokers/bybit/bybit.py`
**Línea:** Nuevo método + integración en parse_json_content
**Criticidad:** 🎯 CONDICIONAL

**⚠️ NOTA:** Este cambio es CONDICIONAL. Solo implementar si la investigación confirma que usuarios operan Bybit crypto options.

#### Código Propuesto (NUEVO MÉTODO) - Si se implementa

```python
def parse_bybit_option_symbol(symbol: str) -> dict:
    """
    Parse Bybit crypto option symbol format.

    Format: BTC-31MAY25-80000-C
            ^   ^       ^      ^
            |   |       |      └─ Type: C (Call) or P (Put)
            |   |       └──────── Strike price
            |   └──────────────── Expiry date (DDMMMYY)
            └──────────────────── Base asset

    Args:
        symbol: Option symbol string

    Returns:
        dict with is_option, base_symbol, expire_date, strike, option_type

    Examples:
        >>> parse_bybit_option_symbol("BTC-31MAY25-80000-C")
        {
            "is_option": True,
            "base_symbol": "#BTCUSDT",
            "expire_date": "2025-05-31",
            "strike": 80000.0,
            "option_type": "CALL"
        }

        >>> parse_bybit_option_symbol("BTCUSDT")
        {"is_option": False}
    """
    parts = symbol.split('-')

    # Not an option if doesn't have 4 parts
    if len(parts) != 4:
        return {"is_option": False}

    base, expiry_str, strike_str, type_code = parts

    try:
        # Parse expiry: "31MAY25" → datetime
        expiry_dt = datetime.strptime(expiry_str, "%d%b%y")
        expiry_iso = expiry_dt.strftime("%Y-%m-%d")

        # Parse strike: "80000" → 80000.0
        strike = float(strike_str)

        # Parse type: "C" → "CALL", "P" → "PUT"
        option_type = "CALL" if type_code.upper() == "C" else "PUT"

        return {
            "is_option": True,
            "base_symbol": f"#{base}USDT",  # Normalize to #BTCUSDT format
            "expire_date": expiry_iso,
            "strike": strike,
            "option_type": option_type,
        }
    except (ValueError, IndexError) as e:
        logger.warning(f"Failed to parse option symbol {symbol}: {e}")
        return {"is_option": False}
```

#### Integración en parse_json_content

```python
# Dentro del loop de parse_json_content (línea ~140)
for execution in executions:
    exec_id = execution.get("execId", "")
    symbol = execution.get("symbol", "")
    category = execution.get("category", "")

    # ... validaciones existentes ...

    # NUEVO: Detectar y parsear options
    asset_type = "crypto"  # Default
    expire_date = None
    strike = None
    option_type = None

    if category == "option":
        option_info = parse_bybit_option_symbol(symbol)

        if option_info["is_option"]:
            asset_type = "crypto option"
            symbol = option_info["base_symbol"]  # BTC-31MAY25-80000-C → #BTCUSDT
            expire_date = option_info["expire_date"]
            strike = option_info["strike"]
            option_type = option_info["option_type"]

    # Añadir a records con campos adicionales
    records.append({
        # ... campos existentes ...
        "symbol": symbol,
        "type": asset_type,
        "expire": expire_date,
        "strike": strike,
        "option": option_type,
    })
```

#### Tests Requeridos

```python
def test_option_symbol_parsing_call():
    """Verifica parsing de Call option"""
    result = parse_bybit_option_symbol("BTC-31MAY25-80000-C")

    assert result["is_option"] is True
    assert result["base_symbol"] == "#BTCUSDT"
    assert result["expire_date"] == "2025-05-31"
    assert result["strike"] == 80000.0
    assert result["option_type"] == "CALL"


def test_option_symbol_parsing_put():
    """Verifica parsing de Put option"""
    result = parse_bybit_option_symbol("ETH-15JUN25-3000-P")

    assert result["is_option"] is True
    assert result["base_symbol"] == "#ETHUSDT"
    assert result["expire_date"] == "2025-06-15"
    assert result["strike"] == 3000.0
    assert result["option_type"] == "PUT"


def test_option_symbol_parsing_non_option():
    """Verifica que símbolos regulares no son parseados como option"""
    result = parse_bybit_option_symbol("BTCUSDT")

    assert result["is_option"] is False


def test_option_normalization_full_flow():
    """Test end-to-end de normalización con option"""
    content = {
        "result": {
            "list": [
                {
                    "execId": "test-opt-1",
                    "symbol": "BTC-31MAY25-80000-C",
                    "category": "option",
                    "side": "Buy",
                    "execQty": "1.0",
                    "execPrice": "5000.0",
                    "execTime": "1234567890000",
                    "execType": "Trade",
                    "execFee": "-0.05",
                }
            ]
        }
    }

    result = BybitInterpreter().normalize_from_json_content(content, account_id=1)

    assert len(result) == 1
    assert result["symbol"][0] == "#BTCUSDT"
    assert result["type"][0] == "crypto option"
    assert result["expire"][0] == "2025-05-31"
    assert result["strike"][0] == 80000.0
    assert result["option"][0] == "CALL"
```

---

## Resumen de Cambios por Archivo

### brokers/bybit/bybit.py
- **Línea ~120-180:** Validación campos requeridos (Cambio 2)
- **Línea ~250:** Validación precio (Cambio 1)
- **Línea ~267:** Conversión fees (Cambio 3)
- **Línea ~293:** Pip value calc (Cambio 4 - CONDICIONAL)
- **Nuevo método:** parse_bybit_option_symbol() (Cambio 5 - CONDICIONAL)

### tests/brokers/test_bybit.py
- Añadir ~15 funciones de test nuevas
- Coverage de todos los cambios (Fase 1, 2, y 3 condicional)

---

## Orden de Implementación Recomendado

1. ✅ **Cambio 1:** Validación precio + tests (Fase 1)
2. ✅ **Cambio 2:** Validación campos + tests (Fase 1)
3. ✅ **Validar Fase 1:** Ejecutar todos los tests, verificar match rate
4. ✅ **Cambio 3:** Conversión fees + tests (Fase 2)
5. ✅ **Validar Fase 2:** Match rate final
6. ⚠️ **Decisión:** Investigar usage de options e inverse
7. ⚠️ **Si aplica - Cambio 4:** Pip value + tests (Fase 3)
8. ⚠️ **Si aplica - Cambio 5:** Options parsing + tests (Fase 3)
9. ✅ **Validar Fase 3:** Match rate ≥95% (si implementado)

---

**Nota:** Todos los cambios mantienen compatibilidad con la arquitectura existente y usan operaciones vectorizadas de Polars para performance óptimo. Los cambios condicionales (Fase 3) solo deben implementarse si la investigación confirma su necesidad.
