"""
Tests for Bitstamp interpreter.

Run with: pytest tests/brokers/test_bitstamp.py -v
"""

import pytest
import polars as pl
import json
import hashlib

from pipeline.p01_normalize.brokers.bitstamp.bitstamp import BitstampInterpreter


class TestBitstampInterpreter:
    """Tests for BitstampInterpreter class."""

    @pytest.fixture
    def interpreter(self):
        """Create interpreter instance."""
        return BitstampInterpreter()

    @pytest.fixture
    def sample_buy_transaction(self):
        """Create sample Bitstamp BUY transaction (BTC/USD)."""
        return {
            "id": 153132639,
            "datetime": "2021-02-22 12:13:21.318000",
            "type": "2",
            "fee": "5.08375",
            "ltc": "5.00000000",
            "usd": "-1016.75",
            "ltc_usd": 203.35,
            "btc": 0.0,
            "eur": 0.0,
            "order_id": 1331854081888257
        }

    @pytest.fixture
    def sample_sell_transaction(self):
        """Create sample Bitstamp SELL transaction (BTC/USD)."""
        return {
            "id": 321107216,
            "datetime": "2024-02-21 12:07:32.142000",
            "type": "2",
            "fee": "6.08592",
            "btc": "-0.03000000",
            "usd": "1521.48",
            "btc_usd": 50716.0,
            "eur": 0.0,
            "order_id": 1719018608578561
        }

    @pytest.fixture
    def sample_stablecoin_transaction(self):
        """Create sample stablecoin transaction (USDC/USD)."""
        return {
            "id": 157390688,
            "datetime": "2021-03-14 21:07:45.289000",
            "type": "2",
            "fee": "0.20367",
            "usdc": "848.38191",
            "usd": "-848.63642457",
            "usdc_usd": 1.0003,
            "btc": 0.0,
            "eur": 0.0,
            "order_id": 1339068787228672
        }

    @pytest.fixture
    def sample_crypto_crypto_transaction(self):
        """Create sample crypto-to-crypto transaction (ETH/BTC)."""
        return {
            "id": 320341878,
            "datetime": "2024-02-16 09:56:32.497000",
            "type": "2",
            "fee": "0.00006091",
            "eth": "-0.28120000",
            "btc": "0.01522852",
            "eth_btc": 0.05415547,
            "usd": 0.0,
            "eur": 0.0,
            "order_id": 1717216943472641
        }

    @pytest.fixture
    def sample_deposit_transaction(self):
        """Create sample deposit transaction (type=0, should be filtered)."""
        return {
            "id": 123456789,
            "datetime": "2021-02-22 10:00:00.000000",
            "type": "0",
            "fee": "0.0",
            "btc": "1.0",
            "usd": "0.0",
            "btc_usd": 0.0,
            "eur": 0.0,
            "order_id": 0
        }

    @pytest.fixture
    def sample_json_content(self, sample_buy_transaction, sample_sell_transaction):
        """Create sample JSON content (array of transactions)."""
        return json.dumps([sample_buy_transaction, sample_sell_transaction])

    def test_parse_json_content(self, interpreter, sample_json_content):
        """Test JSON parsing produces correct DataFrame."""
        df = interpreter.parse_json_content(sample_json_content)

        assert len(df) == 2
        assert "txn_id" in df.columns
        assert "order_id" in df.columns
        assert "symbol" in df.columns
        assert "side" in df.columns
        assert "quantity" in df.columns
        assert "price" in df.columns

    def test_parse_json_filters_trades_only(self, interpreter, sample_buy_transaction, sample_deposit_transaction):
        """Test that only type=2 (trade) transactions are parsed."""
        content = json.dumps([sample_buy_transaction, sample_deposit_transaction])
        df = interpreter.parse_json_content(content)
        assert len(df) == 1
        assert df["txn_id"][0] == str(sample_buy_transaction["id"])

    def test_detect_buy_side(self, interpreter, sample_buy_transaction):
        """Test BUY side detection (positive crypto amount)."""
        content = json.dumps([sample_buy_transaction])
        df = interpreter.parse_json_content(content)
        assert df["side"][0] == "BUY"

    def test_detect_sell_side(self, interpreter, sample_sell_transaction):
        """Test SELL side detection (negative crypto amount)."""
        content = json.dumps([sample_sell_transaction])
        df = interpreter.parse_json_content(content)
        assert df["side"][0] == "SELL"

    def test_symbol_construction(self, interpreter, sample_buy_transaction):
        """Test symbol is constructed correctly (e.g., LTCUSD)."""
        content = json.dumps([sample_buy_transaction])
        df = interpreter.parse_json_content(content)
        assert df["symbol"][0] == "LTCUSD"

    def test_stablecoin_pair(self, interpreter, sample_stablecoin_transaction):
        """Test stablecoin pair detection (USDC/USD)."""
        content = json.dumps([sample_stablecoin_transaction])
        df = interpreter.parse_json_content(content)
        assert df["symbol"][0] == "USDCUSD"
        assert df["side"][0] == "BUY"

    def test_crypto_crypto_pair(self, interpreter, sample_crypto_crypto_transaction):
        """Test crypto-to-crypto pair detection (ETH/BTC)."""
        content = json.dumps([sample_crypto_crypto_transaction])
        df = interpreter.parse_json_content(content)
        assert df["symbol"][0] == "ETHBTC"
        assert df["side"][0] == "SELL"

    def test_normalize_basic(self, interpreter, sample_json_content):
        """Test basic normalization produces correct schema."""
        df = interpreter.parse_json_content(sample_json_content)
        result = interpreter.normalize(df.lazy(), user_id=999, account_id="test123").collect()

        assert len(result) == 2
        # Check core normalizer columns are present
        core_columns = [
            "user_id", "account_id", "execution_id", "symbol", "side",
            "quantity", "price", "timestamp", "commission", "fees", "swap",
            "currency", "asset", "option_strike", "option_expire",
            "multiplier", "pip_value", "original_file_row", "file_row"
        ]
        for col in core_columns:
            assert col in result.columns, f"Missing required column: {col}"

    def test_normalize_user_id(self, interpreter, sample_json_content):
        """Test user_id is set correctly."""
        df = interpreter.parse_json_content(sample_json_content)
        result = interpreter.normalize(df.lazy(), user_id=12345, account_id="").collect()
        assert all(result["user_id"] == 12345)

    def test_normalize_account_id(self, interpreter, sample_json_content):
        """Test account_id is set from input parameter."""
        df = interpreter.parse_json_content(sample_json_content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="my_account").collect()
        assert all(result["account_id"] == "my_account")

    def test_normalize_quantity(self, interpreter, sample_buy_transaction):
        """Test quantity is parsed correctly (absolute value)."""
        content = json.dumps([sample_buy_transaction])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        assert result["quantity"][0] == pytest.approx(5.0, rel=1e-6)

    def test_normalize_price(self, interpreter, sample_buy_transaction):
        """Test price is parsed correctly."""
        content = json.dumps([sample_buy_transaction])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        assert result["price"][0] == pytest.approx(203.35, rel=1e-6)

    def test_normalize_fees(self, interpreter, sample_buy_transaction):
        """Test fees is 0 (legacy stores 0 for Bitstamp, fee is in raw data but not in output)."""
        content = json.dumps([sample_buy_transaction])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        # Legacy stores 0 for Bitstamp fees - the fee is captured in original_file_row
        assert result["fees"][0] == 0.0

    def test_normalize_timestamp(self, interpreter, sample_buy_transaction):
        """Test timestamp parsing from datetime format."""
        content = json.dumps([sample_buy_transaction])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        ts = result["timestamp"][0]
        assert ts.year == 2021
        assert ts.month == 2
        assert ts.day == 22

    def test_normalize_currency(self, interpreter, sample_buy_transaction):
        """Test currency is extracted correctly."""
        content = json.dumps([sample_buy_transaction])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        assert result["currency"][0] == "USD"

    def test_normalize_asset(self, interpreter, sample_buy_transaction):
        """Test asset is set to crypto."""
        content = json.dumps([sample_buy_transaction])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        assert result["asset"][0] == "crypto"

    def test_execution_id_from_txn_id(self, interpreter, sample_buy_transaction):
        """Test execution_id is set from transaction id."""
        content = json.dumps([sample_buy_transaction])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        assert result["execution_id"][0] == str(sample_buy_transaction["id"])

    def test_original_file_row_valid_json(self, interpreter, sample_json_content):
        """Test original_file_row is valid JSON."""
        df = interpreter.parse_json_content(sample_json_content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        for row in result["original_file_row"]:
            parsed = json.loads(row)
            assert isinstance(parsed, dict)

    def test_original_file_row_contains_id(self, interpreter, sample_buy_transaction):
        """Test original_file_row contains the transaction id."""
        content = json.dumps([sample_buy_transaction])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        parsed = json.loads(result["original_file_row"][0])
        assert parsed["id"] == sample_buy_transaction["id"]


class TestEdgeCases:
    """Tests for edge cases."""

    def test_empty_transactions(self):
        """Test handling of empty transactions array."""
        interpreter = BitstampInterpreter()
        df = interpreter.parse_json_content('[]')
        assert len(df) == 0

    def test_all_non_trade_transactions(self):
        """Test handling when all transactions are non-trades (type != 2)."""
        interpreter = BitstampInterpreter()
        content = json.dumps([
            {
                "id": 123,
                "datetime": "2021-02-22 10:00:00.000000",
                "type": "0",  # Deposit
                "fee": "0.0",
                "btc": "1.0",
                "usd": "0.0",
                "btc_usd": 0.0,
                "order_id": 0
            }
        ])
        df = interpreter.parse_json_content(content)
        assert len(df) == 0

    def test_zero_fee(self):
        """Test transaction with zero fee."""
        interpreter = BitstampInterpreter()
        content = json.dumps([{
            "id": 194491188,
            "datetime": "2021-08-23 14:25:24.006000",
            "type": "2",
            "fee": "0.00000",
            "usdt": "-100.00000",
            "usd": "100.03300",
            "usdt_usd": 1.00033,
            "btc": 0.0,
            "eur": 0.0,
            "order_id": 1396300797333504
        }])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        assert result["fees"][0] == 0.0

    def test_multiple_currency_pairs(self):
        """Test transactions with different currency pairs."""
        interpreter = BitstampInterpreter()
        content = json.dumps([
            {
                "id": 1,
                "datetime": "2021-02-22 12:00:00.000000",
                "type": "2",
                "fee": "1.0",
                "btc": "0.01",
                "usd": "-500.0",
                "btc_usd": 50000.0,
                "eur": 0.0,
                "order_id": 1
            },
            {
                "id": 2,
                "datetime": "2021-02-22 13:00:00.000000",
                "type": "2",
                "fee": "0.5",
                "eth": "1.0",
                "usd": "-3000.0",
                "eth_usd": 3000.0,
                "btc": 0.0,
                "eur": 0.0,
                "order_id": 2
            }
        ])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        assert result["symbol"][0] == "BTCUSD"
        assert result["symbol"][1] == "ETHUSD"


class TestCanHandle:
    """Tests for can_handle method."""

    def test_can_handle_valid_data(self):
        """Test can_handle returns True for valid Bitstamp data."""
        df = pl.DataFrame({
            "id": [1],
            "datetime": ["2021-02-22 12:00:00.000000"],
            "type": ["2"],
            "fee": ["1.0"],
            "order_id": [123]
        })
        assert BitstampInterpreter.can_handle(df, {}) is True

    def test_cannot_handle_missing_columns(self):
        """Test can_handle returns False for missing columns."""
        df = pl.DataFrame({
            "id": [1],
            "datetime": ["2021-02-22 12:00:00.000000"],
        })
        assert BitstampInterpreter.can_handle(df, {}) is False


class TestFileRowHash:
    """Tests for file_row hash formula (legacy compatibility)."""

    @pytest.fixture
    def interpreter(self):
        return BitstampInterpreter()

    def test_file_row_is_valid_md5(self, interpreter):
        """Test file_row is 32-char MD5 hex."""
        txn = {
            "id": 153132639,
            "datetime": "2021-02-22 12:13:21.318000",
            "type": "2",
            "fee": "5.08375",
            "ltc": "5.00000000",
            "usd": "-1016.75",
            "ltc_usd": 203.35,
            "btc": 0.0,
            "eur": 0.0,
            "order_id": 1331854081888257
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()
        file_row = result["file_row"][0]
        # Should be exactly 32 hex chars
        assert len(file_row) == 32
        assert all(c in "0123456789abcdef" for c in file_row)

    def test_file_row_deterministic(self, interpreter):
        """Test same input produces same hash."""
        txn = {
            "id": 153132639,
            "datetime": "2021-02-22 12:13:21.318000",
            "type": "2",
            "fee": "5.08375",
            "ltc": "5.00000000",
            "usd": "-1016.75",
            "ltc_usd": 203.35,
            "btc": 0.0,
            "eur": 0.0,
            "order_id": 1331854081888257
        }
        content = json.dumps([txn])

        df1 = interpreter.parse_json_content(content)
        result1 = interpreter.normalize(df1.lazy(), user_id=1, account_id="").collect()

        df2 = interpreter.parse_json_content(content)
        result2 = interpreter.normalize(df2.lazy(), user_id=1, account_id="").collect()

        assert result1["file_row"][0] == result2["file_row"][0]

    def test_file_row_uses_pre_hash_formula(self, interpreter):
        """Test file_row hash uses legacy pre-hash formula (raw txn + created_at + created_at_formated + symbol)."""
        txn = {
            "id": 999888777,
            "datetime": "2021-02-22 12:13:21.318000",
            "type": "2",
            "fee": "5.0",
            "btc": "1.0",
            "usd": "-50000.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 123
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()

        # Compute expected hash using legacy formula:
        # pre_hash = raw txn + created_at + created_at_formated + symbol
        pre_hash = dict(txn)
        pre_hash["created_at"] = txn["datetime"]
        pre_hash["created_at_formated"] = txn["datetime"]
        pre_hash["symbol"] = "BTCUSD"  # btc + usd
        expected_hash = hashlib.md5(json.dumps(pre_hash).encode('utf-8')).hexdigest()
        assert result["file_row"][0] == expected_hash

    def test_file_row_different_ids_produce_different_hashes(self, interpreter):
        """Test different transaction ids produce different hashes."""
        txn1 = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "btc": "0.01",
            "usd": "-500.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 1
        }
        txn2 = {
            "id": 222,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "btc": "0.01",
            "usd": "-500.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn1, txn2])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="").collect()

        assert result["file_row"][0] != result["file_row"][1]


class TestCriticalValidations:
    """Tests for critical data integrity validations."""

    @pytest.fixture
    def interpreter(self):
        """Create interpreter instance."""
        return BitstampInterpreter()

    def test_status_filled_filter(self, interpreter):
        """Test that only FILLED status transactions are processed."""
        txn_filled = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "status": "FILLED",
            "fee": "1.0",
            "btc": "0.01",
            "usd": "-500.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 1
        }
        txn_cancelled = {
            "id": 222,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "status": "CANCELLED",
            "fee": "1.0",
            "btc": "0.01",
            "usd": "-500.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 2
        }
        content = json.dumps([txn_filled, txn_cancelled])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="test").collect()

        # Only the FILLED transaction should be present
        assert len(result) == 1
        assert result["execution_id"][0] == "111"

    def test_price_zero_rejected(self, interpreter):
        """Test that transactions with price=0 are rejected."""
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "btc": "0.01",
            "usd": "-500.0",
            # Missing btc_usd rate (price=0)
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)

        # Should be rejected during parsing (empty DataFrame)
        assert len(df) == 0

    def test_price_negative_rejected(self, interpreter):
        """Test that transactions with negative price are rejected."""
        # Note: Bitstamp API doesn't send negative prices normally,
        # but this tests the validation
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "btc": "0.01",
            "usd": "-500.0",
            "btc_usd": -50000.0,  # Invalid negative price
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)

        # Should be rejected during parsing (empty DataFrame)
        assert len(df) == 0

    def test_datetime_empty_rejected(self, interpreter):
        """Test that transactions without datetime are rejected."""
        txn = {
            "id": 111,
            "datetime": "",  # Empty datetime
            "type": "2",
            "fee": "1.0",
            "btc": "0.01",
            "usd": "-500.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)

        # Should be rejected during parsing (empty DataFrame)
        assert len(df) == 0

    def test_quantity_zero_rejected(self, interpreter):
        """Test that transactions with quantity=0 are rejected."""
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "btc": "0.0",  # Zero quantity
            "usd": "-500.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)

        # Should be rejected during parsing (no valid trade pair detected)
        assert len(df) == 0

    def test_side_validation(self, interpreter):
        """Test that only BUY/SELL sides are allowed."""
        # This is implicitly tested by _detect_trade_pair logic
        # which always returns either BUY or SELL
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "btc": "0.01",
            "usd": "-500.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="test").collect()

        # Side must be BUY or SELL
        assert result["side"][0] in ["BUY", "SELL"]


class TestFeeEstimation:
    """Tests for fee estimation fallback when fee field is missing."""

    @pytest.fixture
    def interpreter(self):
        """Create interpreter instance."""
        return BitstampInterpreter()

    def test_fee_estimation_when_missing(self, interpreter):
        """Test that fee is estimated from total when fee field is 0."""
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "0",  # Missing fee (or 0)
            "total": "1000.0",  # Total field exists
            "btc": "0.01",
            "usd": "-500.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)

        # Fee should be estimated: 1000.0 * 0.00507038 = 5.07038
        assert len(df) == 1
        # Note: fee is stored in the df, not in normalized output (legacy stores 0)

    def test_fee_preserved_when_present(self, interpreter):
        """Test that actual fee is used when present."""
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "10.5",  # Actual fee present
            "total": "1000.0",
            "btc": "0.01",
            "usd": "-500.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)

        # Fee should be 10.5, not estimated
        assert len(df) == 1
        assert df["fee"][0] == 10.5


class TestDecimalPrecision:
    """Tests for decimal precision rounding."""

    @pytest.fixture
    def interpreter(self):
        """Create interpreter instance."""
        return BitstampInterpreter()

    def test_price_rounded_to_8_decimals(self, interpreter):
        """Test that price is rounded to max 8 decimals."""
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "btc": "0.01",
            "usd": "-500.0",
            "btc_usd": 50000.123456789,  # 9 decimals
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="test").collect()

        # Price should be rounded to 8 decimals
        assert len(result) == 1
        price = result["price"][0]
        # Check that price has at most 8 decimal places
        price_str = f"{price:.8f}"
        assert price == float(price_str)

    def test_quantity_rounded_to_8_decimals(self, interpreter):
        """Test that quantity is rounded to max 8 decimals."""
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "btc": "0.123456789",  # 9 decimals
            "usd": "-500.0",
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="test").collect()

        # Quantity should be rounded to 8 decimals
        assert len(result) == 1
        quantity = result["quantity"][0]
        # Check that quantity has at most 8 decimal places
        quantity_str = f"{quantity:.8f}"
        assert quantity == float(quantity_str)


class TestStablecoinMapping:
    """Tests for stablecoin to USD mapping."""

    @pytest.fixture
    def interpreter(self):
        """Create interpreter instance."""
        return BitstampInterpreter()

    def test_usdc_mapped_to_usd(self, interpreter):
        """Test that USDC is treated as USD in currency field."""
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "btc": "0.01",
            "usdc": "-500.0",  # USDC (stablecoin)
            "btc_usdc": 50000.0,
            "usd": 0.0,
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="test").collect()

        # Currency should be USD, not USDC
        assert len(result) == 1
        assert result["currency"][0] == "USD"

    def test_usdt_mapped_to_usd(self, interpreter):
        """Test that USDT is treated as USD in currency field."""
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "eth": "0.1",
            "usdt": "-500.0",  # USDT (stablecoin)
            "eth_usdt": 5000.0,
            "usd": 0.0,
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="test").collect()

        # Currency should be USD, not USDT
        assert len(result) == 1
        assert result["currency"][0] == "USD"

    def test_regular_usd_preserved(self, interpreter):
        """Test that regular USD currency is preserved."""
        txn = {
            "id": 111,
            "datetime": "2021-02-22 12:00:00.000000",
            "type": "2",
            "fee": "1.0",
            "btc": "0.01",
            "usd": "-500.0",  # Regular USD
            "btc_usd": 50000.0,
            "eur": 0.0,
            "order_id": 1
        }
        content = json.dumps([txn])
        df = interpreter.parse_json_content(content)
        result = interpreter.normalize(df.lazy(), user_id=1, account_id="test").collect()

        # Currency should be USD
        assert len(result) == 1
        assert result["currency"][0] == "USD"


if __name__ == "__main__":
    pytest.main([__file__, "-v"])
