# Plan: Análisis de Validaciones Críticas Faltantes - Bitstamp Normalizer

## Resumen Ejecutivo

Tras analizar exhaustivamente ambas implementaciones (legacy: 790 líneas en `bitstamp_export.py` + `brokers_bitstamp.py` vs nueva: 437 líneas en `bitstamp.py`), se identificaron **14 validaciones críticas** que faltan en la nueva implementación y podrían comprometer la integridad de datos.

## Contexto de la Migración

### Arquitectura Legacy
- Dos archivos principales: `bitstamp_export.py` (478 líneas, API sync) + `brokers_bitstamp.py` (312 líneas, CSV parsing)
- **3 formatos CSV diferentes**: getcsv(), getcsv2(), getcsv3() - Fallback en cascada
- Conexión directa a Bitstamp API con autenticación
- Validación de credenciales y detección de cuentas duplicadas
- Rate limiting con sleep(10) on "Way too much request weight used"
- Deduplicación **4-estrategias**: MD5 hash, order_id hash, composite key, webull hash
- Manejo de símbolos inválidos ("Invalid symbol.")
- Fee calculation con **factor hardcoded 0.507038** cuando falta fee field

### Arquitectura Nueva
- Archivo único: `bitstamp.py` (437 líneas, normalización)
- Enfoque en parsing de JSON (transacciones ya obtenidas de API)
- Arquitectura modular basada en intérpretes
- **Detección inteligente de pares**: Heurística de 2+ currencies con amounts no-cero
- **Quote priority system**: USD > EUR > GBP > USDT > USDC > BTC > ETH
- Deduplicación mediante file_row hash MD5 (100% compatible con legacy)
- Solo procesa type="2" (TRADES) - excluye depósitos/retiros
- Stablecoin mapping: USDC/USDT → USD

---

## Validaciones Críticas Faltantes

### CATEGORÍA 1: VALIDACIONES DE INTEGRIDAD DE DATOS [CRITICIDAD: ALTA]

#### ❌ 1. Validación de Order Status = 'FILLED'
**Ubicación Legacy:** `bitstamp_export.py:387-388`
```python
if 'status' in order and order['status'] != 'FILLED':
    continue
```

**Descripción:** El legacy solo procesa órdenes con status='FILLED'. Órdenes CANCELLED, PARTIALLY_FILLED, NEW, etc. deben ser filtradas.

**Impacto si falta:**
- Órdenes no ejecutadas en base de datos
- Métricas de volumen incorrectas
- Posiciones fantasma (órdenes pending contadas como ejecutadas)
- P&L calculations incorrectos

**Nivel de Criticidad:** 🔴 **ALTA** - Afecta directamente la integridad de las posiciones

**Ubicación en Nuevo Código:** `bitstamp.py` solo filtra type="2" (línea 211-214), pero NO filtra status='FILLED'

**Recomendación de implementación:**
```python
# En bitstamp.py:parse_json_content, después de línea 214
# Opción 1: Filtrar en el loop
if txn.get("status") != "FILLED":
    continue

# Opción 2: Añadir campo status y filtrar después en normalize()
"status": str(txn.get("status", "FILLED")),
# Luego en normalize():
.filter(pl.col("status") == "FILLED")
```

**Ubicación sugerida:** `bitstamp.py:211-214` (después del filtro type="2")

---

#### ❌ 2. Validación Explícita de Price > 0
**Ubicación Legacy:** `brokers_bitstamp.py:157, 170`
```python
# getcsv2 línea 157
if not n['price'] or not action or n['action'] not in ['BUY', 'SELL']:
    continue

# getcsv2 línea 170
price = round(float(fp),6) if vfp else 0.00
# Si price=0, no se procesaría correctamente
```

**Descripción:** Trades con precio = 0 o null son inválidos. Legacy valida explícitamente y fallback a 0.00 si no es float.

**Impacto si falta:**
- Trades con $0 afectan P&L
- Imposible calcular valor real de posición
- Reportes financieros rotos

**Nivel de Criticidad:** 🔴 **ALTA** - Afecta cálculos financieros

**Ubicación en Nuevo Código:** `bitstamp.py:171, 174-178` detecta price=0 pero **NO rechaza** la transacción
```python
# Línea 171: Detecta price pero no valida > 0
price = trade_info["price"]
# Línea 174-178: Log warning pero CONTINÚA procesando
if price == 0:
    logger.warning(f"Price is 0 for txn {txn_id}, using rate column value")
```

**Recomendación de implementación:**
```python
# Opción 1: En parse_json_content después de línea 178
if price <= 0:
    logger.warning(f"Invalid price {price} for txn {txn_id}, skipping")
    continue

# Opción 2: En normalize() añadir filtro
.filter(pl.col("price") > 0)
```

**Ubicación sugerida:** `bitstamp.py:178` (después de log warning) O en `normalize()` línea ~400

---

#### ❌ 3. Validación de Datetime No Vacío
**Ubicación Legacy:** `brokers_bitstamp.py:246-247, 256-259`
```python
# getcsv3 línea 246-247
if ('dateutc' in n and not n['dateutc']) or ('date' in n and not n['date']):
    continue

# getcsv3 línea 256-259
date_time = n['dateutc'] if 'dateutc' in n else n['date'] if 'date' in n else ''
any_error, date, time = ImportParams.get_param_datetime(date_time,b)
new_date_time = ImportParams.convert_date(date, time, date_format)
```

**Descripción:** Trades sin timestamp válido deben ser descartados. Legacy valida explícitamente que dateutc o date no sean vacíos.

**Impacto si falta:**
- Trades sin fecha en DB
- Imposible ordenar cronológicamente
- Breaks en cálculos FIFO/LIFO
- Reportes diarios incorrectos

**Nivel de Criticidad:** 🔴 **ALTA** - Esencial para ordenamiento temporal

**Ubicación en Nuevo Código:** `bitstamp.py:229` asume datetime siempre presente
```python
# Línea 229: Sin validación de null/empty
datetime_str = str(txn.get("datetime", ""))
```

**Recomendación de implementación:**
```python
# En parse_json_content después de línea 229
datetime_str = str(txn.get("datetime", ""))
if not datetime_str or datetime_str == "":
    logger.warning(f"Missing datetime for txn {txn_id}, skipping")
    continue

# O en normalize() añadir filtro
.filter(pl.col("datetime") != "")
.filter(pl.col("timestamp").is_not_null())
```

**Ubicación sugerida:** `bitstamp.py:229` (inmediatamente después de extraer datetime_str)

---

#### ❌ 4. Validación de Cantidad != 0 Después de abs()
**Ubicación Legacy:** `brokers_bitstamp.py:61-62, 153-154, 275`
```python
# getcsv línea 61-62
if (not n['value'] and not n['rate']) or n['action'] not in ['BUY', 'SELL']:
    continue

# getcsv2 línea 153-154
amount = float(n['amount'].replace(',', ''))
action = 'BUY' if amount > 0 else 'SELL' if amount < 0 else ''
# Si amount=0, action='' → rechazado después

# getcsv3 línea 275
quantity = abs(float(n['quantity'].replace('-', ''))) if 'quantity' in n else 0
# Si quantity=0, debe rechazarse
```

**Descripción:** Trades sin cantidad o con cantidad = 0 deben ser descartados.

**Impacto si falta:**
- Trades fantasma con 0 coins
- Cálculos de volumen inflados
- Errores en métricas de trading

**Nivel de Criticidad:** 🔴 **ALTA** - Datos no utilizables

**Ubicación en Nuevo Código:** `bitstamp.py:167` toma abs() pero **NO valida != 0**
```python
# Línea 167: Toma valor absoluto pero no verifica > 0
quantity = abs(base_amount)
```

**Recomendación de implementación:**
```python
# Opción 1: En _detect_trade_pair después de línea 167
quantity = abs(base_amount)
if quantity == 0:
    return None  # Reject trade with 0 quantity

# Opción 2: En normalize() añadir filtro
.filter(pl.col("quantity") > 0)
```

**Ubicación sugerida:** `bitstamp.py:167` (en _detect_trade_pair) O en `normalize()` línea ~400

---

#### ❌ 5. Validación de Action/Side en ['BUY', 'SELL']
**Ubicación Legacy:** `brokers_bitstamp.py:61, 157, 253-254`
```python
# getcsv línea 61
if ... n['action'] not in ['BUY', 'SELL']:
    continue

# getcsv2 línea 157
if ... n['action'] not in ['BUY', 'SELL']:
    continue

# getcsv3 línea 253-254
if n['action'] not in ['BUY', 'SELL'] or not n['symbol']:
    continue
```

**Descripción:** Solo se permiten acciones BUY o SELL después de normalización. Actions vacías o desconocidas deben rechazarse.

**Impacto si falta:**
- Trades con dirección desconocida
- Cálculos de posición erróneos
- Imposible determinar long/short

**Nivel de Criticidad:** 🔴 **ALTA** - Crítico para posiciones

**Ubicación en Nuevo Código:** `bitstamp.py:166` determina side pero **NO valida**
```python
# Línea 166: Calcula side pero no valida que esté en ['BUY', 'SELL']
side = "BUY" if base_amount > 0 else "SELL"
```

**Recomendación de implementación:**
```python
# En normalize() añadir filtro explícito
.filter(pl.col("side").is_in(["BUY", "SELL"]))
```

**Ubicación sugerida:** `bitstamp.py` en `normalize()` línea ~400

---

### CATEGORÍA 2: TRANSFORMACIONES DE DATOS ESPECÍFICAS [CRITICIDAD: MEDIA-ALTA]

#### ⚠️ 6. Precisión Decimal Dinámica (2-8 decimales)
**Ubicación Legacy:** `brokers_bitstamp.py:73-75, 168, 266`
```python
# getcsv línea 73-75
fp = str(price).replace(',', '').replace('$','')
decimal = fp[::-1].find('.')
decimal = decimal if decimal > 1 else 2
price = round(float(fp),6) if vfp else 0.00

# getcsv2 línea 168
decimal = decimal if decimal > 1 and decimal <= 6 else 6 if decimal > 6 else 2

# getcsv3 línea 266
decimal = decimal if decimal > 1 and decimal <= 6 else 8 if decimal > 6 else 2
```

**Descripción:** Legacy detecta decimales dinámicamente del precio original y redondea según rango (2-8 decimales según formato).

**Impacto si falta:**
- Pérdida de precisión en precios crypto (necesitan hasta 8 decimales)
- Pequeñas discrepancias acumuladas en P&L
- Inconsistencias entre formatos

**Nivel de Criticidad:** 🟡 **MEDIA** - Polars Float64 maneja precisión pero puede perder decimales históricos

**Ubicación en Nuevo Código:** `bitstamp.py` usa Float64 default, no preserva precisión original

**Recomendación de implementación:**
```python
# En normalize() añadir rounding explícito
.with_columns([
    pl.col("price").round(8).alias("price"),  # Max 8 decimals for crypto
    pl.col("quantity").round(8).alias("quantity")
])
```

**Ubicación sugerida:** `bitstamp.py` en `normalize()` después de línea 375 (timestamp conversion)

---

#### ⚠️ 7. Fee Calculation con Factor 0.507038 (SI FALTA FEE)
**Ubicación Legacy:** `brokers_bitstamp.py:252`
```python
# getcsv3 línea 252 - CRÍTICO: Fee calculation hardcoded
n['fee'] = str(float(n['total'].replace('-', '')) * 0.507038) if 'total' in n and not 'fee' in n else n['fee'] if 'fee' in n else ''
```

**Descripción:** Si el campo 'fee' no existe, legacy lo calcula como 0.507038% del total. Este factor está **hardcoded** sin documentación de origen.

**Impacto si falta:**
- Fees null para ciertos formatos de export
- Subestimación de costos de trading
- Inconsistencia con datos legacy

**Nivel de Criticidad:** 🟠 **MEDIA-ALTA** - Solo afecta exports antiguos sin fee field, pero es **bug latente**

**Ubicación en Nuevo Código:** `bitstamp.py:230` extrae fee pero **NO calcula** si falta
```python
# Línea 230: Fallback a 0 si no existe
fee = float(txn.get("fee", 0))
```

**Recomendación de implementación:**
```python
# En parse_json_content después de línea 230
fee = float(txn.get("fee", 0))
if fee == 0 and "total" in txn:
    # Fallback: estimate fee from total (legacy formula)
    try:
        total = abs(float(str(txn["total"]).replace('-', '')))
        fee = total * 0.00507038  # 0.507038% as decimal
        logger.debug(f"Estimated fee {fee} from total {total} for txn {txn_id}")
    except (ValueError, TypeError):
        fee = 0

# MEJOR: Sacar factor a constante en lugar de hardcode
FEE_ESTIMATION_FACTOR = 0.00507038  # 0.507038% - Bitstamp legacy estimation
```

**Ubicación sugerida:** `bitstamp.py:230` (después de extraer fee) + constante en línea ~70

**⚠️ ADVERTENCIA:** Factor 0.507038 no tiene documentación de origen. Puede estar desactualizado si comisiones de Bitstamp cambiaron.

---

#### ⚠️ 8. Symbol Encoding Normalization
**Ubicación Legacy:** `bitstamp_export.py:315-320`
```python
fecha = strftime(order['datetime'])
order['created_at'] = fecha
order['created_at_formated'] = order['datetime']
order['symbol'] = symbol  # Symbol normalizado
```

**Descripción:** Normaliza symbol con formato consistente. En legacy se construye desde el par del endpoint.

**Impacto si falta:**
- Symbols con encoding inconsistente
- Errores de matching entre sistemas
- Búsquedas de símbolos fallan

**Nivel de Criticidad:** 🟡 **MEDIA** - Bitstamp API es limpio pero puede tener edge cases

**Ubicación en Nuevo Código:** `bitstamp.py:224` construye symbol pero no normaliza encoding
```python
# Línea 224: Uppercase pero sin normalizacion de encoding
symbol = f"{base_currency}{quote_currency}".upper()
```

**Recomendación de implementación:**
```python
# En normalize() añadir normalización de encoding
.with_columns([
    pl.col("symbol").str.to_uppercase().alias("symbol")  # Ya está
    # Opcional: Agregar sanitization de caracteres especiales si necesario
])
```

**Ubicación sugerida:** `bitstamp.py:224` o en `normalize()` línea ~395

---

#### ⚠️ 9. Sanitización de Input CSV (Caracteres Non-ASCII)
**Ubicación Legacy:** `brokers_bitstamp.py:30-34`
```python
content = content.replace(';', '')
content_b = content.splitlines()
content = "\n".join(content_b)
content_b = ''.join([i if ord(i) < 128 else ' ' for i in content])
content_b = re.sub(r'[^\x00-\x7F]+', ' ', content_b)
```

**Descripción:** Legacy limpia CSV de caracteres no-ASCII y semicolons antes de parsear.

**Impacto si falta:**
- Parsing fallido en archivos con encoding inconsistente
- Errores por caracteres especiales

**Nivel de Criticidad:** 🟢 **BAJA** - Nuevo código solo parsea JSON (API limpia)

**Ubicación en Nuevo Código:** No aplica (JSON no CSV)

**Recomendación:** No necesario para JSON API. Solo si se añade soporte CSV en futuro.

---

### CATEGORÍA 3: DEDUPLICACIÓN [CRITICIDAD: MEDIA]

#### 💡 10. Deduplicación Multi-Estrategia (4 estrategias)
**Ubicación Legacy:** `bitstamp_export.py:369-451`
```python
# Estrategia 1: MD5 hash completo (línea 369-371)
njson = hashlib.md5(njson.encode('utf-8')).hexdigest()
if ImportParams.verify_njson(njson, ...):
    continue

# Estrategia 2: Order ID hash (línea 421-427)
if 'orderId' in original_file_row and original_file_row['orderId']:
    njson2 = hashlib.md5(njson2.encode('utf-8')).hexdigest()
    if ImportParams.verify_njson(njson2, ...):
        continue

# Estrategia 3: Composite key (línea 429-437)
if ImportParams.verify_date_file_row(..., argv2='orderId'):
    continue

# Estrategia 4: Webull hash (línea 439-451)
njson3 = {price, date, option_type, action, shares, strike, expire}
if ImportParams.verify_njson_webull(njson3, ...):
    continue
```

**Descripción:** Legacy usa 4 estrategias de deduplicación en cascada para maximizar detección.

**Impacto si falta:**
- Duplicados en caso de reimportación con datos ligeramente modificados
- Usuarios ven trades duplicados

**Nivel de Criticidad:** 🟡 **MEDIA-BAJA** - Nuevo ya tiene file_row hash (100% compatible). DB constraints deberían prevenir duplicados.

**Ubicación en Nuevo Código:** `bitstamp.py:241` solo usa MD5 de pre_hash (Estrategia 1)
```python
# Línea 241: Solo estrategia 1 (MD5 completo)
file_row_hash = hashlib.md5(json.dumps(pre_hash).encode('utf-8')).hexdigest()
```

**Recomendación:** Verificar que deduplicación se hace en capa superior (handler/orchestrator). Si no, considerar añadir order_id como dedup key alternativo.

**Ubicación sugerida:** No en normalizer - maneja en orchestration layer

---

### CATEGORÍA 4: MANEJO DE ERRORES Y EDGE CASES [CRITICIDAD: MEDIA]

#### 💡 11. Rate Limiting Retry Logic
**Ubicación Legacy:** `bitstamp_export.py:329-335`
```python
except Exception as e:
    if 'Invalid symbol.' in str(e):
        self.to_continue = True
        pass
    if "Way too much request weight used" in str(e):
        tt.sleep(10)
        continue
```

**Descripción:** Si hit rate limit, espera 10 segundos y reintenta. Si símbolo inválido, continúa sin error.

**Impacto si falta:**
- Bans de IP por exceso de requests
- Fallos de importación por rate limiting

**Nivel de Criticidad:** 🟢 **BAJA** - No aplica (retrieval separado del normalizer)

**Recomendación:** Implementar en capa de API retrieval, no en normalizer.

---

#### 💡 12. Credential Validation
**Ubicación Legacy:** `bitstamp_export.py:181-185, 189-190, 201-202`
```python
# Línea 181-182: Validar no vacío
if self.data['api_key'] =='' or self.data['secret_key'] == '':
    return "Wrong credentials"

# Línea 184-185: Check duplicados
account_check = BrokersConnections.find_by(**{
    'user_id':self.user.id,
    "api_key":self.data['api_key'].strip(),
    "secret_key":self.data['secret_key'].strip(),
    "active": True
})

# Línea 189-190: Sanitizar
self.api_key = self.data['api_key'].strip()
self.secret_key = self.data['secret_key'].strip()

# Línea 201-202: Error si ya existe
if account_check:
    return "account_is_already_connected"
```

**Descripción:** Valida credenciales, previene duplicados, sanitiza input.

**Impacto si falta:**
- Múltiples conexiones con mismo usuario
- Credenciales con espacios fallan
- Fallos por credenciales vacías

**Nivel de Criticidad:** 🟢 **BAJA** - No aplica (retrieval separado del normalizer)

**Recomendación:** Implementar en capa de API connection/authentication, no en normalizer.

---

#### 💡 13. Fallback entre Formatos CSV (getcsv → getcsv2 → getcsv3)
**Ubicación Legacy:** `brokers_bitstamp.py:116, 209`
```python
# Línea 116: Si getcsv falla, intenta getcsv2
return BrokerBitstamp.getcsv2(params,content)

# Línea 209: Si getcsv2 falla, intenta getcsv3
return BrokerBitstamp.getcsv3(params,content)
```

**Descripción:** Intenta parsear con 3 formatos CSV diferentes en cascada.

**Impacto si falta:**
- Incapacidad de parsear exports de diferentes fechas/versiones

**Nivel de Criticidad:** 🟢 **BAJA** - Nuevo solo parsea JSON API, no CSV

**Recomendación:** No necesario para JSON API. Solo si se añade soporte CSV en futuro.

---

### CATEGORÍA 5: CAMPOS Y TRANSFORMACIONES [CRITICIDAD: INFORMATIVA]

#### ℹ️ 14. Campos de Schema Faltantes
**Ubicación en Nuevo Código:** `bitstamp.py:414-435` schema final

**Campos presentes en nuevo:**
- user_id, account_id, execution_id, symbol, side, quantity, price, timestamp
- commission, fees, swap, currency, asset
- option_strike, option_expire, multiplier, pip_value
- original_file_row, file_row, row_index (20 campos)

**Campos que se agregan DESPUÉS en handler.py:1484-1493:**
- `code_currency_adapted` - Código de currency adaptado
- `code_currency_portfolio` - Código de currency del portfolio

**Impacto:** No crítico - se agregan en post-processing

**Recomendación:** Documentar que estos campos se agregan en capa superior.

---

## Resumen de Prioridades

### 🔴 CRÍTICO (Implementar INMEDIATAMENTE) - 5 validaciones
1. ❌ Order status = 'FILLED' check (o type="2" es suficiente? - verificar)
2. ❌ Price > 0 explicit validation (actualmente detecta pero no rechaza)
3. ❌ Datetime not empty validation
4. ❌ Quantity > 0 after abs() validation
5. ❌ Side in ['BUY', 'SELL'] validation

### 🟠 ALTA (Implementar en Sprint Actual) - 3 validaciones
6. ⚠️ Decimal precision dynamic rounding (2-8 decimals)
7. ⚠️ Fee calculation con factor 0.507038 SI FALTA fee field (⚠️ hardcoded)
8. ⚠️ Symbol encoding normalization

### 🟡 MEDIA-BAJA (Evaluar Necesidad) - 4 validaciones
9. 💡 Sanitización CSV non-ASCII (solo si se añade soporte CSV)
10. 💡 Deduplicación multi-estrategia (verificar capa superior)
11. 💡 Rate limiting retry (retrieval layer, no normalizer)
12. 💡 Credential validation (authentication layer, no normalizer)

### 🟢 INFORMATIVA (No Requiere Acción) - 2 items
13. ℹ️ Fallback formatos CSV (no aplica para JSON)
14. ℹ️ Campos schema faltantes (agregados en handler.py)

---

## Archivos Críticos a Modificar

### `brokers/bitstamp/bitstamp.py`
**Líneas a modificar:**

#### 1. **Añadir campo 'status' y validar** (línea ~211-214):
```python
# Después de línea 214 (type="2" filter)
# Extraer status en línea ~230
status = str(txn.get("status", "FILLED"))

# Opción A: Filtrar aquí
if status != "FILLED":
    continue

# Opción B: Añadir a records y filtrar en normalize()
"status": status,
# Luego en normalize():
.filter(pl.col("status") == "FILLED")
```

#### 2. **Validar price > 0** (línea ~178):
```python
# Después de línea 178 (log warning)
if price <= 0:
    logger.warning(f"Invalid price {price} for txn {txn_id}, skipping")
    continue

# O en normalize() añadir:
.filter(pl.col("price") > 0)
```

#### 3. **Validar datetime not empty** (línea ~229):
```python
# Después de línea 229
datetime_str = str(txn.get("datetime", ""))
if not datetime_str or datetime_str == "":
    logger.warning(f"Missing datetime for txn {txn_id}, skipping")
    continue
```

#### 4. **Validar quantity > 0** (línea ~167 o en normalize()):
```python
# En _detect_trade_pair después de línea 167
quantity = abs(base_amount)
if quantity == 0:
    return None

# O en normalize():
.filter(pl.col("quantity") > 0)
```

#### 5. **Validar side in ['BUY', 'SELL']** (en normalize() ~línea 400):
```python
.filter(pl.col("side").is_in(["BUY", "SELL"]))
```

#### 6. **Fee calculation fallback** (línea ~230):
```python
# Añadir constante en línea ~70
FEE_ESTIMATION_FACTOR = 0.00507038  # 0.507038% - Legacy Bitstamp estimation

# En parse_json_content línea ~230
fee = float(txn.get("fee", 0))
if fee == 0 and "total" in txn:
    try:
        total = abs(float(str(txn["total"]).replace('-', '')))
        fee = total * FEE_ESTIMATION_FACTOR
        logger.debug(f"Estimated fee {fee} from total {total} for txn {txn_id}")
    except (ValueError, TypeError):
        fee = 0
```

#### 7. **Decimal precision rounding** (en normalize() ~línea 380):
```python
# Después de timestamp conversion
.with_columns([
    pl.col("price").round(8).alias("price"),
    pl.col("quantity").round(8).alias("quantity")
])
```

### Tests a Añadir en `tests/brokers/test_bitstamp.py`
- ✅ Test de type="2" filter (ya existe)
- ❌ Test de status='FILLED' filter
- ❌ Test de price=0 rejection
- ❌ Test de price<0 rejection
- ❌ Test de datetime empty rejection
- ❌ Test de quantity=0 rejection
- ❌ Test de side validation (BUY/SELL only)
- ❌ Test de fee estimation cuando falta fee field
- ❌ Test de decimal precision rounding
- ❌ Test de stablecoin mapping (USDC/USDT → USD) - verificar si existe

---

## Estrategia de Verificación

### 1. Validación con Datos Históricos
```bash
# Comparar outputs de legacy vs nuevo con mismo input
python compare_normalizers.py --broker=bitstamp --trades=94
```

**Datasets de test:**
- Usuario 49186: 94 transacciones (ya verificado 100% match en file_row hash)
- Verificar que nuevas validaciones no rompen este match

### 2. Test de Regresión
Cargar 94 trades históricos de Bitstamp y comparar field-by-field:
- ✅ Symbol format (BTCUSD, LTCUSD, etc.)
- ✅ Side (BUY/SELL determinado por signo)
- ❌ Quantity (> 0 SIEMPRE)
- ❌ Price (> 0 SIEMPRE)
- ❌ Status (FILLED only si existe campo)
- ✅ Timezone (UTC→EST)
- ✅ Stablecoin mapping
- ❌ Fee (estimation con factor 0.507038 si falta)

### 3. Edge Cases a Probar
- Trades con status != 'FILLED' (si API los reporta)
- Trades con price=0 (detectados en warning línea 174)
- Trades con datetime vacío o null
- Trades con quantity=0 después de abs()
- Trades con side desconocido (no debería ocurrir)
- Transactions con type != "2" (depósitos/retiros)
- Pares con menos de 2 currencies no-cero
- Stablecoins (USDC, USDT → USD)
- Fee field faltante (debe estimar con factor 0.507038)
- Currencies no en CRYPTO_CURRENCIES o FIAT_CURRENCIES

---

## Implementación Incremental Sugerida

### Fase 1: Validaciones Críticas (Sprint 1)
- Status = 'FILLED' (si API lo reporta)
- Price > 0
- Datetime not empty
- Quantity > 0
- Side validation

### Fase 2: Transformaciones de Datos (Sprint 1)
- Decimal precision rounding
- Fee calculation fallback
- Symbol encoding normalization

### Fase 3: Testing y Verificación (Sprint 2)
- Tests unitarios para cada validación
- Test de regresión con 94 records
- Verificación de file_row hash 100% match
- Performance testing

---

## Riesgos Identificados

### ✋ Riesgo Alto
- **Sin validación price > 0**: Trades con $0 afectan P&L
- **Sin validación quantity > 0**: Trades fantasma con 0 coins
- **Sin validación datetime**: Imposible ordenar trades cronológicamente

### ⚠️ Riesgo Medio
- **Fee calculation factor hardcoded (0.507038)**: Sin documentación de origen, puede estar desactualizado
- **Sin status='FILLED' filter**: Órdenes parciales/canceladas pueden incluirse (si API las reporta)
- **Decimal precision loss**: Crypto necesita hasta 8 decimales

### 💡 Riesgo Bajo
- **Deduplicación single-strategy**: MD5 hash debería ser suficiente con DB constraints
- **Symbol encoding**: API de Bitstamp es generalmente limpia

---

## Diferencias Arquitecturales Importantes

### Legacy
- **Retrieval + Normalization integrados** en mismo código
- API connection, credential validation, rate limiting en `bitstamp_export.py`
- CSV parsing con **3 formatos fallback** en `brokers_bitstamp.py`
- Deduplicación **4-estrategias** (MD5, order_id, composite, webull)
- Fee calculation con factor **hardcoded 0.507038**
- Manual parsing de símbolos desde campos dinámicos

### Nueva
- **Solo Normalization** (retrieval separado)
- Enfoque en interpreter pattern
- JSON parsing únicamente (API sync)
- Single-file hash deduplication MD5 (100% compatible)
- **Detección inteligente de pares** con heurística
- Más simple, más testeable
- Sin lógica de API/networking

**Conclusión:** La nueva arquitectura es **superior** (separación de concerns, heurística inteligente) pero le faltan **validaciones explícitas** del legacy que asumían datos potencialmente "sucios".

---

## Fee Calculation Factor - Investigación Requerida

### ⚠️ ADVERTENCIA CRÍTICA: Factor 0.507038

**Ubicación Legacy:** `brokers_bitstamp.py:252`
```python
n['fee'] = str(float(n['total'].replace('-', '')) * 0.507038)
```

**Origen:** Desconocido. No hay documentación de por qué 0.507038.

**Hipótesis:**
1. **Tasa legacy de Bitstamp**: Podría ser tasa de comisión histórica de Bitstamp
2. **Factor de conversión**: Podría ser un rate de conversión específico
3. **Hardcoded erróneo**: Podría ser un bug nunca corregido

**Verificación necesaria:**
1. Revisar documentación histórica de comisiones de Bitstamp
2. Buscar en codebase legacy comentarios sobre este factor
3. Comparar con datos reales: fee reportado vs fee calculado
4. Si es tasa de comisión, actualizar con tasa actual de Bitstamp

**Recomendación:**
- Sacar a constante con nombre descriptivo
- Añadir comentario con origen si se encuentra
- Considerar deprecar y usar fee reportado por API

---

## Archivos de Output

Los archivos modificados se copiarán a:
```
/home/jomorale/tradersync-trade-adquisition/pipeline/p01_normalize/new_changes_bitstamp/
├── PLAN_ANALISIS_VALIDACIONES_BITSTAMP.md  # Este plan
├── README.md                               # Documentación de cambios
├── brokers/
│   └── bitstamp/
│       └── bitstamp.py                     # 7 validaciones implementadas
└── tests/
    └── brokers/
        └── test_bitstamp.py                # Tests añadidos
```

---

## Conclusiones

La nueva implementación de Bitstamp es **arquitecturalmente superior** (heurística inteligente de pares, modular, testeable) pero le faltan **14 validaciones** del legacy, de las cuales **5 son críticas**.

**Acción Recomendada:** Implementar las 5 validaciones críticas (🔴) y 3 altas (🟠) en el próximo sprint para alcanzar paridad funcional con legacy.

**Diferencias con otros brokers:**
- **Binance**: 18 validaciones (más complejo por spot/futures)
- **Interactive Brokers**: 23 validaciones (más complejo por XML + múltiples assets)
- **Bitstamp**: 14 validaciones (más simple, solo crypto spot, JSON limpio)

Bitstamp es el broker **más simple** de los tres porque:
1. Solo maneja crypto spot (no futures, no options)
2. API JSON más limpia que Binance/IB
3. Menos formatos alternativos (CSV legacy menos relevante)
4. Heurística de detección de pares muy robusta

**Punto crítico de atención:** El factor de fee 0.507038 es un **riesgo técnico** que debe investigarse antes de producción.
