# Plan: Análisis de Validaciones Críticas Faltantes - Binance Normalizer

## Resumen Ejecutivo

Tras analizar exhaustivamente ambas implementaciones (legacy: 1,177 líneas en `binance_export.py` + `brokers_binance.py` vs nueva: 309 líneas en `binance.py`), se identificaron **18 validaciones críticas** que faltan en la nueva implementación y podrían comprometer la integridad de datos.

## Contexto de la Migración

### Arquitectura Legacy
- Dos archivos principales: `binance_export.py` (API sync) + `brokers_binance.py` (CSV parsing)
- Conexión directa a Binance API con autenticación
- Validación de credenciales y permisos de API
- Rate limiting con exponential backoff y Lambda fallback
- Procesamiento paralelo (ThreadPoolExecutor para Spot, 10 workers)
- Top 800 símbolos por volumen de trading
- Deduplicación multi-estrategia (MD5, order_id, composite)
- Manejo de Spot y Futures por separado

### Arquitectura Nueva
- Archivo único: `binance.py` (normalización)
- Enfoque en parsing de JSON (órdenes ya obtenidas)
- Arquitectura modular basada en intérpretes
- Separación de responsabilidades (lógica de API movida a otro módulo)
- Deduplicación mediante file_row hash (100% compatible con legacy)
- Manejo unificado de Spot y Futures

---

## Validaciones Críticas Faltantes

### CATEGORÍA 1: VALIDACIONES DE INTEGRIDAD DE DATOS [CRITICIDAD: ALTA]

#### ❌ 1. Validación de Order Status = 'FILLED'
**Ubicación Legacy:** `binance_export.py:775`
```python
if order['status'] == 'FILLED':
    # Only process filled orders
```

**Descripción:** El legacy solo procesa órdenes con status='FILLED'. Órdenes CANCELLED, PARTIALLY_FILLED, NEW, etc. deben ser filtradas.

**Impacto si falta:**
- Órdenes no ejecutadas en base de datos
- Métricas de volumen incorrectas
- Posiciones fantasma (órdenes pending contadas como ejecutadas)
- P&L calculations incorrectos

**Nivel de Criticidad:** 🔴 **ALTA** - Afecta directamente la integridad de las posiciones

**Recomendación de implementación:**
```python
# En binance.py, después de parse_json_content, filtrar por status
.filter(pl.col("status") == "FILLED")  # Solo órdenes ejecutadas
```

**Ubicación sugerida:** `binance.py` después de parsing JSON, antes de normalization

---

#### ❌ 2. Validación de Quantity Zero/Null
**Ubicación Legacy:** `brokers_binance.py:124-127`
```python
quantity = n.get('quantity', n.get('qty', n.get('volume', n.get('amount', ''))))
if quantity == '':
    continue  # Skip records without quantity
```

**Descripción:** Trades sin cantidad o con cantidad = 0 deben ser descartados.

**Impacto si falta:**
- Trades fantasma con 0 shares/coins
- Cálculos de volumen inflados
- Errores en métricas de trading

**Nivel de Criticidad:** 🔴 **ALTA** - Datos no utilizables

**Recomendación de implementación:**
```python
# En binance.py, después de cast de quantity
.filter(pl.col("quantity") > 0)
```

**Ubicación sugerida:** `binance.py` en método `normalize()`

---

#### ❌ 3. Validación de Price Zero/Invalid
**Ubicación Legacy:** `brokers_binance.py:119-122, 169`
```python
price = n.get('price', n.get('exec price', n.get('rate', '')))
if price and ImportParams.isfloat(price):
    price = float(price)
else:
    continue  # Skip if price is invalid
```

**Descripción:** Trades con precio = 0 o no numérico son inválidos.

**Impacto si falta:**
- Trades con $0 afectan P&L
- Imposible calcular valor real de posición
- Reportes financieros rotos

**Nivel de Criticidad:** 🔴 **ALTA** - Afecta cálculos financieros

**Recomendación de implementación:**
```python
# En binance.py, después de cast de price
.filter(pl.col("price") > 0)
```

**Ubicación sugerida:** `binance.py` en método `normalize()`

---

#### ❌ 4. Validación de Symbol Vacío
**Ubicación Legacy:** `brokers_binance.py:142`
```python
if action in ['BUY', 'SELL'] and symbol:
    # Only process if symbol exists
```

**Descripción:** Trades sin símbolo son inválidos.

**Impacto si falta:**
- Trades sin identificación
- Errores en búsquedas
- Imposible hacer matching con market data

**Nivel de Criticidad:** 🔴 **ALTA** - Datos no utilizables

**Recomendación de implementación:**
```python
# En binance.py, después de normalización de symbol
.filter(pl.col("symbol") != "")
.filter(pl.col("symbol").is_not_null())
```

**Ubicación sugerida:** `binance.py` en método `normalize()`

---

#### ❌ 5. Validación de Date/Time Inválido
**Ubicación Legacy:** `brokers_binance.py:109-110`
```python
if not row.get('dateutc') or not row.get('date'):
    continue  # Skip entries without valid date
```

**Descripción:** Trades sin timestamp válido deben ser descartados.

**Impacto si falta:**
- Trades sin fecha en DB
- Imposible ordenar cronológicamente
- Breaks en cálculos FIFO/LIFO
- Reportes diarios incorrectos

**Nivel de Criticidad:** 🔴 **ALTA** - Esencial para ordenamiento temporal

**Recomendación de implementación:**
```python
# En binance.py, después de parse de timestamp
.filter(pl.col("timestamp").is_not_null())
```

**Ubicación sugerida:** `binance.py` en método `normalize()`

---

### CATEGORÍA 2: TRANSFORMACIONES DE DATOS ESPECÍFICAS [CRITICIDAD: MEDIA-ALTA]

#### ⚠️ 6. Symbol Encoding Normalization
**Ubicación Legacy:** `binance_export.py:767`
```python
order['symbol'] = self.normalize_encoding(order['symbol'])
```

**Descripción:** Normaliza encoding de símbolos para prevenir caracteres especiales.

**Impacto si falta:**
- Símbolos con encoding corrupto
- Errores de matching entre sistemas
- Búsquedas de símbolos fallan

**Nivel de Criticidad:** 🟠 **MEDIA** - Afecta búsquedas y matching

**Recomendación de implementación:**
```python
# En binance.py, añadir normalización de encoding
.with_columns([
    pl.col("symbol").str.encode("utf-8").str.decode("utf-8", strict=False)
    .alias("symbol")
])
```

**Ubicación sugerida:** `binance.py` después de uppercase de symbol

---

#### ⚠️ 7. Chinese Character Filtering (币安人生)
**Ubicación Legacy:** `binance_export.py:781`
```python
if '币安人生' in order['symbol']:
    continue  # Skip BinancePeople promotional trades
```

**Descripción:** Filtra trades promocionales de Binance que contienen caracteres chinos.

**Impacto si falta:**
- Trades promocionales en datos reales
- Métricas infladas con trades falsos
- Confusión en reportes de usuario

**Nivel de Criticidad:** 🟠 **MEDIA** - Afecta integridad de métricas

**Recomendación de implementación:**
```python
# En binance.py, filtrar símbolos con caracteres chinos
.filter(~pl.col("symbol").str.contains("币安人生"))
```

**Ubicación sugerida:** `binance.py` en método `normalize()`

---

#### ⚠️ 8. Fee Scientific Notation Handling
**Ubicación Legacy:** `brokers_binance.py:137-138`
```python
if 'e-' in fee:
    fee = "{:.8f}".format(float(fee))  # Convert scientific notation to decimal
```

**Descripción:** Binance a veces reporta fees en notación científica (e.g., 1e-8) que debe convertirse a decimal.

**Impacto si falta:**
- Fees en formato científico no procesadas correctamente
- Pequeños valores de fee truncados
- Cálculos de comisión incorrectos

**Nivel de Criticidad:** 🟠 **MEDIA** - Afecta precisión de fees

**Recomendación de implementación:**
```python
# En binance.py, añadir conversión de notación científica
.with_columns([
    pl.when(pl.col("commission").cast(pl.Utf8).str.contains("e-"))
      .then(pl.col("commission").cast(pl.Float64).round(8))
      .otherwise(pl.col("commission"))
      .alias("commission")
])
```

**Ubicación sugerida:** `binance.py` después de cast de commission

---

### CATEGORÍA 3: VALIDACIONES DE SIDE/ACTION [CRITICIDAD: MEDIA]

#### ⚠️ 9. Validación de Side/Action Inválido
**Ubicación Legacy:** `brokers_binance.py:142`
```python
if action in ['BUY', 'SELL'] and symbol:
    # Only valid if action is BUY or SELL
```

**Descripción:** Solo se permiten acciones BUY o SELL después de normalización.

**Impacto si falta:**
- Trades con dirección desconocida
- Cálculos de posición erróneos
- Imposible determinar long/short

**Nivel de Criticidad:** 🟠 **MEDIA** - Crítico para posiciones

**Recomendación de implementación:**
```python
# En binance.py, después de normalización de side
.filter(pl.col("side").is_in(["BUY", "SELL"]))
```

**Ubicación sugerida:** `binance.py` en método `normalize()`

---

#### ⚠️ 10. Fallback Action Mapping (COVER/SHORT)
**Ubicación Legacy:** `brokers_binance.py:114-117`
```python
action = action.replace('COVER', 'BUY').replace('SHORT', 'SELL')
```

**Descripción:** Normaliza acciones alternativas (COVER→BUY, SHORT→SELL).

**Impacto si falta:**
- Acciones con terminología legacy no procesadas
- Algunos trades de futures ignorados

**Nivel de Criticidad:** 🟡 **MEDIA-BAJA** - Solo afecta ciertos formatos

**Recomendación de implementación:**
```python
# En binance.py, añadir mapping de acciones alternativas
.with_columns([
    pl.col("side").str.replace("COVER", "BUY")
                  .str.replace("SHORT", "SELL")
    .alias("side")
])
```

**Ubicación sugerida:** `binance.py` antes de validación de side

---

### CATEGORÍA 4: MANEJO DE COMISIONES Y CONVERSIÓN [CRITICIDAD: MEDIA]

#### ⚠️ 11. Commission Currency Conversion a USD
**Ubicación Legacy:** `binance_export.py:851-869`
```python
if commission_asset != 'USDT' and commission_asset != 'USD':
    # Convert non-USD commission to USD using daily rate
    pip_value = get_daily_crypto_price_conversion(commission_asset, trade_date)
    commission_usd = commission * pip_value
```

**Descripción:** Comisiones en BNB, BTC u otros assets deben convertirse a USD usando rates diarios.

**Impacto si falta:**
- Comisiones en múltiples currencies no comparables
- Totales de comisión incorrectos
- Imposible calcular costos reales de trading

**Nivel de Criticidad:** 🟠 **MEDIA** - Afecta análisis financiero

**Recomendación de implementación:**
```python
# En binance.py, añadir conversión de commission a USD
# NOTA: Requiere acceso a tabla de conversión de precios diarios
.with_columns([
    pl.when(pl.col("currency") != "USD")
      .then(pl.col("commission") * pl.col("conversion_rate"))  # Needs external lookup
      .otherwise(pl.col("commission"))
      .alias("commission_usd")
])
```

**Ubicación sugerida:** `binance.py` después de normalización de commission
**Nota:** Requiere integración con servicio de conversión de precios

---

### CATEGORÍA 5: VALIDACIONES DE DUPLICADOS [CRITICIDAD: BAJA-MEDIA]

#### 💡 12. Duplicate Detection Multi-Estrategia
**Ubicación Legacy:** `brokers_binance.py:214-228`
```python
# Strategy 1: MD5 hash of full order
hash = hashlib.md5(njson.encode('utf-8')).hexdigest()

# Strategy 2: Order ID matching
if 'order #' in n and n['order #']:
    existing = check_order_id(n['order #'])

# Strategy 3: Composite validation (date+price+qty+action+symbol)
duplicate = check_composite(date, price, qty, action, symbol)
```

**Descripción:** Triple validación de duplicados (hash, order_id, composite).

**Impacto si falta:**
- Duplicados en caso de reimportación
- Usuarios ven trades duplicados

**Nivel de Criticidad:** 🟡 **MEDIA-BAJA** - Afecta UX pero DB constraints deberían prevenir

**Recomendación:** La nueva implementación ya tiene file_row hash (100% compatible). Verificar que se hace deduplicación en capa superior.

---

### CATEGORÍA 6: TRANSFORMACIONES DE PRECIO [CRITICIDAD: BAJA]

#### 💡 13. Decimal Precision Detection
**Ubicación Legacy:** `brokers_binance.py:167-168`
```python
reverse_price = str(price)[::-1]
decimal_count = reverse_price.index('.') if '.' in reverse_price else 0
```

**Descripción:** Detecta y preserva precisión decimal del precio original.

**Impacto si falta:**
- Pérdida de precisión en precios
- Pequeñas discrepancias acumuladas en P&L

**Nivel de Criticidad:** 🟢 **BAJA** - Polars maneja precisión automáticamente

**Recomendación:** No necesario si Polars Float64 preserva precisión suficiente.

---

#### 💡  14. Currency Symbol Removal ($)
**Ubicación Legacy:** `brokers_binance.py:32, 166`
```python
price = price.replace('$', '')
```

**Descripción:** Elimina símbolos de moneda antes de conversión numérica.

**Impacto si falta:**
- Errores de conversión si precio viene con '$'
- Parse failures en casos raros

**Nivel de Criticidad:** 🟢 **BAJA** - Binance API no reporta con '$'

**Recomendación:** No crítico para API JSON, solo para CSV legacy.

---

#### 💡 15. Comma/Special Character Removal
**Ubicación Legacy:** `brokers_binance.py:78`
```python
price = re.sub("[^\d\.]", "", price)  # Remove everything except digits and dot
```

**Descripción:** Elimina comas y caracteres especiales de números.

**Impacto si falta:**
- Parse failures si números vienen formateados (1,000.50)

**Nivel de Criticidad:** 🟢 **BAJA** - API JSON no usa formato con comas

**Recomendación:** No necesario para formato JSON de API.

---

### CATEGORÍA 7: VALIDACIONES DE FECHA [CRITICIDAD: BAJA]

#### 💡 16. First Day of Month Special Handling
**Ubicación Legacy:** `brokers_binance.py:68, 153-156`
```python
if day == 1:
    # Special logic for first day of month
    start_date = previous_month_last_day
```

**Descripción:** Ajusta rango de fechas cuando el primer día del mes es inicio de sync.

**Impacto si falta:**
- Potencialmente perder trades del último día del mes anterior

**Nivel de Criticidad:** 🟡 **MEDIA-BAJA** - Solo afecta edge case de sincronización

**Recomendación:** Verificar que este edge case se maneja en capa de orchestration.

---

### CATEGORÍA 8: MANEJO DE MÚLTIPLES CATEGORÍAS [CRITICIDAD: INFORMATIVA]

#### ℹ️ 17. Futures vs Spot Separation
**Ubicación Legacy:** `binance_export.py:111-112, 420-421`
```python
list_futures_account_trades = []
list_spot_account_trades = []
# Processes separately with different API endpoints
```

**Descripción:** Spot y Futures procesados en listas separadas con diferentes endpoints.

**Impacto si falta:**
- Mezcla de Spot y Futures en misma cola de procesamiento

**Nivel de Criticidad:** 🟢 **BAJA** - Probablemente manejado en orchestration layer

**Recomendación:** Verificar que orchestrator separa categorías antes de llamar normalize().

---

#### ℹ️ 18. Top 800 Symbols Limit
**Ubicación Legacy:** `binance_export.py:330`
```python
self.symbols = [symbol['symbol'] for symbol in symbols_list[:800]]
```

**Descripción:** Limita a top 800 símbolos por volumen para optimizar performance.

**Impacto si falta:**
- Procesamiento de todos los símbolos (puede ser muy lento)
- Símbolos con bajo volumen incluidos

**Nivel de Criticidad:** 🟢 **BAJA** - Optimización de performance, no validación

**Recomendación:** Implementar en capa de API retrieval, no en normalizer.

---

## Resumen de Prioridades

### 🔴 CRÍTICO (Implementar INMEDIATAMENTE) - 5 validaciones
1. ✅ Order status = 'FILLED' check
2. ✅ Quantity zero/null check
3. ✅ Price zero/invalid check
4. ✅ Symbol vacío check
5. ✅ Timestamp null check

### 🟠 ALTA (Implementar en Sprint Actual) - 6 validaciones
6. ✅ Symbol encoding normalization
7. ✅ Chinese character filtering (币安人生)
8. ✅ Fee scientific notation handling
9. ✅ Side/Action validation
10. ✅ COVER/SHORT mapping
11. ✅ Commission currency conversion (requiere servicio externo)

### 🟡 MEDIA-BAJA (Evaluar Necesidad) - 5 validaciones
12. Duplicate detection multi-estrategia (verificar capa superior)
13. Decimal precision detection (Polars lo maneja)
14. Currency symbol removal (no necesario para JSON)
15. Comma removal (no necesario para JSON)
16. First day of month handling (verificar orchestration)

### 🟢 INFORMATIVA (No Requiere Acción) - 2 items
17. Futures vs Spot separation (orchestration layer)
18. Top 800 symbols limit (API retrieval layer)

---

## Archivos Críticos a Modificar

### `brokers/binance/binance.py`
**Líneas a modificar:**

1. **Añadir campo 'status' al parsing** (~línea 80):
   - Extraer campo `status` de orders
   - Necesario para filtrar solo FILLED orders

2. **Validaciones CRÍTICAS** (después de normalize, ~línea 250):
```python
# Status filter
.filter(pl.col("status") == "FILLED")
# Data integrity
.filter(pl.col("quantity") > 0)
.filter(pl.col("price") > 0)
.filter(pl.col("symbol") != "")
.filter(pl.col("symbol").is_not_null())
.filter(pl.col("timestamp").is_not_null())
# Side validation
.filter(pl.col("side").is_in(["BUY", "SELL"]))
```

3. **Symbol transformations** (~línea 220):
```python
# Chinese character filter
.filter(~pl.col("symbol").str.contains("币安人生"))
# Encoding normalization
.with_columns([
    pl.col("symbol").str.encode("utf-8").str.decode("utf-8", strict=False)
    .alias("symbol")
])
```

4. **COVER/SHORT mapping** (antes de side validation):
```python
.with_columns([
    pl.col("side").str.replace("COVER", "BUY")
                  .str.replace("SHORT", "SELL")
    .alias("side")
])
```

5. **Fee scientific notation** (después de commission cast):
```python
.with_columns([
    pl.when(pl.col("commission").cast(pl.Utf8).str.contains("e-"))
      .then(pl.col("commission").cast(pl.Float64).round(8))
      .otherwise(pl.col("commission"))
      .alias("commission")
])
```

### Tests a Añadir en `tests/brokers/test_binance.py`
- Test de status='FILLED' filter
- Test de quantity=0 rejection
- Test de price=0 rejection
- Test de symbol="" rejection
- Test de timestamp null rejection
- Test de side validation (BUY/SELL only)
- Test de Chinese character filtering
- Test de COVER/SHORT mapping
- Test de fee scientific notation conversion
- Test de symbol encoding normalization

---

## Estrategia de Verificación

### 1. Validación con Datos Históricos
```bash
# Comparar outputs de legacy vs nuevo con mismo input
python compare_normalizers.py --broker=binance --trades=26182
```

### 2. Test de Regresión
- Cargar 26,182 trades históricos de Binance (ya verificados con file_row hash)
- Pasar por ambos normalizers
- Comparar field-by-field:
  - ✅ Symbol format
  - ✅ Side (BUY/SELL)
  - ✅ Quantity (positive)
  - ✅ Price (> 0)
  - ✅ Status (FILLED only)
  - ✅ Commission handling
  - ✅ Timestamps

### 3. Edge Cases a Probar
- Trades con status != 'FILLED' (CANCELLED, NEW, PARTIALLY_FILLED)
- Trades con quantity=0 o null
- Trades con price=0 o null
- Trades sin symbol
- Trades con timestamp inválido (0, null, futuro)
- Símbolos con caracteres chinos (币安人生)
- Fees en notación científica (1e-8, 2.5e-6)
- Actions alternativas (COVER, SHORT)
- Commission en BNB, BTC (no USDT)
- Spot vs Futures format differences
- Multiple side indicators (side vs isBuyer vs buyer)

---

## Implementación Incremental Sugerida

### Fase 1: Validaciones Críticas (Sprint 1)
- Status = 'FILLED'
- Quantity > 0
- Price > 0
- Symbol not empty
- Timestamp not null
- Side validation

### Fase 2: Transformaciones de Símbolos (Sprint 1)
- Chinese character filter
- Symbol encoding normalization
- COVER/SHORT mapping

### Fase 3: Manejo de Comisiones (Sprint 2)
- Fee scientific notation
- Commission currency conversion (requiere servicio)

### Fase 4: Fine-Tuning (Sprint 2)
- Edge cases adicionales
- Performance optimization

---

## Riesgos Identificados

### ✋ Riesgo Alto
- **Sin filtro de status='FILLED'**: Órdenes no ejecutadas en producción
- **Sin validación de quantity=0**: Trades fantasma con 0 coins
- **Price=0 no validado**: P&L calculations rotos

### ⚠️ Riesgo Medio
- **Chinese characters no filtrados**: Trades promocionales mezclados con reales
- **Fee scientific notation**: Pequeños fees truncados o mal procesados
- **Commission currency no convertida**: Imposible comparar costos entre trades

### 💡 Riesgo Bajo
- **Decimal precision**: Polars Float64 debería ser suficiente
- **Duplicate detection**: File_row hash ya compatible con legacy

---

## Diferencias Arquitecturales Importantes

### Legacy
- **Retrieval + Normalization integrados** en mismo código
- API connection, rate limiting, pagination en `binance_export.py`
- CSV parsing en `brokers_binance.py`
- ThreadPoolExecutor para paralelización
- Lambda fallback para rate limiting
- Top 800 symbols optimization
- Multi-strategy deduplication

### Nueva
- **Solo Normalization** (retrieval separado)
- Enfoque en interpreter pattern
- JSON parsing únicamente
- Single-file hash deduplication (100% compatible)
- Más simple, más testeable
- Sin lógica de API

**Conclusión:** La nueva arquitectura es superior (separación de concerns) pero le faltan validaciones críticas del legacy que asumían datos "sucios" de la API.

---

## Commission Currency Conversion - Consideraciones

### Requerimientos
La validación #11 (Commission currency conversion) requiere:
1. Tabla de precios diarios de crypto assets
2. Servicio de lookup por fecha y par (e.g., BNB/USDT en 2024-01-15)
3. Integración con pipeline de datos de precios

### Alternativas
**Opción A:** Implementar en normalizer (complejo)
- Requiere acceso a tabla de precios
- Añade dependencia externa
- Performance impactado por lookups

**Opción B:** Implementar en capa posterior (recomendado)
- Normalizer solo captura `currency` field
- Post-processing step hace conversión
- Mantiene normalizer simple y rápido

**Recomendación:** Opción B - Dejar para fase posterior, fuera del normalizer.

---

## Archivos de Output

Los archivos modificados se copiarán a:
```
/home/jomorale/tradersync-trade-adquisition/new_changes_binance/
├── PLAN_ANALISIS_VALIDACIONES_BINANCE.md  # Este plan
├── README.md                              # Documentación de cambios
├── brokers/
│   └── binance/
│       └── binance.py                     # 11 validaciones implementadas
└── tests/
    └── brokers/
        └── test_binance.py                # 40+ tests añadidos
```

---

## Conclusiones

La nueva implementación de Binance es **arquitecturalmente superior** (modular, testeable, separación de concerns) pero le faltan **18 validaciones** del legacy, de las cuales **11 son críticas o altas**.

**Acción Recomendada:** Implementar las 11 validaciones prioritarias (🔴🟠) en los próximos 2 sprints para alcanzar paridad funcional con legacy sin comprometer la nueva arquitectura.

**Nota Importante:** A diferencia de Interactive Brokers (23 validaciones), Binance tiene menos validaciones faltantes (18) porque:
1. La API de Binance es más consistente
2. Menos transformaciones de símbolo necesarias (no forex/futures prefix)
3. Formato JSON más limpio que XML
4. Menos asset types (solo crypto)
